(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Wednesday 20 September 06 15:12:54 tews@tandem.cs.ru.nl>
 *
 * adding components
 *
 * $Id: attribute_pass.ml,v 1.7 2006-09-20 13:22:32 tews Exp $
 *
 *)

open Util
open Top_variant_types
open Error
open Name_space
open Names
open Types_util
open Member_class
;;


(***********************************************************************
 ***********************************************************************
 *
 * Update Method creation
 *
 *)

exception Update_Method_Error


let update_method_error token uname = 
  begin
    error_message (remove_option token.loc)
      ("Name clash during generation of update method for attribute " 
       ^ token.token_name ^ ".\n" ^ 
       "Method " ^ uname ^ " already declared in this signature.");
    raise Update_Method_Error
  end




let create_update cl a =
  let uname = update_method_name a in
  let _ = 
    try 
      let _other = cl#find_local_member uname in
	(* abort with an error if we found some _other *)
	update_method_error a#get_token uname
    with Member_not_found -> () in
  let udom = Product( Self :: (method_arg_list a) @ [a#get_codomain]) in
  let u_token = {token_name = uname; loc = None} in
  let umethod = new ccsl_pre_member_class cl
		     u_token [] udom Self
		     a#get_visibility Update_Method
  in 
    begin 
      a#register_update_method umethod;
      cl#add_member umethod
    end


(***********************************************************************
 ***********************************************************************
 *
 * Assertion pass
 *
 *)

let method_term m =
  TypeAnnotation(BasicExpr(
		   Member(NoIface, Resolved(m))),
		 m#get_full_type)


let id_record_from_string_typ (name,typ) =
  { id_token  = 
      { token_name = name;
	loc = None
      };
    id_type = typ;
    id_parameters = [];
    id_origin = CCSL_Var;
    id_variance = Unset;
    id_sequence = -1;
    id_components = [];
  }


let do_update cl assns a =
  let varns = Name_space.create eq_ccsl_types in
  let u = a#get_update_method in
  let assname = create_id_with_preference assns 
		  (name_of_update_assertion a u)
		  "assertion" in
  let x = create_one_id varns Self in
  let xterm = (Term(x,Always,[])) in
  let indextypes = method_arg_list a in
  let ids1,ids2 = create_id_pairs varns indextypes in
  let args1 = List.map (fun (v,t) -> Term(v,Always,[])) ids1 in
  let args2 = List.map (fun (v,t) -> Term(v,Always,[])) ids2 in
  let field = create_one_id varns a#get_codomain in
  let fieldid = (field,a#get_codomain) in 
  let fieldterm = Term(field,Always,[]) in 
  let body =
    Application(method_term u,
		Tuple(xterm :: args1 @ [fieldterm]))
  in let res =
      if args1 = [] 
      then fieldterm
      else 
	If([And(List.map2 (fun a1 a2 -> Equal(a1,a2)) args1 args2),
	    fieldterm
	   ],
	   Application(method_term a,
		       Tuple(xterm :: args2)))
  in
  let form = 
    Equal(
      Application(
	method_term a,
	if args2 = [] 
	then body
	else Tuple(body :: args2)),
      res)
  in let ass =
    { free_variables = 
	List.map id_record_from_string_typ (ids1 @ ids2 @ [fieldid]);
      self_variable = Some(id_record_from_string_typ (x,Self));
      assertion_name = {token_name = assname; loc = None};
      is_generated = true;
      assertion_formula = Symbolic form
    }
  in 
    cl#add_assertion ass



let do_independent cl assns a a' =
  let varns = Name_space.create eq_ccsl_types in
  let u = a'#get_update_method in
  let assname = create_id_with_preference assns 
		  (name_of_independent_assertion a u)
		  "assertion" in
  let x = create_one_id varns Self in
  let xterm = (Term(x,Always,[])) in
  let uargtypes = method_arg_list u in
  let uids = create_ids varns uargtypes in
  let uargs = List.map (fun (v,t) -> Term(v,Always,[])) uids in
  let aargtypes = method_arg_list a in
  let aids = create_ids varns aargtypes in
  let aargs = List.map (fun (v,t) -> Term(v,Always,[])) aids in
  let body = 
    Application(method_term u, Tuple(xterm :: uargs))
  in let form = 
      Equal(
	Application(
	  method_term a,
	  if aargs = [] 
	  then body
	  else Tuple(body :: aargs)),
	Application(
	  method_term a,
	  if aargs = [] 
	  then xterm
	  else Tuple(xterm :: aargs))
      )
  in let ass =
    { free_variables = 
	List.map id_record_from_string_typ (uids @ aids);
      self_variable = Some(id_record_from_string_typ (x,Self));
      assertion_name = {token_name = assname; loc = None};
      is_generated = true;
      assertion_formula = Symbolic form
    }
  in 
    cl#add_assertion ass



let do_class_assertions cl =
  let assns = Name_space.create eq_ccsl_types in
  let _ = reserve assns (List.map (fun ass -> ass.assertion_name.token_name) 
			   cl#get_assertions) in
  let _ = reserve assns (List.map (fun ass -> ass.assertion_name.token_name) 
			   cl#get_creations) in
  let _ = reserve assns (List.map (fun ass -> ass.assertion_name.token_name) 
			   cl#get_theorems) in
  let inherited_attribute = 
    List.flatten
      (List.map (fun acl -> acl#get_all_attributes) 
	 cl#get_resolved_ancestors)
  in let new_attributes = cl#get_attributes 
  in
    begin
      List.iter (fun a -> List.iter 
		     (fun u -> 
			do_independent cl assns a u;
			do_independent cl assns u a) 
		     inherited_attribute)
	new_attributes;
      List.iter (fun a -> List.iter 
		     (fun u -> (if a == u 
				then do_update cl assns a
				else do_independent cl assns a u))
		     new_attributes)
	new_attributes
    end

(***********************************************************************
 ***********************************************************************
 *
 * top level units
 *
 *)


let update_ast = function
  | CCSL_class_dec cl -> 
      List.iter (create_update cl) cl#get_attributes
  | CCSL_adt_dec adt -> ()
  | CCSL_sig_dec si -> ()


let ccsl_update_method_pass ast =
  List.iter update_ast ast


let assertion_ast = function
  | CCSL_class_dec cl -> do_class_assertions cl
  | CCSL_adt_dec adt -> ()
  | CCSL_sig_dec si -> ()


let ccsl_attribute_assertion_pass ast = 
  List.iter assertion_ast ast



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

