(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 17.6.99 by Hendrik
 *
 * Time-stamp: <Monday 4 August 03 23:13:41 tews@debian>
 *
 * Lifttings and invariants
 *
 * $Id: invariance_theory.ml,v 1.9 2003-08-21 15:15:03 tews Exp $
 *
 *)

open Util
open Global
open Top_variant_types
open Top_classtypes
open Name_space
open Names
open Top_names
open Classtypes
open Types_util
open Pvs_proof_util
open Lifting
open Theory_class;;

  
(***********************************************************************
 ***********************************************************************
 *
 * Liftings
 *
 *)

class ['class_type, 'member_type] ccsl_pre_lift_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      ((if cl#has_constructors then
	  [name_of_algebra]
       	else [])
       @
       [ name_of_coalgebra;
	 name_of_private_class_pred cl;
	 name_of_public_class_pred cl;
	 name_of_initially cl;
	 name_of_private_invariance cl;
	 name_of_public_invariance cl
       ]
       @
       (List.map fst cl#get_lifting_requests)
      )

    method get_name = ccsl_lift_theory_name cl

    method get_parameters = self#simple_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

	(* the workhorse for predicate lifting
	 * takes a method list and produces the predicate lifting wrt all 
	 * ancestors and the methods in the list.
	 *)
    method private do_class_lifting class_lifting method_list =
      let sub_ns = sub_space ns in
      let p = create_one_id sub_ns (Predtype True) in
      let self_var = create_one_id sub_ns Self in
      let self_lifting = 
	(None, Some (Expression(ConstantPredicate(p))))
      in
      let param_lifting = [] in
      let ancestor_list = 
	List.map (function
		    | Resolved_renaming(_,args,_,ianc) -> ianc,args
					(* no other stuff here *)
		    | Unresolved_renaming _
		    | _ -> assert(false)
		 ) cl#get_ancestors
      in
      	Defn(class_lifting cl,
	     [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	     (Function(Function(Self,Bool),
		       Function(Self,Bool))),
	     Abstraction
	       ([p,Function(Self,Bool)],
		Abstraction
		  ([self_var,Self],
		   Expression
		     (class_method_ancestor_pred_lifting 
			class_lifting                 (* anc_lifting *)
			param_lifting
			self_lifting
			sub_ns
			coalgebra_term
			ancestor_list
			method_list
			self_var
			self_argument
		     ))))

    method private private_class_lifting = 
      self#do_class_lifting name_of_private_class_pred cl#get_methods
	
    method private public_class_lifting =
      self#do_class_lifting name_of_public_class_pred 
	(List.filter
	   (fun m -> m#get_visibility = Public)
	   cl#get_methods)


    method private initially =
      let sub_ns = sub_space ns in
      let p = create_one_id sub_ns (Function(Self,Bool)) in
      let self_lifting = 
	(None,
	 Some(Expression(ConstantPredicate(p))))
      in
      	if cl#has_constructors then
      	  [Defn(name_of_initially cl,
	      	[[Declared(name_of_algebra, self#algebra_type)]],
	      	Function(Function(Self,Bool), 
			 Bool),
	      	Abstraction(
		  [p, Function(Self, Bool)],
		  Expression(
		    predlift self_lifting sub_ns 
			       self#get_constructor_functor_type
			       (Term(name_of_algebra,Always,[])))))
	  ]
	else
			(* do nothing if user didn't supply constructors *)
	  []
		

    method private do_lifting_request (name,typ) = 
      let sub_ns = sub_space ns in
      let p = create_one_id sub_ns (Function(Self,Bool)) in
      let self_lifting = 
	(None,
	 Some(Expression(ConstantPredicate(p))))
      in
      let t_var = create_one_id sub_ns typ in
      	Defn(name,
	     [],
	     Function(
	       Function(Self,Bool),
	       Function(typ,Bool)),
	     Abstraction(
	       [p, Function(Self, Bool)],
	       Abstraction(
		 [t_var, typ],
		 Expression(
		   predlift self_lifting sub_ns 
			      typ (Term(t_var,Always,[]))))))
      

    method private invariance name lifting =
      let sub_ns = sub_space ns in
      let p = create_one_id sub_ns (Predtype True) in
      let self_var = create_one_id sub_ns Self in
      let self_ex = Term(self_var,Always,[]) in
      let pred_ex = Term(p,Always,[]) in
      Defn(name,
	   [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	   Function(Function(Self,Bool), Bool),
	   Abstraction(
	     [p,Function(Self,Bool)],
	     Expression(
	       Forall(
		 [self_var,Self],
		 Implies(Formula(Application(pred_ex, self_ex)),
			 Formula(
			   Application(
			     Application(
			       Application(
				 Term(lifting,Always,[]),
				 coalgebra_term),
			       pred_ex),
			     self_ex)))))))
					      
    method private private_invariance =
      self#invariance (name_of_private_invariance cl) 
	(name_of_private_class_pred cl)

    method private public_invariance =
      self#invariance (name_of_public_invariance cl) 
	(name_of_public_class_pred cl)

    method make_body = 
      [	Import([ccsl_interface_theory_name cl, self#simple_arguments]);
	(self#coalgebra_decl)
      ]
      @
      (if cl#has_constructors then
	 [(self#algebra_decl)]
       else [])
      @
      [(self#private_class_lifting);
       (self#public_class_lifting)
      ]
      @
      self#initially
      @
      (List.map self#do_lifting_request cl#get_lifting_requests)
      @
      [
       	self#private_invariance;
       	self#public_invariance
      ]

end (* ccsl_pre_lift_theory *)

class ccsl_lift_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_lift_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Rewrite lemmas for invariance
 * make first a generic class, with a virtual method for the invariant 
 * predicate.
 *
 * Define later the two rewrite theories for full and public invariance
 * via inheritance and overriding
 *
 *)

class virtual ['class_type, 'member_type] ccsl_pre_invariance_rewrite_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  (method_filter : ('class_type, 'member_type) ccsl_pre_member_type -> bool) 
  = 
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      ((if cl#has_constructors then
	  [name_of_algebra]
       	else [])
       @
       [
	 name_of_coalgebra;
	 "P"; "Q";
      	 name_of_inv_true_lemma;
	 name_of_inv_and_lemma;
	 name_of_inv_list_lemma
       ])

    method get_parameters = self#simple_parameters

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

	(* this gets overriden in derived classes to specialize
	 * this class for full or public invariance 
	 *)
    method private virtual invariant_pred_string : string

    method private invariant_pred =
      Expression(ConstantPredicate(self#invariant_pred_string))

      (* 
       *   inv_add : LEMMA
       *     (FORALL ( x: Self , p1: Name , p2: Data) :
       *     (invariant?(c)(P))
       *     IMPLIES
       * 	  ((P(x))) IMPLIES (every(P)(add(c)(x , p1 , p2))))
       *)

      (* invariant lemmas for methods and constructors differ only in
       * one premise, the innermost expression and the proof.
       * For the methods we have invariant? and an application of 
       * the method to the coalgebra, 
       * for constructors we have initially? and a recordfield selection 
       * for the constructor.
       * We first write the generic pattern, 
       * which takes this preise, the inner expression and the proof
       * as arguments.
       *)
    method private do_member m prem inner_expr proof =
(* 					   (* facilitate debugging *)
 * 	 let d_cl_name = cl#get_name in
 * 	 let d_m_name = m#get_name in
 * 	 let d_m_dom = m#get_domain in
 * 	 let d_m_codom = m#get_codomain in
 * 
 *)
      let sub_ns = sub_space ns in
      let _ = make_flat sub_ns in
      let self_lifting = 
	(None,
	 Some(Expression(ConstantPredicate("P"))))
      in
      let form = predlift self_lifting
		   sub_ns 
		   (Function(m#get_domain,m#get_codomain))
		   inner_expr in
      let form2 = match !output_mode with
	| Pvs_mode -> make_pvs_rewrite_form (Implies(prem, form))
	| Isa_mode -> make_isa_rewrite_form (Implies(prem, form))
      in
	Proved(
	  Lemma(
	    name_of_inv_lemma m,
	    form2),
	  proof)

	  (* specialize for methods *)
    method private do_method m = 
      self#do_member m (Formula(
			  Application(
			    Application(self#invariant_pred,
				       	coalgebra_term),
			    Term("P",Always,[]))))
	(Application(Term(m#get_name,Always,[]), coalgebra_term))
	(self#inv_method_proof m)

	(* specialize for constructors *)
    method private do_constructor m = 
      self#do_member m (Formula(
			  Application(
			    Application(Term(name_of_initially cl,Always,[]),
				       	algebra_term),
			    Term("P",Always,[]))))
	(RecordSelection( m#get_name, algebra_term))
	self#inv_constructor_proof

    (***************************************************
     *
     * invariant lemma proofs
     *)

    (* invariant lemma proof
     * 
     * (""
     * 	(SKOSIMP* )
     * 	(EXPAND "invariant?")
     * 	(INST?)
     * 	(ASSERT)
     * 	(Auto-rewrite-theories ...)
     * 	(FLATTEN)
     * 	(APPLY (REPEAT
     * 		(TRY (BRANCH (INST? -1) ((SKIP) (FAIL))) (HIDE -1)
     * 		 (HIDE -1)))))
     *)
 
    method private inv_method_proof m =
      let rec collect_anc_names cl =
	(List.flatten(
	   List.map (function
		       | Resolved_renaming(anc,_,_,_) -> 
			   collect_anc_names anc
					(* no other stuff here *)
		       | Unresolved_renaming _
		       | _ -> assert(false)
		    ) cl#get_ancestors))
	 @
	 (List.map (function
		      | Resolved_renaming(anc,_,_,_) -> 
			  ccsl_lift_theory_name anc
					(* no other stuff here *)
		      | Unresolved_renaming _
		      | _ -> assert(false)
		   ) cl#get_ancestors) in	

      let anc_inv_th_list = 
	(ccsl_lift_theory_name cl) :: (collect_anc_names cl)
      in
	PvsProof(
      	  Anon_proof( make_simple_proof(
		      [ skosimp_star;
		      	expand self#invariant_pred_string;
		      	inst_num (-1);
		      	pvs_assert;
		      	auto_rewrite_theories anc_inv_th_list;
		      	pvs_assert;
		      	flatten;
		      	repeat (rewrite_left m#get_name);
		      	super_inst])))

    method private inv_constructor_proof =
      let inv_th = (ccsl_lift_theory_name cl) in
	PvsProof(
      	  Anon_proof( make_simple_proof(
		      [ skosimp_star;
			expand (name_of_initially cl);
		      	inst_num (-1);
			pvs_assert;
		      	auto_rewrite_theories [inv_th];
		      	pvs_assert;
		      	flatten;
		      	super_inst])))

    (***************************************************
     *
     * standard lemmas for truth and finite conjunction
     *)

    val p_pred = Expression(ConstantPredicate("P"))
    val q_pred = Expression(ConstantPredicate("Q"))

    method private pq_declaration =
      [Vardecl("P", Function(Self,Bool));
       Vardecl("Q", Function(Self,Bool))
      ]

      (*
       *   inv_true : LEMMA invariant?(c)((LAMBDA(x: Self) : TRUE))
       *)
    method private inv_true = 
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns Self in
	Lemma( 
	  name_of_inv_true_lemma,
	  Formula(
	    Application(
	      Application(
	      	self#invariant_pred,
	      	coalgebra_term),
	      Abstraction([x,Self],
			  Expression(True)))))
	  
      (*   
       *   inv_and : LEMMA
       *     ((invariant?(c)(P)) AND (invariant?(c)(Q)))
       *     IMPLIES (invariant?(c)((LAMBDA(x: Self) : (P(x)) AND (Q(x)))))
       *)
  
    method private inv_and = 
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns Self in
	Lemma(
	  name_of_inv_and_lemma,
	  Implies(
	    And(
	      [Formula(
		 Application(
		   Application(self#invariant_pred,coalgebra_term),
		   p_pred));
	       Formula(
		 Application(
		   Application(self#invariant_pred,coalgebra_term),
		   q_pred))]),
	    Formula(
	      Application(
	      	Application(self#invariant_pred,coalgebra_term),
	      	Abstraction(
		  [x,Self],
		  Expression(
		    And(
		      [Formula(Application(p_pred,Term(x,Always,[])));
		       Formula(Application(q_pred,Term(x,Always,[])))
		      ])))))))
      (* 
       *   inv_list : LEMMA
       *     FORALL ( z: list[[Self -> bool]]) :
       * 	 (every(invariant?(c))(z))
       * 	 IMPLIES
       * 	 (invariant?(c)
       * 	    (reduce
       * 	       ((LAMBDA(x: Self) : TRUE) ,
       * 		(LAMBDA(P: [Self -> bool] , Q: [Self -> bool]) :
       * 		  (LAMBDA(x: Self) : (P(x)) AND (Q(x)))))
       * 	       (z)))
       *)

    val list_adt = 
		(match Symbol.find "list" with 
		   | AdtSymb( adt ) -> adt
		   | _ -> assert(false)
		)
		       
    method private inv_list =
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns Self in
      let list_type = Adt(list_adt, Always,
			  [TypeArgument(Function(Self,Bool))]) in
      let z = create_one_id sub_ns list_type in
	Proved(
	  Lemma(
	    name_of_inv_list_lemma,
	    Forall(
	      [z, list_type],
	      Implies(
	      	Formula(
		  Application(
		    Application(
	      	      Term(name_of_adt_every list_adt,Always,[]),
		      Application(self#invariant_pred,coalgebra_term)),
		    Term(z,Always,[]))),
	      	Formula(
		  Application(
		    Application(self#invariant_pred,coalgebra_term),
		    Application(
		      Application(
		      	Term(name_of_adt_reduce list_adt, Always,[]),
		      	Tuple(
		      	  [Abstraction([x,Self], Expression(True));
			   Abstraction(
			     ["P",Function(Self,Bool);
			      "Q",Function(Self,Bool)],
			     Abstraction(
			       [x,Self],
			       Expression(
				 And(
				   [Formula(Application(p_pred,
						      	Term(x,Always,[])));
				    Formula(Application(q_pred,
						      	Term(x,Always,[])))
				   ]))))
		      	  ])),
		      Term(z,Always,[]))))))),
	  (self#inv_list_proof z))
 

    (*
     * (""
     * 	(SKOLEM 1 ("c" "_"))
     * 	(INDUCT "z")
     * 	(("1" (PROP) (EXPAND "reduce") (REWRITE "inv_true"))
     * 	 ("2"
     * 	  (SKOSIMP* )
     * 	  (EXPAND "reduce" 1)
     * 	  (EXPAND "every" -2)
     * 	  (FLATTEN)
     * 	  (ASSERT)
     * 	  (REWRITE "inv_and")
     * 	  (HIDE 2)
     * 	  (CASE "reduce((LAMBDA (x: Self): TRUE),
     * 				   (LAMBDA (P: [Self -> bool],
     * 					    Q: [Self -> bool]):
     * 				      (LAMBDA (x: Self):
     * 					 (P(x))
     * 					     AND
     * 					   (Q(x)))))(cons2_var!1)
     * = LAMBDA (x: Self):
     * 			      (reduce((LAMBDA (x: Self): TRUE),
     * 				      (LAMBDA (P: [Self -> bool],
     * 					       Q: [Self -> bool]):
     * 					 (LAMBDA (x: Self):
     * 					    (P(x))
     * 					       AND (Q(x)))))(cons2_var!1)(x))")
     * 	  (("1" (ASSERT)) ("2" (APPLY-EXTENSIONALITY :HIDE? T))))))
     *                                                             (* "\"" *)
     *) 
    method private inv_list_proof list_name =
      Anon_proof( 
	PTree(
	  [ induct list_name ],
	  [ make_simple_proof [ skosimp_star; 
				expand "reduce";
				rewrite "inv_true"];
	    PTree(
	      [ skosimp_star;
		expand_num "reduce" 1;
		expand_num "every" (-2);
		flatten;
		pvs_assert;
		rewrite "inv_and";
		(case "reduce((LAMBDA (x: Self): TRUE),
      		   (LAMBDA (P: [Self -> bool],
      			    Q: [Self -> bool]):
      		    (LAMBDA (x: Self):
      		     (P(x))
      		       AND
      		       (Q(x)))))(cons2_var!1)
	      	= LAMBDA (x: Self):
      		  (reduce((LAMBDA (x: Self): TRUE),
      			  (LAMBDA (P: [Self -> bool],
      				   Q: [Self -> bool]):
      			   (LAMBDA (x: Self):
      			    (P(x))
      			      AND (Q(x)))))(cons2_var!1)(x))")],
		    [make_simple_proof [ inst_question; pvs_assert ];
		     make_simple_proof [ extensionality ]])]))


    method make_body =
      [
	Import([ccsl_lift_theory_name cl, self#simple_arguments]);
   	self#coalgebra_decl;
      ]
      @
      (if cl#has_constructors then
	 [(self#algebra_decl)]
       else [])
      @
      self#pq_declaration
      @ 
      [Comment("General results about invariants");
       self#inv_true;
       self#inv_and;
       self#inv_list]
      @
      Comment("Rewrite lemmas for methods")::
      (List.map self#do_method
	 (List.filter method_filter cl#get_all_members))
      @
      if cl#has_constructors then
      	Comment("Rewrite lemmas for constructors")::
	List.map self#do_constructor cl#get_constructors
      else [
]


end

(***********************************************************************
 ***********************************************************************
 *
 * Rewrite lemmas for private invariance
 *
 *)

class ['class_type, 'member_type] ccsl_pre_private_invariance_rewrite_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_invariance_rewrite_theory 
      cl eq_types
      (fun m -> (m#get_sort = Unary_Method) || (m#get_sort = Nary_Method)) 


    method get_name = ccsl_private_invariance_rewrite_name cl

    method private invariant_pred_string = 
      name_of_private_invariance cl

end


class ccsl_private_invariance_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_private_invariance_rewrite_theory cl eq_ccsl_types

(***********************************************************************
 ***********************************************************************
 *
 * Rewrite lemmas for public invariance
 *
 *)

class ['class_type, 'member_type] ccsl_pre_public_invariance_rewrite_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_invariance_rewrite_theory 
      cl eq_types
      (fun m -> 
	 ((m#get_sort = Unary_Method) || (m#get_sort = Nary_Method))
	 && (m#get_visibility = Public))

    method get_name = ccsl_public_invariance_rewrite_name cl

    method private invariant_pred_string = 
      name_of_public_invariance cl

end


class ccsl_public_invariance_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type]
    ccsl_pre_public_invariance_rewrite_theory cl eq_ccsl_types

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

