(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Adopted 29.1.99 by Hendrik to Loop Version 1
 *
 * Time-stamp: <Wednesday 29 September 10 14:48:42 tews@blau.inf.tu-dresden.de>
 *
 * CCSL Lexic 
 *
 * $Id: lexer.mll,v 1.10 2010-10-19 12:51:07 tews Exp $
 *
 *)

{
       (* to get token_type *)
  open Global
  open Error
  open Parser_util
  open Top_variant_types			(* for token type *)
  open Grammar
  open Lexing;;

  let d s =
    if debug_level _DEBUG_LEXER
    then begin
      prerr_string ("SHIFT " ^ s ^ "\n");
      flush stderr
    end

  let get_loc lexbuf = 
    let loc = { file_name = !current_file;
		start_line = !line;
		start_char = (lexeme_start lexbuf) - !line_start;
		end_line = !line;
		end_char = (lexeme_end lexbuf) - !line_start
	      }
    in
      Parser.last_loc := loc;
      loc

  let updloc lexbuf =
    ignore(get_loc lexbuf)

  let get_token lexbuf = 
    {token_name = (lexeme lexbuf);
     loc = Some( get_loc lexbuf )
    }

;;

}

let symbolchar = 
  ['!' '$' '&' '*' '+' '-' '.' '/' '\\' ':' 
   '<' '=' '>' '?' '@' '^' '|' '~' '#']

rule token =
   parse 
     | [' ' '\t']           	{ token lexbuf }	(* skip blanks *)
     | '\n' 			{ d ("Eol " ^ (string_of_int !line)); 
				  newline(lexeme_end lexbuf); 
				  token lexbuf }
     | "#include" [' ' '\t'] '"'
				{ let f = string lexbuf
				  in 
				    d( "Include: \"" ^f^ "\"");
				    INCLUDE( f, get_loc lexbuf )
				}
     | '='			{ d "Equal"; 
				  updloc lexbuf; EQUAL }
     | '~'                      { d "Obseq"; 
				  updloc lexbuf; OBSEQ }
     | "::"			{ d "DoubleColon"; 
				  updloc lexbuf; DOUBLECOLON }
     | ":="			{ d "Assign"; 
				  updloc lexbuf; ASSIGN }
     | ':'			{ d "Colon"; 
				  updloc lexbuf; COLON }
     | ';'			{ d "Semicolon"; 
				  updloc lexbuf; SEMICOLON }
     | ','			{ d "Comma"; 
				  updloc lexbuf; COMMA }
     | '['			{ d "Obracket"; 
				  OBRACKET (get_loc lexbuf)}
     | ']'			{ d "CBracket"; 
				  CBRACKET (get_loc lexbuf)}
     | '('			{ d "OParen"; 
				  OPAREN (get_loc lexbuf)}
     | ')'			{ d "CParen"; 
				  CPAREN (get_loc lexbuf)}
     | "->"			{ d "Arrow"; 
				  updloc lexbuf; ARROW }
     | '{'			{ d "OBrace"; 
				  OBRACE (get_loc lexbuf)}
     | '}'			{ d "CBrace"; 
				  CBRACE (get_loc lexbuf)}
     | '.' 			{ d "Dot"; 
				  updloc lexbuf; DOT}
     | '?' 			{ d "Questionmark"; 
				  updloc lexbuf; QUESTIONMARK }
     | '%' [^'\n'] *		{ d "Pvs Comment"; 
				  token lexbuf }
     | "(*"			{ d "Comment"; 
				  comment lexbuf; token lexbuf }
					(* help fontification "*)" *)

     | "**" symbolchar *	{ d "Infix **";
				  INFIX_EXP( get_token lexbuf )
				}
     | ['*' '/' '\\' ] symbolchar *	
				{ d "Infix */";
				  INFIX_MUL( get_token lexbuf )
				}
     | ['+' '-'] symbolchar *	{ d "Infix +-";
				  INFIX_ADD( get_token lexbuf )
				}
     | ['@' '^' '#' ] symbolchar *
				{ d "Infix @";
				  INFIX_SHARP( get_token lexbuf )
				}
     | ['=' '~' '<' '>' '|' '&' '$'] symbolchar *	
				{ d "Infix =";
				  INFIX_REL( get_token lexbuf )
				}

     | ['P''p']['V''v']['S''s'] [ ' ''\t''\n']
	 			{ d "PVS"; 
				  if lexeme_char lexbuf 3 = '\n' then
				    begin
				      d ("Eol " ^ (string_of_int !line)); 
				      newline(lexeme_end lexbuf);
				    end;				  
				  let f = pvs lexbuf
				  in 
				    d( "Formula: \"" ^ f ^ "\"");
				    updloc lexbuf;
				    PVS_FORMULA( Pvs_String f )
                                }

     | '"' 			{ let s = string lexbuf
				  in 
				    d( "String: \"" ^s^ "\"");
				    updloc lexbuf;
				    STRING( s )
				}

     | ['P''p']['R''r']['O''o']['J''j']'_'['0'-'9']+ 
      				{ let mylexstr = (lexeme lexbuf) in
				  let _ = 
				    if mylexstr.[5] = '0' 
				    then 
				      (error_message (get_loc lexbuf)
					 ("Projection index must not start " ^
					  "with 0.");
				       raise Parsing.Parse_error
				      )
				  in				       
				  let n_str = String.sub mylexstr 5 
						((String.length mylexstr) - 5) 
				  in 
				    d ( "Proj_" ^ n_str);
				    PROJ_N(int_of_string n_str, 
					   get_loc lexbuf)
				}
     | ['A'-'Z' 'a'-'z']['A'-'Z' 'a'-'z' '_' '0'-'9' '?']*
				{ d( "Id " ^ lexeme lexbuf); 
				  ID( get_token lexbuf )
				}
     | ['0'-'9']+		{ d "Number"; 
				  VALUE(lexeme lexbuf, get_loc lexbuf)}
     | _			{ d("Illegal " ^ (lexeme lexbuf));
				  error_message (get_loc lexbuf)
				    "Illegal Character";
				  raise Parsing.Parse_error
       				}

     | eof			{ d "Eof"; EOF }
   
and pvs =
   parse
       ['E''e']['N''n']['D''d']['P''p']['V''v']['S''s']
				{ d "ENDPVS"; "" }
     | '\n'			{ d ("Eol " ^ (string_of_int !line)); 
				  newline(lexeme_end lexbuf); 
				  "\n" ^ (pvs lexbuf) }
     | 'E'			{ d "E"; "E" ^ (pvs lexbuf) }
     | 'e'			{ d "e"; "e" ^ (pvs lexbuf) }
     | [^ '\n' 'E' 'e' ] +	{ d "Text"; 
				  let t = (lexeme lexbuf) 
				  in t ^ (pvs lexbuf) 
				}
     | eof			{ error_message (get_loc lexbuf) 
				    ( "EOF encountered before PVS \
						   Formula was complete" );
				  raise Parsing.Parse_error}		   

and string = 
   parse
       '"'			{ d "EOS"; "" }
     | '\\' '"'			{ d "\\\""; "\"" ^ (string lexbuf) }
     | "\\\\"			{ d "\\\\"; "\\" ^ (string lexbuf) }
     | '\\' _			{ let c = String.sub (lexeme lexbuf) 1 1
				  in begin
				       d c;
				       c ^ (string lexbuf)
				     end
                                }
     | [^ '\\' '"' ]+		{ d "Text";
				  let t = lexeme lexbuf
				  in t ^ (string lexbuf)
       				}
     | eof			{error_message (get_loc lexbuf) 
				    ( "EOF encountered in the middle \
				               of a string" );
				  raise Parsing.Parse_error}

and comment =
   parse
       '\n'			{ d ("Eol " ^ (string_of_int !line));
				  newline(lexeme_end lexbuf); 
				  comment lexbuf }
     | "*)"			{ () }
     | '*'			{ comment lexbuf }
     | "(*"			{ comment lexbuf; comment lexbuf }
     | '(' 			{ comment lexbuf }
     | [^ '\n' '*' '('] +	{ comment lexbuf }
     | eof			{ error_message (get_loc lexbuf) 
				    ( "EOF encountered before \
				       Comment was complete" );
				  raise Parsing.Parse_error }


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
