(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Wednesday 30 June 10 11:31:51 tews@blau.inf.tu-dresden.de>
 *
 * Name resolution & Typechecking with unification
 *
 * $Id: newtypecheck.ml,v 1.26 2010-06-30 09:38:01 tews Exp $
 *
 *
 * the algorithm is homegrown and works in three passes
 * 1. resolution of all identifiers except for MethodSelection's
 * 2. bottem up type derivation and resolution of MethodSelection's
 * 2. top down type checking
 * All passes are performed in sequence for every 
 * assertion/definition/theorem. They 
 * share a (per assertion) persistent store of bindings for free 
 * type variables.
 * 
 * Resolution is straightforward. It is interwind with typechecking 
 * to simplify the treatment of the symboltable.
 * 
 * Type derivation derives types for the expressions and introduces 
 * free type variables when neccessary. Certain nodes (such as application) 
 * perform type unification. If the unification fails an error is reported. 
 * Otherwise the unification may bind free type variables to types. 
 * At places where type information is lost in the derivation (for 
 * instance Application, Equal) some types are stored in the ast 
 * for the following type check.
 * 
 * Type checking works top down and checks if all free type variables 
 * are bound to a ground type. For Members it checks if a complete 
 * instanciation for the class/adt is available. If so the instanciation 
 * is stored in the ast.
 * For projections their type is checked, for Obseq an appropiate 
 * lifting is requested via add_rel_lifting.
 * 
 * Type unification is straightforward. It keeps a store of bindings for 
 * free type variables. For the occur check to work there is an 
 * important side condition (see below in the type variable 
 * management section)
 *)

(* 
 * open Util
 * 
 * 
 * 
 * 
 * open Classtypes
 * 
 *)
open Error
open Global
open Top_variant_types
open Ccsl_pretty
open Types_util
open Resolution
open Derive
;;


(***********************************************************************
 ***********************************************************************
 *
 * Error messages
 *
 *)

let proj_codom_mismatch projtype i loc ith_typ codom =
  error_message loc
    ("Type error. Derived\n" ^
     (string_of_ccsl_type projtype) ^
     " for " ^ 
     (match i with 
	| 1 -> "1st"
	| 2 -> "2nd"
	| 3 -> "3rd"
	| _ -> (string_of_int i) ^ "th"
     ) ^ " projection. Type " ^
     (string_of_ccsl_type ith_typ) ^ "\n" ^
     "is not compatible with type " ^
     (string_of_ccsl_type codom) ^ "."
    )

let type_constr_error loc typ constr =
  error_message loc
    ("Type constraint violated. Derived type\n" ^
     (string_of_ccsl_type typ) ^
     " is not compatible with constraint\n" ^
     (string_of_ccsl_type constr)
    )


let invalid_definition loc construct =
  error_message loc
    (construct ^ " is not allowed in definitional extension.");
  raise Typecheck_error

let drc s = 
  if debug_level _DEBUG_TYPECHECK || debug_level _DEBUG_RESOLUTION then
    print_verbose s

let dc s = 
  if debug_level _DEBUG_TYPECHECK then
    print_verbose ( "    C " ^ s)



(***********************************************************************
 ***********************************************************************
 *
 * Type Checking
 *
 *)

let check_ground loc tv_bindings typ message =
  let ntyp = normalize_type tv_bindings typ in
  let _ = 
    if type_is_nonground ntyp then
      begin
	error_message loc
	  ("Type not uniquely determined.\n" ^
	   "Derived " ^
	   (string_of_ccsl_type ntyp) ^
	   " for " ^ message ^ ".\n" ^
	   "Please add appropriate type annotations."
	  );
	raise Typecheck_error
      end
  in
    ntyp

let check_ground_arglist loc tv_bindings iface args message =
  let nargs = normalize_args tv_bindings args in
  let _ = 
    if args_are_nonground nargs then
      begin
	error_message loc
	  ("Type not uniquely determined.\n" ^
	   "Derived " ^
	   (iface#get_name) ^ "::" ^
	   (string_of_ccsl_arguments nargs) ^
	   " for " ^ message ^ ".\n" ^
	   "Please add appropriate type annotations."
	  );
	raise Typecheck_error
      end
  in
    nargs
    

let generate_lifting_name typ =
  let s = string_of_ccsl_type typ in
  let r = String.make (String.length s) 'X' in
  let rec copy i j under =
    if i = String.length s 
    then j
    else
      if (s.[i] = ' ') 
	  || (s.[i] = '[') 
	  || (s.[i] = ']') 
	  || (s.[i] = ',') 
	  || (s.[i] = '-') 
	  || (s.[i] = '>') 
      then (if under
	    then 
	      (r.[j] <- '_';
	       copy (i + 1) (j + 1) false
	      )
	    else
	       copy (i + 1) j false
	   )
      else
	(r.[j] <- s.[i];
	 copy (i + 1) (j + 1) true
	)
  in let n = (copy 0 0 false) in
  let n_ = if (n > 0) && (r.[n-1] = '_') then n-1 else n
  in
    Names.lifting_name_prefix ^ (String.sub r 0 n_)


let rec check_expression ccl loc tv_bindings expr_typ expr = 
  let recurse_exp = check_expression ccl loc tv_bindings in
  let recurse_form = check_formula ccl loc tv_bindings in
  let _ = assert(not (type_is_nonground expr_typ)) in
  let nexpression = 
    match expr with
      | ExprLoc(ex,l) -> 
	  ExprLoc(check_expression ccl l tv_bindings expr_typ ex, l)
					       (* we are in check_expression *)
      | TypeAnnotation(BasicExpr(Member(instiface, 
					Resolved(m))), typ) ->
	  let _ = dc ("start derive for member " ^ m#get_name) in
	  let ntyp = 
	      check_ground loc tv_bindings typ 
		("for member " ^ m#get_name)
	  in
	  let ninstiface = match instiface with
	    | CurrentIface -> CurrentIface
	    | InstIface(iface, args, instloc) -> 
		let nargs = check_ground_arglist loc tv_bindings 
			      iface args
			      ("hosting class of " ^
			       m#get_name)
		in begin
			(* iterate components unless ccl is a signature 
			 * and m is a ground term from it
			 *)
		    if ccl#is_sig && m#hosting_class#get_name = ccl#get_name
		    then
		      ()
		    else begin
		      dc "add some components";
		      iter_component_arglist_add_component ccl
			nargs (get_member_parameters m);
(* Hendrik: don't use add_component for definitions in ground signatures *)
		      ignore(ccl#add_component (Pos,iface,nargs))
		    end;
				(* add a special import if m is coreduce *)
		    (if m#get_sort = Class_Coreduce 
		     then
		       let self1,self2 = match ntyp with
			 | Function(_,Function(self1,self2)) -> self1,self2
			 | _ -> assert(false)
		       in let _ = assert(iface#is_class) 
		       in
			 ccl#add_iface_import
			   (Names.ccsl_finality_theory_name iface,
			    TypeArgument(self1) :: 
			      TypeArgument(self2) :: 
			      (normalize_args tv_bindings nargs))
		    );
		    InstIface(iface,nargs,instloc)
		  end
	    | NoIface -> 
		assert(false)
	  in 
	  let _ = dc ("derive " ^
		      (string_of_ccsl_inst_iface ninstiface) ^
		      " for member " ^
		      m#get_name) 
	  in
	  let _ = assert(types_must_unify expr_typ ntyp) 
	  in
	    TypeAnnotation(
	      BasicExpr(Member(ninstiface, Resolved(m))),
	      expr_typ)
	      
					       (* we are in check_expression *)
      | BasicExpr(TermVar(Resolved(id_rec))) -> 
	  (* if the type of id_rec was not determined completely, 
	   * then this is checked in Let or CCSL_Case
	   *)
	  let _ = assert(types_must_unify expr_typ id_rec.id_type)
	  in
	    expr
				(* other BasicExpr cannot appear here *)
      | BasicExpr(TermVar(Unresolved _))
      | BasicExpr(Member _) -> 
	  assert(false)

					       (* we are in check_expression *)
      | TypedTerm(Term _, typ) -> 
	  begin
	    assert(types_must_unify expr_typ typ);
	    expr
	  end

      | TypedTerm(ex, typ) -> 
	  begin
	    assert(types_must_unify expr_typ typ);
	    TypedTerm( recurse_exp expr_typ ex, typ)
	  end
	  
      | MethodSelection(TypeAnnotation(ex,ex_typ), instiface, m) ->
	  let nex_typ = 
	    check_ground loc tv_bindings ex_typ 
	      "for subject of method selection"
	  in
	    (* if the object is ground the iface must also be ground *)
	  let _ = match instiface with
	    | CurrentIface -> ()
	    | InstIface(iface, args, instloc) ->
		assert(not (args_are_nonground 
			      (normalize_args tv_bindings args)))
	    | NoIface -> assert false

	  in let _ = 
	    iter_components_add_component ccl ex_typ
	  in 
	    MethodSelection(
	      TypeAnnotation(recurse_exp nex_typ ex, ex_typ), 
	      instiface, m) 

				(* MethodSelection needs TypeAnnotation *)
      | MethodSelection _ ->
	  assert(false)

					       (* we are in check_expression *)
      | Tuple(ex_list) -> 
	  let tl = match expand_type_def eq_ccsl_types expr_typ with
	    | Product tl -> tl
	    | _ -> assert(false)
	  in
	    Tuple(List.map2 recurse_exp tl ex_list)

      | Projection(i,_) -> 
	  let expanded_expr_typ = expand_type_def eq_ccsl_types expr_typ in
	  let domtl, codom = 
	    match expanded_expr_typ with
	      | Function(Product tl, codom) -> tl, codom
	      | Function(domtyp,_) -> 
		  begin
		    proj_dom_error loc domtyp;
		    raise Typecheck_error
		  end
	      | _ -> assert(false)
	  in
	  let tuple_len = List.length domtl in
	  let _ = if i > tuple_len then
	    begin
	      proj_i_error expr loc (Product domtl);
	      raise Typecheck_error
	    end
	  in let _ = 
			(* all involved types are ground, 
			 * therefore type context is not changed here *)
	      unify_types tv_bindings (List.nth domtl (i - 1)) codom
		(proj_codom_mismatch expr_typ i loc)
	  in let _ = dc ("derive " ^ (string_of_ccsl_type expr_typ) ^
			 " for " ^ (string_of_ccsl_expression expr))
	  in
	    TypeAnnotation(Projection(i,tuple_len), expanded_expr_typ)

					       (* we are in check_expression *)
      | Abstraction(decl_list,ex) ->
	  let codom = match expand_type_def eq_ccsl_types expr_typ with
	    | Function(_, codom) -> codom
	    | _ -> assert(false)
	  in 
	    Abstraction(decl_list, recurse_exp codom ex)

      | Application( TypeAnnotation(ex1,ex1_typ), ex2) -> 
	  let nex1_typ = check_ground (get_ex_loc ex1) tv_bindings 
			   ex1_typ "function"
	  in
	  let _ = dc ("derive " ^ (string_of_ccsl_type nex1_typ) ^
		      " for fun " ^ (string_of_ccsl_expression ex1))
	  in let dom,codom = match expand_type_def eq_ccsl_types nex1_typ with
	    | Function(dom,codom) -> dom,codom
	    | _ -> assert(false)
	  in let _ = assert(types_must_unify codom expr_typ)
	  in
	    Application( 
	      TypeAnnotation(recurse_exp nex1_typ ex1, nex1_typ),
	      recurse_exp dom ex2)

					(* Application needs TypeAnnotation *)
      | Application _ ->
	  assert(false)

					       (* we are in check_expression *)
      | TypeAnnotation(InfixApplication(ex1, instiface, (Resolved m), ex2), 
		       memtype) ->
	  let ninstiface = match instiface with
	    | InstIface(iface, args, instloc) -> 
		let nargs = check_ground_arglist loc tv_bindings 
			      iface args
			      ("infix operator " ^
			       m#get_name)
		in 
		  begin
			(* iterate components unless ccl is a signature 
			 * and m is a ground term from it
			 *)
		    if ccl#is_sig && m#hosting_class#get_name = ccl#get_name
		    then
		      ()
		    else begin
		      iter_component_arglist_add_component ccl
			nargs (get_member_parameters m);
(* Hendrik: don't use add_component for definitions in ground signatures *)
		      ignore(ccl#add_component (Pos,iface,nargs))
		    end;
		    InstIface(iface,nargs,instloc)
		  end
				(* may happen in a top level definition *)
	    | CurrentIface -> CurrentIface

	    | NoIface -> 
		assert(false)
	  in 
	  let nmemtype = check_ground loc tv_bindings memtype 
			   "infix operator"
	  in
	  let _ = dc ("derive " ^ (string_of_ccsl_type nmemtype) ^
		      " for infix " ^ m#get_name )
	  in
	  let lefttype, righttype, resulttype =
	    (match expand_type_def eq_ccsl_types nmemtype with
	       | Function(Product [t1;t2], t3) -> t1, t2, t3
	       | Function(t1, Function(t2,t3)) -> t1, t2, t3
	       | _ -> assert(false)
	    )
	  in let _ = assert(types_must_unify resulttype expr_typ)
	  in
	    TypeAnnotation(
	      InfixApplication(
		recurse_exp lefttype ex1,
		ninstiface, Resolved m,
		recurse_exp righttype ex2),
	      nmemtype)
					       (* we are in check_expression *)

				(* InfixApplication needs TypeAnnotation *)
      | InfixApplication _ ->
	  assert(false)

      | FunUpdate(fex, changes) -> 
	  let dom_type, codom_type = 
	    match expand_type_def eq_ccsl_types expr_typ with
	      | Function(dom,codom) -> dom,codom
	      | _ -> assert(false)
	  in 
	    FunUpdate(
	      recurse_exp expr_typ fex,
	      List.map (fun (arg,nval) ->
			  (recurse_exp dom_type arg, 
			   recurse_exp codom_type nval))
		changes)

      | Let(decl_list, ex) ->
	  let ndecl_list = 
	    List.map (fun (id_rec, typeopt, ex) ->
			let ntyp_ex = 
			  check_ground (get_ex_loc ex) tv_bindings 
			    id_rec.id_type "expression"
			in let _ = dc ("derive " ^
				       (string_of_ccsl_type ntyp_ex) ^
				       " for let var " ^
				       id_rec.id_token.token_name)
			in let _ = id_rec.id_type <- ntyp_ex 
			in
			  (id_rec, typeopt, recurse_exp ntyp_ex ex)
		     )
	      decl_list
	  in
	    Let(ndecl_list,
		recurse_exp expr_typ ex)

					       (* we are in check_expression *)
      | If(conds,else_ex) -> 
	  let nconds = 
	    List.map 
	      (fun (cond,ex) -> (recurse_form Bool cond,
				 recurse_exp expr_typ ex)
	      )
	      conds
	  in
	    If(nconds, 
	       recurse_exp expr_typ else_ex)

					       (* we are in check_expression *)
      | CCSL_Case(TypeAnnotation(ex,ex_typ), variants) -> 
	  let adt_type = check_ground (get_ex_loc ex) tv_bindings
			   ex_typ "expression"
	  in let _ = dc ("derive " ^
			 (string_of_ccsl_type adt_type) ^
			 " for case expr " ^
			 (string_of_ccsl_expression ex))
	  in let nex = recurse_exp adt_type ex in
	  let do_variant (const,args,ex) = 
	    let _ = List.iter
		      (fun id_rec ->
			 let nid_typ = (try
					  check_ground loc tv_bindings 
					    id_rec.id_type "INTERNAL ERROR"
					with
					  | Typecheck_error -> 
					      assert(false)
				       )
			 in 
			   id_rec.id_type <- nid_typ
		      )
		      args
	    in
	      (const,args,
	       recurse_exp expr_typ ex)
	  in
	  let nvariants = List.map do_variant variants 
	  in
	    CCSL_Case(
	      TypeAnnotation(nex, adt_type),
	      nvariants)

					       (* we are in check_expression *)

					(* CCSL_Case needs a TypeAnnotation *)
      | CCSL_Case _ -> 
	  assert(false)

      | Modality(modal,typ,pred,tlist) ->
	  let _ = assert( types_must_unify expr_typ (Function(typ,Bool)) )
	  in
	    Modality(modal, typ,
		recurse_exp expr_typ pred,
		tlist)

      | Expression form ->
	  Expression(recurse_form expr_typ form)

					       (* we are in check_expression *)

			(* type annotations are already catched above *)
      | TypeAnnotation _
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
	  in 
	    nexpression


and check_formula ccl loc tv_bindings formula_type formula =
  let recurse_exp = check_expression ccl loc tv_bindings in
  let recurse_form = check_formula ccl loc tv_bindings in
  let _ = assert(not (type_is_nonground formula_type)) in
  let nformula = 
    match formula with
      | FormLoc(f,l)    -> 
	  FormLoc(check_formula ccl l tv_bindings formula_type f, l)
      | True 		-> assert(types_must_unify formula_type Bool); formula
      | False 		-> assert(types_must_unify formula_type Bool); formula
      | Not f 		-> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Not(recurse_form formula_type f)
	  end
      | And f_list 	-> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    And(List.map (recurse_form formula_type) f_list)
	  end
      | Or f_list 	-> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Or(List.map (recurse_form formula_type) f_list)
	  end
						  (* we are in check_formula *)
      | Implies(assum,concl) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Implies( recurse_form formula_type assum, 
		     recurse_form formula_type concl)
	  end
      | Iff(assum,concl) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Iff( recurse_form formula_type assum, 
		 recurse_form formula_type concl)
	  end
      | Equal( TypeAnnotation(ex_a,eq_typ), ex_b) -> 
	  let _ = assert(types_must_unify formula_type Bool) in
	  let neq_typ = check_ground loc tv_bindings eq_typ 
			  "both sides of equality"
	  in let _ = dc ("derive " ^
			 (string_of_ccsl_type neq_typ) ^
			 " for equality")
	  in
	    Equal( TypeAnnotation(recurse_exp neq_typ ex_a, neq_typ),
		   recurse_exp neq_typ ex_b)

					(* Equal needs TypeAnnotation *)
      | Equal _ -> assert(false)

      | Forall(quant_list, f) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Forall(quant_list, recurse_form formula_type f)
	  end
						  (* we are in check_formula *)
      | Exists(quant_list, f) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Exists(quant_list, recurse_form formula_type f)
	  end
      | Formula ex -> 
	  Formula(recurse_exp formula_type ex)

						  (* we are in check_formula *)

					(* t equals None *)
      | Obseq(t, TypeAnnotation(ex_a,eq_typ), ex_b) ->		
	  let _ = assert(types_must_unify formula_type Bool) in
	  let neq_typ = check_ground loc tv_bindings eq_typ 
			  "both sides of equality" 
	  in let _ = dc ("derive " ^
			 (string_of_ccsl_type neq_typ) ^
			 " for beh equality")
					(* check for greatest bisim *)
	  in let violation = ref false in
	  let _ = iter_components Pos 
		    (fun _ typ -> 
		       let args = match typ with
			 | Class(_,args) -> args
			 | Adt(_,_,args) -> args
			 | Groundtype(_,args) -> args
			 | _ -> assert(false)
		       in
			 if (not (type_has_feature HasRelLiftFeature typ)) &&
			   (count_self_args args <> 0)
			 then violation := true
		    ) neq_typ
	  in let _ = (if (count_self neq_typ) <> 0 && 
			(not (ccl#has_feature HasGreatestBisimFeature))
		      then violation := true)
	  in let _ = 
	      if !violation then
		if !expert_mode then
		  warning_message loc 
		    ("Observational equivalence for type " ^
		     (string_of_ccsl_type neq_typ) ^
		     " is not well defined.")
		else
		  begin
		    error_message loc 
		      ("Observational equivalence for type " ^
		       (string_of_ccsl_type neq_typ) ^
		       " is not well defined.");
		    raise Typecheck_error
		  end

						  (* we are in check_formula *)
	  in let obsname =
	      (match neq_typ with
					(* Self gets Bisim later *)
		 | Self -> ""
					(* gets bisim for that class *)
		 | Class(cl,[]) -> ""
		 | neq_typ ->
		     try
		       ccl#find_rel_lifting_for_type neq_typ
		     with
		       | Not_found ->
			   let tname = generate_lifting_name neq_typ in
			   let _ = ccl#add_rel_lifting tname neq_typ 
			   in
			     tname
	      )
	  in
	    Obseq( Some(obsname, neq_typ), 
		   recurse_exp neq_typ ex_a, 
		   recurse_exp neq_typ ex_b)

						  (* we are in check_formula *)

					(* Obseq needs TypeAnnotation *)
      | Obseq _ -> assert(false)

					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | Bisim _
      | ConstantPredicate _
	-> assert(false)
  in 
    nformula
    


let check_assertion_formula ccl loc tv_bindings formula =
  let nformula = check_formula ccl loc tv_bindings Bool formula 
  in
    nformula


let check_definition_expression ccl loc tv_bindings typ expr =
  let nex = check_expression ccl loc tv_bindings typ expr
  in
    nex



(***********************************************************************
 ***********************************************************************
 *
 * definition checking
 *
 *)

let rec check_valid_definition_expr loc expr = 
  let recurse_exp = check_valid_definition_expr loc in
  let recurse_form = check_valid_definition_form loc in
    match expr with
      | ExprLoc(ex,loc) -> check_valid_definition_expr loc ex
      | BasicExpr bexpr -> ()
      | TypedTerm(Term _, typ) -> ()
      | TypedTerm(ex, typ) -> recurse_exp ex
      | MethodSelection(ex, typeopt, m) -> recurse_exp ex
      | Tuple(ex_list) -> List.iter recurse_exp ex_list
      | Projection(i,_) -> ()
      | Abstraction(decl_list,ex) -> recurse_exp ex
      | Application(ex1,ex2) -> recurse_exp ex1; recurse_exp ex2
      | InfixApplication(ex1, iface, tokcontainer, ex2) ->
	  recurse_exp ex1; recurse_exp ex2 
      | FunUpdate(fex, changes) -> 
	  recurse_exp fex;
	  List.iter (fun (ex1,ex2) -> (recurse_exp ex1; recurse_exp ex2))
	    changes
      | Let(decl_list, ex) ->
	  List.iter (fun (id_rec, typopt, ex) -> recurse_exp ex)
	    decl_list;
	  recurse_exp ex
      | If(conds,ex) -> 
	  List.iter (fun (f,ex) -> (recurse_form f; recurse_exp ex)) conds;
	  recurse_exp ex
      | CCSL_Case(ex,variants) -> 
	  recurse_exp ex;
	  List.iter (fun (mem,ids,exp) -> recurse_exp exp) variants
      | Modality(modal,typ, pred, tlist) ->
	  (match typ with
	     | Self -> invalid_definition loc "Modal operator"
	     | _ -> recurse_exp pred
	  )

      | Expression form -> recurse_form form
      | TypeAnnotation(exp,typ) -> recurse_exp exp
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
	
and check_valid_definition_form loc formula =
  let recurse_exp = check_valid_definition_expr loc in
  let recurse_form = check_valid_definition_form loc in
    match formula with
      | FormLoc(f,loc) -> check_valid_definition_form loc f
      | True -> ()
      | False -> ()
      | Not f -> recurse_form f
      | And f_list -> List.iter recurse_form f_list
      | Or f_list -> List.iter recurse_form f_list
      | Implies(assum,concl) -> 
	  recurse_form assum; 
	  recurse_form concl
      | Iff(assum,concl) -> 
	  recurse_form assum;
	  recurse_form concl
      | Equal(ex_a,ex_b) ->
	  recurse_exp ex_a; 
	  recurse_exp ex_b
      | Forall(decll, f) -> recurse_form f
      | Exists(decll, f) -> recurse_form f
      | Formula ex -> recurse_exp ex
      | Obseq(Some(obsname, eq_typ), ex1,ex2) -> 
	  if count_self eq_typ <> 0 
	  then
	    invalid_definition loc "Observational equality involving Self"

      | Obseq _ 
					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | Bisim _
      | ConstantPredicate _
	-> assert(false)







(***********************************************************************
 ***********************************************************************
 *
 * Assertions and classes
 *
 *)


let resolve_typecheck_assertion ccl assertion = 
  if not assertion.is_generated 
  then
    match assertion.assertion_formula with
      | Symbolic (FormLoc(formula,loc)) -> 
	  begin
	    
	    (* resolve *)
	    
	    drc ("  * Resolve assertion " ^ 
		         assertion.assertion_name.token_name);
	    ignore(Symbol.start_block());
	    (match assertion.self_variable with
	       | None -> ()
	       | Some id_rec -> Symbol.create_var id_rec
	    );
	    List.iter Symbol.create_var assertion.free_variables;
	    let res_form = resolve_formula ccl loc formula in
	    let _ = ignore(Symbol.close_block()) in

	      (* derive *)

	    let _ = drc ("  * TC assertion " ^ 
			 assertion.assertion_name.token_name)
	    in let tv_bindings = new_tv_binding () in
	    let der_form = 
	      derive_assertion_formula ccl loc tv_bindings res_form
	    in

	    (* check *)
	      
	    let check_form = 
	      check_assertion_formula ccl loc tv_bindings der_form
	    in
	      assertion.assertion_formula <- Symbolic(FormLoc(check_form,loc))
	  end
	  
      | Symbolic _ -> assert(false)
      | Pvs_String _
      | Isa_String _ -> ()



(* Process a definition in a ground signature or in a class
 * 
 * From the parser the definition comes in as a formula, which 
 * should better be an equation.
 * From the left hand side of the equation I extract variable declarations, 
 * by matching these variables against the declared type.
 * After that the left hand side is thrown away, the extracted variables 
 * are stored in the definition record and the right hand side is wrapped
 * in a TypeAnnotation to remember its supposed type
 *)

let resolve_typecheck_symbolic_definition ccl def =
  let def_eq = match def.definition with
    | Symbolic eq -> eq
    | _ -> assert(false)
  in let mem = def.defined_method in
				(* create the block for this equation *)
  let local_block = Symbol.start_block() in
  let _ = drc ("  * Resolve definition " ^ def.defined_method#get_name) 
  in

					(* create local type parameters *)
  let _ = List.iter (function TypeParameter id_rec ->
		       Symbol.create_type_parameter id_rec) 
	    mem#get_local_parameters
  in					(* extract left and right hand sides *)
  let left, right, right_loc = 
    match def_eq with
      | FormLoc(Equal(left, ExprLoc(right,right_loc)), _)
	  -> left,right,right_loc
      | _ -> 
	  begin
	    error_message (get_form_loc def_eq)
	      "Definition not in form of an equation.";
	    raise Resolution_error
	  end

		(* recursively extract variables from the left hand side 
		 * take care to handle cases like
		 *   f = ...
		 *   f x y = ...
		 *   f (x,y) (a,b) = ...
		 *   (x + y) (a,b) = ...
		 *)
  in let rec get_args accu = function
					(* base case normal application *)
    | ExprLoc(Application(
		ExprLoc(BasicExpr(TermVar(Unresolved defname)),_),
		args),_) 
	when defname.token_name = mem#get_name ->
	args :: accu

			(* we are in resolve_typecheck_symbolic_definition *)

					(* base case def =  *)
    | ExprLoc(BasicExpr(TermVar(Unresolved defname)),_)
	when defname.token_name = mem#get_name 
	  ->
	accu
					(* base case with infixes *)
    | ExprLoc(InfixApplication(leftex, _, Unresolved op, rightex), loc) 
	when op.token_name = mem#get_name ->
	(match mem#get_full_type with
	   | Function(Product [_;_], _) -> 
	       ExprLoc(Tuple([leftex;rightex]), loc) :: accu
	   | Function(_, Function(_)) -> leftex :: rightex :: accu
	   | _ -> assert false
	)
					(* non base case *)
    | ExprLoc(Application(innerterm, args),_) ->
	get_args (args :: accu) innerterm
    | _ ->				(* error no application *)
	begin
	  error_message (get_ex_loc left)
	    ("Wrong form of definition. Expected \"" ^ 
	     (if mem#get_sort = InfixGroundTerm 
	      then
		"x " ^ mem#get_name ^ " y\""
	      else
		mem#get_name ^ " ... =\"")
	    );
	  raise Resolution_error
	end
  in let arg_list_list = get_args [] left in

			(* we are in resolve_typecheck_symbolic_definition *)

  (* pair arguments with types; compute right hand side type *)

			(* pair one argument and one type in a id record *)
  let make_arg_id typ = function
    | ExprLoc(BasicExpr(TermVar(Unresolved tok)), vloc) ->
	let _ =				(* check for doubble occurences *)
	  try
	    ignore(Symbol.find_local local_block tok.token_name);
	    error_message vloc
	      ("Variables on the left hand side of a defining " ^
	       "equation must be unique.");
	    raise Resolution_error
	  with 
	    | Table.Not_defined -> ()
	in 
	let id_rec = Symbol.identifier_record tok CCSL_Var
	in
	  id_rec.id_type <- typ;
	  Symbol.create_var id_rec;
	  id_rec
    | v ->
	begin
	  error_message (get_ex_loc v) "Variable expected.";
	  raise Resolution_error
	end
  in
			(* we are in resolve_typecheck_symbolic_definition *)

				(* recursively match arguments and types *)
  let rec match_args argslist funtype = 
    let recurse restargs codomain ids =
      let rest_ids, typ = match_args restargs codomain 
      in
	((ids :: rest_ids), typ)
    in
      match argslist,funtype with
	| ExprLoc(Tuple arggroup, loc ) :: restargs, 
	  Function(Product grouptyp, codomain)
	  -> 
	    if List.length grouptyp <> List.length arggroup 
	    then
	      begin
		error_message loc
		  "Number of arguments does not match type declaration.";
		raise Resolution_error
	      end
	    else
	      recurse restargs codomain
		(List.map2 make_arg_id grouptyp arggroup)
	| (ExprLoc _ as singlearg) :: restargs, 
	  Function(typ, codomain) ->
	    recurse restargs codomain [make_arg_id typ singlearg]
	| _ , Groundtype(idrec, args) when is_type_def idrec ->
	    (* do not use recurse here, 
	     * because we ded not produce any binding
	     *)
	    match_args argslist (expand_type_def eq_ccsl_types funtype)

	| [], resttyp -> [], resttyp
	| ExprLoc(_, loc) :: _, _ ->
	    error_message loc "Superfluous variable.";
	    raise Resolution_error
	| _ -> 
	    assert false
			(* we are in resolve_typecheck_symbolic_definition *)

  in let id_list_list, righttyp = match_args arg_list_list mem#get_full_type
  in let _ = def.variables <- id_list_list in

  (* do resolution *)

  let res_right = resolve_expression ccl right_loc right in
  let _ = ignore(Symbol.close_block()) in


  (* do type derivation *)

  let _ = drc ("  * TC definition " ^ mem#get_name) in
  let tv_bindings = new_tv_binding () in
  let der_right = derive_definition_expression ccl right_loc tv_bindings 
		    righttyp res_right
  in

  (* do type check *)

  let check_right = check_definition_expression ccl right_loc tv_bindings 
		      righttyp der_right
  in
  let _ = (if not !expert_mode 
	   then check_valid_definition_expr right_loc check_right
	   else ())
  in
    def.definition <- 
    Symbolic(FormLoc(Formula(TypeAnnotation(check_right, righttyp)), 
		     right_loc))


let resolve_typecheck_definition ccl def =
  match def.definition with
    | Symbolic _ -> resolve_typecheck_symbolic_definition ccl def
    | _ -> ()


(***********************************************************************
 ***********************************************************************
 *
 * top level units
 *
 *)


let resolve_typecheck_class ccl = 
  let nest = Symbol.nesting_size() in 
  let original_local = ccl#get_local
  in begin
      drc (" ** Resolve/Typecheck class " ^ ccl#get_name);
      (* d (ccl#dump_iface); *)
      ccl#set_local (Symbol.start_block());

      (* print_verbose( "Symboltable: " );
       * print_verbose (Symbol.symbol_table_dump 
       * 			  Top_variant_types_util.dump_symbol_brief);
       *)

      List.iter (function TypeParameter id_rec ->
		   Symbol.create_type_parameter id_rec) ccl#get_parameters;
      List.iter (fun acl -> List.iter Symbol.create_member
		     acl#get_all_actions)
	ccl#get_resolved_ancestors;
      List.iter Symbol.create_member ccl#get_sig_actions;

      List.iter (fun def -> 
		   resolve_typecheck_definition ccl def;
		   Symbol.create_member def.defined_method
		)
	ccl#get_definitions;
      assert(Symbol.nesting_size() = nest + 1);

         (* make class specials available in assertions *)
      List.iter (fun mem -> (if mem#get_sort = Class_Sig_Special
			     then
			       Symbol.create_member mem)
		)
	ccl#get_members;

      (* print_verbose( "Symboltable: " );
       * print_verbose (Symbol.symbol_table_dump 
       * 			  Top_variant_types_util.dump_symbol_brief);
       *)

      List.iter 
	(fun ass -> resolve_typecheck_assertion ccl ass) 
	ccl#get_assertions;
      assert(Symbol.nesting_size() = nest + 1);

      List.iter (fun mem -> Symbol.create_member mem) ccl#get_constructors;

      List.iter 
	(fun crea -> resolve_typecheck_assertion ccl crea) 
	ccl#get_creations; 

      List.iter 
	(fun thm -> resolve_typecheck_assertion ccl thm) 
	ccl#get_theorems; 

      (* print_verbose( "Symboltable: " );
       * print_verbose (Symbol.symbol_table_dump 
       * 			  Top_variant_types_util.dump_symbol_brief);
       *)

      assert(Symbol.nesting_size() = nest + 1);
      ignore(Symbol.close_block());
      ccl#set_local original_local;
      Symbol.create_class_content ccl
    end


let resolve_typecheck_adt adt =
  begin
    drc (" ** Resolve/Typecheck adt in " ^ adt#get_name );
    Symbol.create_adt_content adt
  end


let resolve_typecheck_sig si =
  let nest = Symbol.nesting_size() in
  let original_local = si#get_local in
			(* pair the definitions with the declarations *)
  let rec pair_consts members definitions =
    match members,definitions with
      | [],[] -> []
      | m::ms,[] ->  (m,None) :: (pair_consts ms definitions)
      | [], d::_ -> assert(false)
      | m::ms, d::ds ->
	  if m == d.defined_method 
	  then
	    (m, Some d) :: (pair_consts ms ds)
	  else
	    (m, None) :: (pair_consts ms definitions)
  in
  let const_list = pair_consts si#get_members si#get_definitions
  in
    begin
      drc (" ** Resolve/Typecheck signature in " ^ si#get_name );
      si#set_local (Symbol.start_block());
      List.iter (function TypeParameter id_rec ->
		   Symbol.create_type_parameter id_rec) si#get_parameters;

      List.iter (fun (m,defopt) -> 
		   (match defopt with
		      | None -> ()
		      | Some def -> resolve_typecheck_definition si def
		   );

		   Symbol.create_member m
		)
	const_list;

      assert(Symbol.nesting_size() = nest + 1);
      ignore(Symbol.close_block());
      si#set_local original_local;
      Symbol.create_sig_content si
    end


let typecheck_ast = function
  | CCSL_class_dec cl -> resolve_typecheck_class cl
  | CCSL_adt_dec adt -> resolve_typecheck_adt adt
  | CCSL_sig_dec si -> resolve_typecheck_sig si


let ccsl_resolution_typecheck_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter typecheck_ast ast;;



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

