(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Wednesday 30 June 10 16:53:03 tews@blau.inf.tu-dresden.de>
 *
 * generic classes for all ccsl class and adt theories
 *
 * $Id: theory_class.ml,v 1.21 2010-07-02 10:55:56 tews Exp $
 *
 *)

(***********************************************************************
 ***********************************************************************
 *
 * The inheritance tree looks like this:
 * 
 * 				top_theory_body
 * 				       |
 * 				       |
 * 				      \/
 * 				ccsl_pre_theory_body
 * 			       /    |      	 \
 * 		 -------------/     |		  \
 * 		/	       	    |		   \
 * 	ccsl_sig_theory      ccsl_adt_theory    ccsl_pre_virtual_class_theory
 * 	      |    	            |              |
 * 	      |                     |              |
 * 	      |		            |        	   |
 * 	ground signature     all adt theories   ccsl_pre_class_theory
 *        theories                               (all class theories)
 * 
 *
 * some of those classes are currently empty (ie. do not add any features)
 * but who knows, whats happening ...
 *
 *)

open Util
open Top_names
open Top_variant_types
open Top_classtypes
open Types_util
open Top_classes
open Name_space
open Names
open Classtypes
open Logic_util
;;


class virtual ['class_type, 'member_type ] ccsl_pre_theory_body_class 
  (cl : 'class_type)
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_theory_body_type
  = object (self : 'self)
    inherit ['class_type, 'member_type] top_pre_theory_body_class

    val self1 = 
		 { id_token = { token_name = (name_of_self^"1");
				loc = None };
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }

    val self2 = 
		 { id_token = { token_name = (name_of_self^"2");
				loc = None};
		   (* the other fields are to ignored *)
		   id_type = Self;
		   id_origin = CCSL_TypeParameter;
		   id_parameters = [];
		   id_variance = Unset;
		   id_sequence = -1;
		   id_components = [];
		 }

    method self1_parameter = TypeParameter self1

    method self2_parameter = TypeParameter self2

    val self_argument = 
      (TypeArgument Self : ('class_type, 'member_type) top_pre_argument_type)

    method self1_argument = TypeArgument(BoundTypeVariable self1)

    method self2_argument = TypeArgument(BoundTypeVariable self2)

    val orig_parameters = cl#get_parameters
			    
    val orig_arguments = arguments_from_parameters(cl#get_parameters)

    method simple_parameters = self_parameter :: orig_parameters

    method simple_arguments = self_argument :: orig_arguments 

    method double_self_parameters = 
      self#self1_parameter :: self#self2_parameter :: orig_parameters

    method double_self_arguments =
      self#self1_argument :: self#self2_argument :: orig_arguments


    method private make_double_parameter_ids = 
      let change_id id s =
	{id with id_token = { token_name = (id.id_token.token_name ^ s);
			      loc = None}}
      in
	List.map 
	  (function
	     | TypeParameter id -> (change_id id "1", change_id id "2")
	  )
	  orig_parameters

    val mutable store_double_ids =
        (None : 
	   (('class_type, 'member_type) top_pre_identifier_record_type *
	    ('class_type, 'member_type) top_pre_identifier_record_type )
	   list option)

    method double_parameter_ids = match store_double_ids with
      | Some l -> l
      | None -> (let l = self#make_double_parameter_ids
		 in
		   store_double_ids <- Some l;
		   l)

    method double_parameters = 
	List.split
	  (List.map 
	     (fun (id1,id2) -> (TypeParameter id1, TypeParameter id2))
	     self#double_parameter_ids)

    method double_arguments = 
	List.split
	  (List.map 
	     (fun (id1,id2) -> 
		(TypeArgument (BoundTypeVariable id1), 
		 TypeArgument (BoundTypeVariable id2)))
	     self#double_parameter_ids)

    val ns = Name_space.create eq_types


	  (* the same is already hardcoded in assert_coalgebra_type 
	   * 
	   *)
    val coalgebra_term = (Term(name_of_coalgebra,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)

    val algebra_term = (Term(name_of_algebra,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)
	
    method get_member_fun (mem : 'member_type) =
      let top_class = 
	match mem#hosting_class#get_heir with
	  | None -> mem#hosting_class
	  | Some ocl -> ocl
      in let in_this_class = top_class#get_name = cl#get_name
      in
	match mem#get_sort with
	  | Proper_Attribute _
	  | Update_Method
	  | Defined_Method
	  | Normal_Method
	  | Class_Coreduce ->
	      if in_this_class 
	      then Some coalgebra_term
	      else Some top_class#get_model_coalgebra
	  | Var_Constructor
	  | Const_Constructor ->
	      if in_this_class 
	      then Some algebra_term
	      else Some top_class#get_model_algebra

	  | Class_Sig_Special
	  | Class_Map
	    -> None

	  | Adt_Var_Constructor
	  | Adt_Const_Constructor
	  | Adt_Accessor
	  | Adt_Recognizer
	  | Adt_Special
	  | GroundTerm
	  | InfixGroundTerm 
	    -> assert(false)
      
    method get_iface_fun (typ : ('class_type, 'member_type) top_pre_types) =
      match typ with
	| Class(ocl,args) -> 
	    if ocl#get_name = cl#get_name 
					(* Change for -final switch *)
	    then coalgebra_term
	    else ocl#get_model_coalgebra
	| Self -> coalgebra_term
	| _ -> assert(false)


       (* reserve all method/constructor/ancestor names *)
    initializer 
      begin
        reserve ns 
          (List.map (fun m -> m#get_name) cl#get_all_members);
        reserve ns
          (List.map 
             (function
                | Resolved_renaming (ifa,_,_,_,_) -> 
                    super_label ifa
                | Unresolved_renaming _      (* no other stuff *)
		  -> 
                    assert(false)
             )
             cl#get_ancestors)

      end

  end (* ccsl_pre_theory_body_class  *)


class virtual ['class_type, 'member_type ] 
  ccsl_pre_class_theory_class 
  (cl : 'class_type)
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_virtual_class_theory_type
  = object (self : 'self)
    inherit ['class_type, 'member_type] ccsl_pre_theory_body_class cl eq_types


	  (* the same is already hardcoded in assert_coalgebra_type 
	   * 
	   *)
    val coalgebra1_term = (Term(name_of_coalgebra1,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)

    val coalgebra2_term = (Term(name_of_coalgebra2,Always,[]) : 
			  ('class_type, 'member_type) top_pre_expressions)

    (*******************************************************************
     *
     * class specific things to be computed only once
     *)

					(* old version below *)
    method get_method_functor_type = 
      Record(
	(List.map
	   (function
	      | Resolved_renaming (_,args,_,ianc,_) -> 
		  (super_label ianc,
		   IFace(ianc, Always, self_argument :: args))
	      | Unresolved_renaming _ 
		  (* no java stuff *)
		-> 
		  assert(false)
	   )
	   cl#get_ancestors)
	@
	(List.map
	   (fun m ->
	      (method_label m,
	       m#get_full_type
	      ))
	   cl#get_sig_actions))

       (* because PVS does not have primitive Coproducts, we formalize
	* the constructors as a record of functions
	*)
    method get_constructor_functor_type =
      Record(List.map
	       (fun m -> (if m#get_sort = Const_Constructor then
			    m#get_name, m#get_codomain
			  else
			    m#get_name, Function(m#get_domain, 
						 m#get_codomain)))
	       cl#get_constructors)


       (* the coalgebra type *)
    method coalgebra_type =
      IFace( cl, Always,self#simple_arguments)

    method coalgebra1_type = 
      IFace( cl, Always, self#self1_argument :: orig_arguments)

    method coalgebra2_type = 
      IFace( cl, Always, self#self2_argument :: orig_arguments)

    method assert_named_coalgebra_hypothesis coalgebra_name =
      Formula(
	Application(
	  Expression(ConstantPredicate(name_of_assert cl)),
	  Term(coalgebra_name, Always, [])))

    method assert_coalgebra_hypothesis = 
      self#assert_named_coalgebra_hypothesis name_of_coalgebra

    method creation_algebra_hypothesis =
      Formula(
	Application(
	  Application(
	    Expression(ConstantPredicate(name_of_creation cl)),
	    Term(name_of_coalgebra,Always,[])),
	  Term(name_of_algebra,Always,[])))

      (* should only be used in PVS specific code 
       * for self_argument use either
       * self_argument       - for Self 
       * self#self1_argument - for Self1
       * self#self2_argument - for Self2
       *)
    method assert_coalgebra_type self_argument = 
      Predtype(Formula(Term(name_of_assert cl,
    			    Always,
			    self_argument :: orig_arguments)))

      (* This is a bit strange! Since we model the algebra functor
       * as Record of functions (instead of a sum of the domains) the 
       * type of the algebra is then just this record 
       * (instead of a function going to Self)
       *)
    method algebra_type =
      TypeConstant(name_of_constructor_functor cl,Always,self#simple_arguments)

      (* the coalgebra declaration *)
    method coalgebra_decl = 
      	Vardecl(name_of_coalgebra, self#coalgebra_type)

    method coalgebra1_decl = 
      	Vardecl(name_of_coalgebra1, self#coalgebra1_type)

    method coalgebra2_decl = 
      Vardecl(name_of_coalgebra2, self#coalgebra2_type)

    (* assert_coalgebra_decl's should not be used, at least not in lemmas,
     * because this would give wrong lemmas in the isabelle output.
     *
     * method assert_coalgebra_decl =
     *   Vardecl(name_of_coalgebra, self#assert_coalgebra_type)
     *)

      (* the algebra declaration *)
    method algebra_decl = 
      	Vardecl(name_of_algebra, self#algebra_type)

    (* return the tuple type for the coalgebra-as-tuple variant *)
    method coalgebra_tuple_type self_type =
      let m_types = 
	List.map (fun m -> m#get_full_type) cl#get_all_sig_actions in
      (* avoid one-tuples *)
      let original_tuple = match m_types with
	| [] -> assert false
	| [mt] -> mt
	| mts -> Product mts
      in
      substitute_types_only eq_types [(Self, self_type)] original_tuple

      (* helper method for the struct versions of invariant,
       * and assert
       * returns a tuple, 
       * first element is a declaration list for all (method-) 
       * members of the tuple
       * second element is the corresponding coalgebra term 
       * (IFACE_struct applied to the tuple).
       *
       * !! this code relies on the traversion order of 
       * !! get_all_members 
       *)
    method coalgebra_as_tuple sub_ns =
      let method_name m = (("o_" ^ m#get_name), m#get_full_type) in
      let pre_mems = List.map method_name cl#get_all_sig_actions in
      let mems = create_ids_with_preference sub_ns pre_mems in
      let decl_list = 
	List.map (fun (name,typ) -> Undeclared(name,typ)) mems in
      let coalgebra_from_tuple =
	SmartApplication(
	  Term(name_of_struct_of cl, Always,[]),
	  List.map (fun (id,_) -> Term(id,Always,[])) mems
	)
      in
	decl_list, coalgebra_from_tuple

      (* same as coalgebra_as_tuple but returns two tuples corresponding 
       * to two coalgebras
       *
       * !! this code relies on the traversion order of 
       * !! get_all_members 
       *)
    method two_coalgebras_as_tuple sub_ns =
      let prepare_method prefix self_subst m = 
	((prefix ^ m#get_name), 
	 substitute_types_only eq_types self_subst m#get_full_type)
      in
      let all_act = cl#get_all_sig_actions in
      let n = List.length all_act in
      let pre_mems = 
	(List.map (prepare_method "o1_" [Self, BoundTypeVariable self1])
	   all_act) @
	(List.map (prepare_method "o2_" [Self, BoundTypeVariable self2])
	   all_act)  in
      let all_names = create_ids_with_preference sub_ns pre_mems in
      let s1_names = take all_names n in
      let s2_names = tail all_names n in
      let decl1_list = 
	List.map (fun (name,typ) -> Undeclared(name,typ)) s1_names in
      let decl2_list = 
	List.map (fun (name,typ) -> Undeclared(name,typ)) s2_names in
      let make_coalgebra mems =
	SmartApplication(
	  Term(name_of_struct_of cl, Always,[]),
	  List.map (fun (id,_) -> Term(id,Always,[])) mems
	) in
      let c1_from_tuple = make_coalgebra s1_names in
      let c2_from_tuple = make_coalgebra s2_names 
      in
	(decl1_list, decl2_list, c1_from_tuple, c2_from_tuple)
  end

(*******************************************************************
 *******************************************************************
 *
 * special class only for the method lifting related classes
 *
 *)
class ccsl_method_inv_helper_class (cl: 'class_type)
  :  ccsl_method_inv_helper_type
  = object (self: 'self)

    val pre_list_adt =             
      (match Symbol.find  "list" with 
               | AdtSymb( adt ) -> adt
               | _ -> assert(false)
      )

(* for method invariance theories only: *)
      
    val mutable name_of_list = "" (* needs to be set!! *)

    val mutable name_of_step = ""

    val mutable name_of_path1 = ""
    
    val mutable name_of_path2 = ""

    method private list_adt = pre_list_adt
	      
    method private method_enum_type =
      TypeConstant(name_of_method_enum_type cl, Always, [])

    method private list_type =
      Adt(pre_list_adt, Always, [TypeArgument(self#method_enum_type)])

    method private list_term =
      ( Term(name_of_list, Always, []) : ccsl_expressions)

    method private method_term =
      ( Term(name_of_method_id, Always, []) : 
	ccsl_expressions)

    method private list_decl =
      Vardecl(name_of_list,self#list_type)
     
    method private method_decl =
      Vardecl(name_of_method_id, self#method_enum_type)

       (* compute list of methods that need lifting *)
    val mutable lifting_methods = None

    method private get_lifting_methods =
      match lifting_methods with
	| Some ms -> ms
	| None -> 
	    let methods = 
	      List.filter (fun m -> m#is_method && m#needs_lifting) 
		cl#get_all_members
	    in
	      lifting_methods <- Some methods;
	      methods

  end

(*******************************************************************
 *******************************************************************
 *
 * Adt's
 *
 *)

class virtual ['class_type, 'member_type ] ccsl_pre_adt_theory_class 
  adt
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_virtual_adt_theory_type
  = object
    inherit ['class_type, 'member_type] ccsl_pre_theory_body_class adt eq_types

    method this_adt_type = Adt(adt, Isabelle_only, orig_arguments)
  end


(*******************************************************************
 *******************************************************************
 *
 * Ground signatures
 *
 *)

class virtual ['class_type, 'member_type ] ccsl_pre_sig_theory_class 
  si
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_virtual_sig_theory_type
  = object
    inherit ['class_type, 'member_type] ccsl_pre_theory_body_class si eq_types

  end



(*******************************************************************
 *******************************************************************
 *
 * Isabelle Starters/Stoppers
 *
 *)

class ['class_type, 'member_type] 
  ccsl_pre_isabelle_delimiter_theory_class 
  (cl : 'class_type) 
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  delimiter 
  (file_name : string)
  (isar_imports : string list)
  (do_proofs : bool)
  : ['class_type, 'member_type] ccsl_pre_sig_theory_type
  =
  object (self : 'self)
    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class cl eq_types
      as top_theory

    method get_name = "Isabelle Delimiter theory"

    method get_parameters = []

    initializer top_theory#override_file_name file_name

    method make_body = []

    method kind = match delimiter with
      | IsabelleStartFile
      | IsabelleCloseFile -> delimiter
      | Theory
      | Datatype -> assert false

    method get_isar_imports = isar_imports

      (* Override this method, if proofs should not be written *)
    method do_proofs = do_proofs

end (* ccsl_isabelle_delimiter_theory_class *)

  

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

