(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * predicate and relation lifting
 *
 * $Id: lifting.mli,v 1.6 2002/01/23 16:00:16 tews Exp $
 *
 *)

open Top_variant_types
open Top_classtypes
open Classtypes

(***********************************************************************
 *
 * type param_lifting_type is an assoc list that
 * associates with every type parameter id two 
 * optional expressions. The first is the lifting for 
 * negative occurences, the second for positive occurences.
 * If None, then no lifting is performed for this occurence.
 * 
 * The constraint ensure that the function can access the 
 * variances of components and compute their lifing.
 *)

type ('cl, 'mem) param_lifting_type =
    (('cl, 'mem) top_pre_identifier_record_type
     * (('cl, 'mem) top_pre_expressions option 
	* ('cl, 'mem) top_pre_expressions option )
    ) list
constraint 'cl = <get_name : string; 
                  get_parameters : ('cl,'mem) top_pre_parameter_type list;
		  get_adt_constructors : 'mem list;
		  get_all_sig_actions : 'mem list;
		  get_self_variance : variance_type;
		  get_model_coalgebra : ('cl, 'mem) top_pre_expressions;
                  .. > 
constraint 'mem = <get_domain : ('cl,'mem) top_pre_types; 
		   get_codomain : ('cl,'mem) top_pre_types; 
		   get_name : string;
		   get_accessors : 'mem list;
		   get_recognizer : 'mem;
		   get_sort : 'mem top_member_sort;
		   ..>

(***********************************************************************
 *
 * type self_lifting_type contains the negative (first) and the 
 * positive lifting for self. 
 *)


type ('cl, 'mem) self_lifting_type = 
    ('cl, 'mem) top_pre_expressions option 
    * ('cl, 'mem) top_pre_expressions option
constraint 'cl = <get_name : string; 
                  get_parameters : ('cl,'mem) top_pre_parameter_type list;
		  get_adt_constructors : 'mem list;
		  get_all_sig_actions : 'mem list;
		  get_self_variance : variance_type;
		  get_model_coalgebra : ('cl, 'mem) top_pre_expressions;
                  .. > 
constraint 'mem = <get_domain : ('cl,'mem) top_pre_types; 
		   get_codomain : ('cl,'mem) top_pre_types; 
		   get_name : string;
		   get_accessors : 'mem list;
		   get_recognizer : 'mem;
		   get_sort : 'mem top_member_sort;
		   ..>

(***********************************************************************
 *
 * fullpredlift 
 *
 * This implements full predicate lifting for 
 * full higher order polynomial functors with optimization
 * 
 * for classes, adt's, groundtypes a symbolic Every is produced
 *
 * Arguments
 *     eq_types		equality on types (needed for inlining liftings)
 *     self_subst	substitution for Self
 *     param_lifting	lifting for typeparameters
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 * 
 *)

val fullpredlift : 
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types) -> 
      ('cl, 'mem) param_lifting_type ->
	('cl, 'mem) self_lifting_type ->
	  ('cl, 'mem) Name_space.t -> 
	    ('cl, 'mem) top_pre_types -> 
	      ('cl, 'mem) top_pre_expressions -> 
		('cl, 'mem) top_pre_formulas

(***********************************************************************
 *
 * predlift -- version of fullpredlift without parameter lifting 
 *             and substitution
 *
 * Arguments
 *     eq_types		equality on types (needed for inlining liftings)
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 *)

val predlift : 
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    ('cl, 'mem) self_lifting_type ->
      ('cl, 'mem) Name_space.t -> 
	('cl, 'mem) top_pre_types -> 
	  ('cl, 'mem) top_pre_expressions -> 
	    ('cl, 'mem) top_pre_formulas



(***********************************************************************
 *
 * argument_list_fullpredlift -- predicate lifting for argument lists
 * 
 * Arguments
 *     eq_types		    equality on types (needed for inlining liftings)
 *     self_subst	    substitution for Self
 *     param_lifting	    lifting for typeparameters
 *     self_lifting	    lifting for self
 *     top_name_space	    name space for argument name creation
 *     parameter_list       original parameter list of the 
 *			    ancestor or component (needed to get variances)
 *     argument_list        argument type list that drives the lifting
 *)


val argument_list_fullpredlift : 
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types) -> 
      ('cl, 'mem) param_lifting_type ->
	('cl, 'mem) self_lifting_type ->
	  ('cl, 'mem) Name_space.t -> 
	    ('cl, 'mem) top_pre_parameter_type list -> 
	      ('cl, 'mem) top_pre_argument_type list -> 
		('cl, 'mem) top_pre_expressions list


(*************************************************************************
 *
 * inline_adt_pred_lifting -- predicate lifting of adt as case expression
 * 
 * Arguments
 * 
 *     eq_types		equality on types (for substitution)
 *     self_subst		substitution for carrier
 *     param_lifting	lifting for parameters
 *     self_lifting	lifting for self
 *     top_name_space	the name space
 *     adt			the adt
 *     args		its arguments 
 *     expr 		the expression to lift
 *)


val inline_adt_pred_lifting :
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types) -> 
      ('cl, 'mem) param_lifting_type ->
	('cl, 'mem) self_lifting_type ->
	  ('cl, 'mem) Name_space.t -> 
	    'cl ->
	      ('cl,'mem) top_pre_argument_type list ->
		('cl,'mem) top_pre_expressions ->
		  ('cl,'mem) top_pre_expressions


(******************************
 * Same as predlift, with the following differences:
 * - the lifting wrt. to a one method instead of a type
 * - applications of this method occurs in standard form: m(c)( args )
 * 
 * In class_method_lifting anc_lifting name_space predicate coalgebra 
 * 	      ancestor_list method_list selfvar self_type_argument
 * the arguments are
 *     eq_types		  -> equality on types (needed for inlining liftings)
 *     param_lifting	  -> assoc list of param_lifting_type that gives 
 *			     the liftings form typeparameters
 *     self_lifting	  -> lifting for Self
 *     name_space         -> name space for name creation
 *     coalgebra          -> the coalgebra for getting the methods right
 *     meth		  -> the method
 *     selfvar            -> the variable of Self, to which we apply
 * 				the lifting
 *)


val class_pred_lifting :
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    ('cl, 'mem) param_lifting_type ->
      ('cl, 'mem) self_lifting_type ->
	('cl, 'mem) Name_space.t -> 
	  ('cl, 'mem) top_pre_expressions ->
	    <get_name : string; 
	     last_name : string;
	     original_name : string;
	     get_domain : ('cl, 'mem) top_pre_types; 
	     get_codomain: ('cl, 'mem) top_pre_types;.. 
            > ->
	      string -> 
		('cl, 'mem) top_pre_formulas


(************************************************************************
 * lifting wrt sets of methods:
 *
 *)

					   
val class_method_pred_definition :
  ccsl_name_space_type -> 
    ccsl_output_types ->
      string ->
	ccsl_output_types ->
	  ccsl_member_type ->
	    ccsl_theory_declaration


val class_real_method_pred_lifting :
  ccsl_iface_type ->
    ccsl_argument_type list ->
      ccsl_name_space_type ->
	ccsl_expressions ->				  (* list *)
	  ccsl_formulas ->				  (* predicate *) 
	    ccsl_expressions ->				  (* coalgebra *)
	      string ->					  (* method_id *)
		ccsl_output_types ->
		  string -> 
		    ccsl_formulas

(* legacy code
 * 
 * (************************************************************************
 *  * full constructor wise adt predicate lifting, needed for Isabelle
 *  *
 *  * arguments
 *  *     param_pred_lifting -> lifting for type parameters
 *  *     name_space         -> name space for name creation
 *  *     predicate          -> the predicate, that we lift 
 *  *     constr		  -> constructor
 *  * 
 *  * The result is an implicit case of a (PVS-) cases statement, 
 *  * it consits of a pair (expr, bool_expr), where expr is a (smart) 
 *  * application of the constructor to some (fresh) variables. 
 *  * bool_expr is the lifting for the type of this constructor.
 *  *
 *  * From the result pair either a PVS-cases statement or an 
 *  * Isabelle primrec definition can be generated.
 *  *)
 * 
 * val adt_constr_predicate_lifting : 
 *   (('cl, 'mem) top_pre_types -> ('cl,'mem) top_pre_types) ->
 *     ('cl, 'mem) param_lifting_type ->
 * 	 ('cl, 'mem) Name_space.t -> 
 * 	    ('cl, 'mem) top_pre_formulas ->
 * 	      <get_name : string; get_domain : ('cl, 'mem) top_pre_types; 
 * 			get_codomain: ('cl, 'mem) top_pre_types;.. > ->
 * 		 ('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions
 * 
 *)

(***********************************************************************
 ***********************************************************************
 *
 * RELATION LIFTING
 *
 ***********************************************************************
 ***********************************************************************
 *
 * This implements relation lifting for 
 * full higher order polynomial functors
 * 
 * for classes and adt's a symbolic RelEvery is produced
 * 
 * Arguments
 *     eq_types		    equality on types (needed for inlining liftings)
 *     self1_subst          substitution function for Self for expr1
 *     self2_subst          substitution function for Self for expr2
 *     param_rel_lifting    the lifting for typeparameters
 *     self_lifting	    the lifting for Self/Carrier
 *     top_name_space       name space for argument name creation
 *     typ                  type to lift
 *     expr1, expr2         expressions of type typ, to which the lifting 
 *         		    is applied
 *)

val fullrellift : 
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) param_lifting_type -> 
	  ('cl, 'mem) self_lifting_type -> 
	    ('cl, 'mem) Name_space.t ->
	      ('cl, 'mem) top_pre_types -> 
		('cl, 'mem) top_pre_expressions -> 
		  ('cl, 'mem) top_pre_expressions -> 
		    ('cl, 'mem) top_pre_formulas


(* same as fullrellift, but with param_lifting = [] *)

val rellift : 
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) self_lifting_type -> 
	  ('cl, 'mem) Name_space.t ->
	    ('cl, 'mem) top_pre_types -> 
	      ('cl, 'mem) top_pre_expressions -> 
		('cl, 'mem) top_pre_expressions -> 
		  ('cl, 'mem) top_pre_formulas


(* version of fullrellift for argument lists of components and ancestors
 * this function performs separation of variances, so the resulting list 
 * might be longer than the argument list supplied
 *
 * Arguments
 *     eq_types		    equality on types (needed for inlining liftings)
 *     self1_subst          substitution function for Self for expr1
 *     self2_subst          substitution function for Self for expr2
 *     param_rel_lifting    the lifting for typeparameters
 *     self_lifting	    the lifting for Self/Carrier
 *     top_name_space       name space for argument name creation
 *     parameter_list       original parameter list of the 
 *			    ancestor or component (needed to get variances)
 *     argument_list        argument type list that drives the lifting
 *
 *)


val argument_list_fullrellift :
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) param_lifting_type -> 
	  ('cl, 'mem) self_lifting_type -> 
	    ('cl, 'mem) Name_space.t ->
	      ('cl, 'mem) top_pre_parameter_type list -> 
		('cl, 'mem) top_pre_argument_type list -> 
		  ('cl, 'mem) top_pre_expressions list


(**************************************************************************
 *
 * exported version of inlined adt relation lifting 
 * 
 *     eq_types           -> equality on types
 *     self1_subst        -> substitution function for Self for selfvar1
 *     self2_subst        -> substitution function for Self for selfvar2
 *     param_lifting	  -> lifting for type parameters
 *     self_lifting	  -> lifting for self
 *     top_name_space     -> name space for name creation
 *     adt		  -> the adt to lift
 *     expr1, expr2       -> the expressions
 *)

val inline_adt_rel_lifting:
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl, 'mem) param_lifting_type -> 
	  ('cl, 'mem) self_lifting_type -> 
	    ('cl, 'mem) Name_space.t ->
	      'cl ->
		('cl, 'mem) top_pre_expressions -> 
		  ('cl, 'mem) top_pre_expressions -> 
		    ('cl, 'mem) top_pre_expressions
		


(******************************
 * Same as fullrellift, with the following differences:
 * - the lifting wrt. to one method instead of a type
 * - applications of this method occur in standard form: m(c)( args )
 *
 * parameters
 *     eq_types		    equality on types (needed for inlining liftings)
 *     param_lifting	  -> lifting for type parameters
 *     self_lifting	  -> lifting for self
 *     name_space         -> name space for name creation
 *     self1_subst        -> substitution function for Self for selfvar1
 *     self2_subst        -> substitution function for Self for selfvar2
 *     coalgebra1
 *     coalgebra2         -> the coalgebras for getting the methods right
 *     meth		  -> the method
 *     selfvar1
 *     selfvar2           -> the variables of Self, to which we apply
 *           		     the lifting
 * 
 *)

val class_method_full_rel_lifting :
  (('cl,'mem) top_pre_types -> ('cl,'mem) top_pre_types -> bool) -> 
    (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
      (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types) -> 
	('cl,'mem) param_lifting_type ->
	  ('cl,'mem) self_lifting_type ->
	    ('cl, 'mem) Name_space.t -> 
	      ('cl, 'mem) top_pre_expressions ->
		('cl, 'mem) top_pre_expressions ->
		  'mem ->
		    string -> 
		      string ->
			('cl, 'mem) top_pre_formulas 


(***************************************************************** 
 * relation lifting for adt's, we do the whole adt here and 
 * generate the arguments for reduce
 *
 * Because of this higher order definition of the lifting, 
 * substitutions play a important role, and are quite difficult.
 * To avoid to much code duplication, this fuction computes the 
 * substitution for the adt relation lifting. It is exported, 
 * because the substitutions are needed, to generate precise 
 * importings.
 * 
 * arguments
 * 	   adt		the adt
 * 	   typed_args1  	parameter list 1
 * 	   typed_args2  	parameter list 2
 * result 
 * 	   a pair of substitution functions
 *)

val adt_rel_lifting_substitutions :
  ccsl_iface_type ->
    ccsl_parameter_type list -> 
	 ccsl_parameter_type list -> 
	   ((ccsl_output_types * ccsl_output_types) list *
	    (ccsl_output_types * ccsl_output_types) list)


(***********************************************************************
 * now the real adt lifting
 *
 * arguments
 *   ns 			name space
 *   type_param1		first set of type parameters
 *   type_param2		second set of type parameters
 *   adt			the adt for which the lifting is done
 *   param_rel_assoc	an assoc list for parameter relations,
 * 			   this should associate original parameters 
 * 			   from the adt with relations. Equal is taken
 * 			   for missing associations.
 * 
 * The result is a list of expressions, which can be applied to a 
 * reduce.
 * 
 *)


val adt_rel_lifting :
  ccsl_parameter_type list -> 
    ccsl_parameter_type list -> 
      (ccsl_iface_type, ccsl_member_type) param_lifting_type -> 
	ccsl_name_space_type ->
	   ccsl_iface_type -> 
	     ccsl_expressions list
  


(***********************************************************************
 ***********************************************************************
 *
 *  Here, we generate the constructor declarations
 *
 *  step_constructor_declarations ns l
 *  takes arguments
 *    ns  --> the namespace
 *    cl  --> the class description for wich the adt is generated
 *
 * delivers a list of constructor declarations
 *)


val step_constructor_declarations:
  (ccsl_iface_type, ccsl_member_type)
  Name_space.t ->
    ccsl_iface_type ->
	 (ccsl_iface_type, ccsl_member_type) 
	 Top_variant_types.top_pre_theory_declaration 

val admissible_cases_for_method:
  (ccsl_iface_type, ccsl_member_type)
  Name_space.t ->
    ccsl_member_type ->
	 (ccsl_iface_type, ccsl_member_type) top_pre_expressions ->
	   ((ccsl_iface_type, ccsl_member_type) top_pre_expressions * 
	    (ccsl_iface_type, ccsl_member_type) top_pre_expressions) list

val transition_cases_for_method:
  (ccsl_iface_type, ccsl_member_type)
  Name_space.t ->
    ccsl_member_type ->
	 (ccsl_iface_type, ccsl_member_type) top_pre_expressions ->
	   (ccsl_iface_type, ccsl_member_type) top_pre_expressions ->
	     ((ccsl_iface_type, ccsl_member_type) top_pre_expressions * 
	      (ccsl_iface_type, ccsl_member_type) top_pre_expressions) list



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)

