(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 15.1.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * global control variables of the loop compiler
 *
 * $Id: global.mli,v 1.8 2002/01/23 16:00:22 tews Exp $
 *
 *)

open Top_variant_types

(***********************************************************************
 ***********************************************************************
 *
 * constant string section
 *
 *)


val compiler_version : string


(***********************************************************************
 ***********************************************************************
 *
 * compiler behaviour flag section
 *
 *)


    (* set to Pvs_mode via -pvs in main.ml
     * set to Isa_mode via -isa in main.ml
     * initially set to None
     *)
val output_mode : output_mode ref


    (* if true parse the prelude(s)
     * set in main.ml, via -no-prelude option
     *)
val parse_prelude : bool ref


    (* if true enable the java2 option in the java compiler
     * set in main.ml, via -java2 option
     *)
val java2_mode : bool ref


    (* if true enable the jml option in the java compiler
     * set in main.ml, via -jml option
     *)
val jml_mode : bool ref


    (* 
     * do things, a normal user should never see
     * set via -expert
     *)
val expert_mode : bool ref


    (* generate method bodies without local variables
     * set via -hoare
     *)
val hoare_mode : bool ref


    (* don't use the fixed point definition for method calls,
     * but the direct rewrite instead. This simplifies verification
     * when there are no (mutually) recursive method definitions.
     * set via -no-recursion
     *)
val no_java_recursion : bool ref


    (* if <> "" generate all output in this file
     * set in main.ml, via -o option
     *)
val get_target_file : unit -> string


    (* access functions for target_file
     *)
val set_target_file : string -> unit


    (* if <> "" create output in this directory
     * set in main.ml, via -d option
     *)
val get_target_directory : unit -> string


    (* access functions for target_directory
     *)
val set_target_directory : string -> unit


(***********************************************************************
 ***********************************************************************
 *
 * debug level section
 *
 *)


    (* debug_level_var: controls amount of status information
     * 1 : identical to -v (verbose)
     * 2 : print information about lexer
     * 4 : print information about parser
     * 8 : print information about resolution
     * 16 : print information about inheritance
     * 32 : print information about typecheck
     * 64 : print information about syntax tree
     * 128 : print information about symbol table
     * 512 : print information about unification
     * 1024 : print assertion/creation condition right after parsing
     * set in main.ml, via -D [int] option
     *)
val _VERBOSE :                 int 
val _DEBUG_LEXER : 	       int 
val _DEBUG_PARSER : 	       int 
val _DEBUG_RESOLUTION :        int 
val _DEBUG_INHERITANCE :       int 
val _DEBUG_TYPECHECK : 	       int 
val _DEBUG_DUMP_SYNTAX_TREE :  int 
val _DEBUG_DUMP_SYMBOL_TABLE : int 
val _DEBUG_PRELUDE : 	       int
val _DEBUG_UNIFICATION :       int
val _DEBUG_PARSERCHECK :       int
val _DEBUG_VARIANCE :	       int
val _DEBUG_FEATURES : 	       int

    (* access functions for debug_level_var
     *)

    (* lor more bits into debug_level_var
     * argument of 0 resets debug_level_var
     *)
val add_debug_level : int -> unit


val get_debug_level : unit -> int


val debug_level : int -> bool


    (* if false, do not generate any theories
     * set in main.ml, via option -prf
     *)
val theory_generation : bool ref


    (* if true, unparse the AST
     * set in main.ml, via option -unparse
     *)
val unparse : bool ref


    (* if true, print the output on stdout
     * set in main.ml, via option -c
     *)
val filter_style : bool ref


    (* 
     * generate batch file for theorem prover
     * set via -batch
     *)
val generate_batch : bool ref


    (* 
     * list of proof names to for proof testing 
     *)
val do_proofs : string list ref


    (* 
     * do not generate any theories or proof files,
     * only print proofs in do_proofs on stdout
     * 
     * set via -prooftest
     *)
val proof_test : bool ref


    (* 
     * do theory generation only for selected classes
     * 
     * set via -class
     *)
val selected_classes_only : bool ref 


    (* 
     * list of classes to generate theories for
     * 
     * set via -class
     *)
val selected_classes : string list ref


    (* 
     * don't do theory generation for selected classes
     * 
     * set via -butclass
     *)
val deselected_classes_not : bool ref 

    (*
     * list of classes to exclude for theory generation
     *
     * set via -butclass
     *)
val deselected_classes : string list ref

    (*
     * enable opt_formula and opt_expression
     *
     * unset via -no-opt
     *)
val optimize_expressions : bool ref


val inline_liftings : bool ref

(***********************************************************************
 ***********************************************************************
 *
 * other variables
 *
 *)


    (* input_file holds the working file
     * set in main.ml, extracted from the command line
     *)

    (* access function for input_file
     *)
val set_input_file : string -> unit

val add_input_file : string -> unit

val get_input_file : unit -> string option

val get_input_files : unit -> string list

    (* might fail, if there is no current input file *)
val get_input_filename : unit -> string

    (* add ccsl files, get ccsl files *)
val add_ccsl_input_file : string -> unit
val get_ccsl_input_files : unit -> string list

    (* add java files, get java files *)
val add_java_input_file : string -> unit
val get_java_input_files : unit -> string list

    (* add jml files, get jml files *)
val add_jml_input_file : string -> unit
val get_jml_input_files : unit -> string list


    (* holds the environment variable "CLASSPATH"
     * set in main.ml, extracted from environment
     *)
val get_class_path : unit -> string option

val set_class_path : string -> unit

    (* holds the type used for natural numbers 
     *)
val set_nat_type : string -> unit

val get_nat_type : unit -> string

    (* the path for the fixedpoints library
     *)

val set_pvs_ccsl_lib_location : string -> unit

val get_pvs_ccsl_lib_location : unit -> string


(*******************************************************************
 *
 * java specialities
 *)


    (* For keeping track of the position of the memory cell
     * for holding the static fields of a class
     *)
val next_free_static_position : int ref


(*******************************************************************
 *
 * ccsl specialities
 *)

    (* true when parsing the prelude *)

val parsing_prelude : bool ref

    (* 
     * this flag modifies iface creation in the parser,
     * if set, creates special objects of special subclasses
     * this way it is possible, to change the iface generated for the 
     * list adt in the prelude, such that the internal pvs lists are used
     * in the translation
     *)
val parsing_builtin_prelude : bool ref


    (* generate theories for inductive invariants *)
val ccsl_generate_paths : bool ref


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)
