(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 22.4.99 by Bart
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * top level type definitions
 *
 * $Id: top_variant_types.ml,v 1.23 2002/01/24 14:44:03 tews Exp $
 *
 *)



(*
 * Declaration of the polymorphic variant types,
 * to be used for all translations (Ccsl, Java, ...).
 *
 * Because type variables 'cl or 'mem for classes and
 * members are used, appropriate instantiations of these
 * for Ccsl or Java can be used inside types, formulas, 
 * expressions etc. In general, the convention is used
 * that these polymorphic types have `pre' in their
 * name, and that instantiations, e.g. for the Java
 * translation, have `java' instead of `pre'.
 *
 * The following independent types are introduced.
 *   output_mode
 *   input_mode
 *   token_type
 *   'a container
 *   variance
 *   argument_flag
 *   top_visibility
 *   top_member_sort
 *   top_origin
 *   top_name_space
 *   import
 *   isa_tactic
 *   'a top_pre_shape_type
 *   top_theory_type
 *   
 * For PVS proofs we have
 *   lisp_expr
 *   pvs_proof_tree
 *   pvs_proof
 * 
 * Additionally, the following types are defined in a big
 * mutual recursion.
 *   ('cl, 'mem) top_pre_types
 *   ('cl, 'mem) top_pre_expressions
 *   ('cl, 'mem) top_pre_formulas
 *				(* for ccsl assertions or java annotations *)
 *   ('cl, 'mem) top_pre_assertion 
 *   ('cl, 'mem) top_pre_assertion_record
 *   ('cl, 'mem) top_pre_symbol_type 
 *   ('cl, 'mem) definition_parameter
 *   ('cl, 'mem) constructor
 *   ('cl, 'mem) top_pre_theory_declaration
 *   ('cl, 'mem) top_pre_proof_file_entry
 *   ('cl, 'mem) top_pre_parameter_type
 *   ('cl, 'mem) top_pre_argument_type
 *   ('cl, 'mem) top_pre_identifier_record_type
 *
 *
 * Finally, the next few types incoporate these types.
 *   ('cl, 'mem) top_pre_renaming
 *   ('cl, 'mem) top_pre_ancestor_type
 *   ('cl, 'mem) top_pre_component_type
 *   ('cl, 'mem) top_pre_ast
 *   ('cl, 'mem) top_pre_symbol_table_type
 *   ('cl, 'mem) top_pre_scope_type
 *
 *)



(***********************************************************************
 ***********************************************************************
 *
 * Independent types 
 *
 *)


type output_mode =
  | Pvs_mode
  | Isa_mode


type input_mode =
  | Ccsl_mode


type location_type =
  {
    file_name : string;
    start_line : int;
    start_char : int;
    end_line : int;
    end_char : int
  }


type token_type = 
  {
    token_name : string;
    loc : location_type option
  }


type 'a container =
  | Unresolved of token_type
  | Resolved of 'a


type variance_type = 
  | Pos  
  | Neg  
  | Mixed  
  | Unused
  | Unset
  | Pair of int * int

type functor_type =
  | UnknownFunctor
  | StrictlyConstantFunctor
  | ConstantFunctor
  | StrictlyPolynomialFunctor
  | PolynomialFunctor
  | ExtendedCartesianFunctor
  | ExtendedPolynomialFunctor
  | HigherOrderFunctor


(* features are flags that get set for some classes and adt's,
 * with them I track some properties of classes and adt's to 
 * determine other properties. For instance a class A has morphisms 
 * if all components define the map combinator. If A has a binary 
 * method then the components need the full map combinature, which has 
 * separated positive and negative variances.
 *)

type feature_type = 
  | BuildinFeature		    (* adt is builtin in target 
				     * theorem prover
				     *)
  | NeedsMapFeature                 (* for Adt's with negative parameter 
				     * map and every must be generated 
				     * for PVS
				     *)
  | FinalSemanticsFeature	    (* final semantics chosen
				     *)
  | MixedSelfInstFeature	    (* self is put into a mixed variance 
				       parameter position
				     *)
  | ComponentNoRelLiftFeature	    (* class has a component instanciated 
				       with Self that has 
				       no greatest bisim
				    *)
  | ComponentNoFullRelLiftFeature   (* class has a component instanciated 
				       with a type parameter or with Self 
				       that has no greatest bisim
				    *)
  | ComponentNoMapFeature	    (* class has a component incstanciated 
				       with Self that has no map
				    *)
  | ComponentNoFullMapFeature	    (* class has a component instanciated 
				       with a parameter or self that 
				       has not map
				    *)
  | ClassComponentFeature	    (* adt with nonconstant class component *)
  | CarrierClassComponentFeature    (* adt with class component 
				       instanciated with carrier
				    *)
  | HasBisimulationFeature	    (* class has no definition of
				       bisimulation
				    *)
  | HasFullRelLiftingFeature	    (* class has full relation lifting
				     *)
  | HasGreatestBisimFeature	    (* Class with greatest bisimulation
				     *)
  | HasRelLiftFeature		    (* iface defines relation lifting *)
  | HasMorphismFeature		    (* class has no coalgebra morphism 
				       definition *)
  | HasMapFeature		    (* Class has map 
				     *)



(* argument flag is used together with type parameters, 
 * see constructors TypeConstant IFace Term QualifiedTerm
 * 
 * Always -> Arguments are alawys printed
 * Isabelle_only -> pvs_pretty ignores the arguments
 * Never -> arguments are ignored
 * 
 * You should use Isabelle_only for arguments of Types, which are declard 
 * in the same theory. Otherwise use Always. Never is only there for 
 * completeness. If you don't want arguments, then supply the empty list, 
 * together with the flag of your choice
 *)
type argument_flag = 
 | Always
 | Isabelle_only
 | Never

type top_visibility = 
  | NoVisibility
  | Public 
  | Private

      
type 'mem top_member_sort = 
  | Proper_Attribute of 'mem option   (* declared attribute with set_method *)
  | Update_Method			(* update method for an attribute *)
  | Defined_Method			(* definitional extention *)
  | Normal_Method			(* method before variance checking *)
  | Var_Constructor			(* class constructor with args *)
  | Const_Constructor			(* class constructor without args *)
  | Adt_Var_Constructor			(* adt constructor with args *)
  | Adt_Const_Constructor		(* adt constructor without args *)
  | Adt_Accessor			(* an accessor *)
  | Adt_Recognizer			(* an recognizer *)
  | Adt_Reduce
  | Class_Coreduce
  | GroundTerm				(* member of a ground signature *)
  | InfixGroundTerm			(* infix GroundTerm *)

(* this is the type of name spaces, usually the same identifier can denote
 * different identities, ie. a class and a variable, the syntax of the 
 * language ditinguishes both (because a class name must not occur, 
 * where a variable name can occur and vice versa). Internally these 
 * syntactically distiguished sets of identifiers are represented as 
 * name spaces.
 * 
 * For CCSL it is convenient to store instanciated classes and adt 
 * in the symboltable. They get their own name_space. No identifier of
 * the source language will be in this name_space
 *)
type top_name_space =
  | CCSL_ID
  | InstAdtOrClass


(* an instance of top_pre_iface can represent a class, an adt or
 * a ground signature. Because down casts are not available everything 
 * is packed in one class and distinguished with a flag of top_spec_kind. 
 * Spec_Spec descibes the non specified state that can occure shortly
 * during parsing.
 *)

type top_spec_kind = 
  | Spec_Spec				(* indefinate state *)
  | Spec_class
  | Spec_adt
  | Spec_sig



type import = string


(* For simplicity, isa_tactic now is a string. To be changed in the future.
 * FIXME later
 *)
type isa_tactic = string


type top_theory_type =
  | Theory 
  | Datatype 



(***********************************************************************
 ***********************************************************************
 * 
 * PVS Proofs and List expressions
 *
 *)

type lisp_expr       =   LAtom of string
		       | LList of lisp_expr list

type pvs_proof_tree  =   PTree of lisp_expr list * pvs_proof_tree list

type pvs_proof       =   Named_proof of string * pvs_proof_tree
		       | Anon_proof of pvs_proof_tree

(***********************************************************************
 ***********************************************************************
 *
 * mutually recursive variant types, parametrized by 
 * iface ('cl) and member ('mem) classes. 
 *
 *)


type ('cl, 'mem) top_pre_types = 
  | Groundtype of 
      ('cl, 'mem) top_pre_identifier_record_type * 
      ('cl, 'mem) top_pre_argument_type list
  | TypeConstant of 
      string * argument_flag * ('cl, 'mem) top_pre_argument_type list
  | BoundTypeVariable of ('cl, 'mem) top_pre_identifier_record_type
  | FreeTypeVariable of Type_variable.t
  | Bool
  | Self
  | Carrier
  | Function of (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types)
      (* Sidecondition in SmartFunction: the list must be non-empty *)
  | SmartFunction of 
      (('cl, 'mem) top_pre_types list * ('cl, 'mem) top_pre_types)
  | Product of ('cl, 'mem) top_pre_types list
  | Record of (string * ('cl, 'mem) top_pre_types) list
  | Class of 'cl * ('cl, 'mem) top_pre_argument_type list
  | Adt of 'cl * argument_flag * ('cl, 'mem) top_pre_argument_type list

      (* the Functor of the class, the argument list is left empty, 
       * if there is no instanciation necessary
       *)
  | IFace of 'cl * argument_flag * ('cl, 'mem) top_pre_argument_type list
  | Array of 'cl * ('cl, 'mem) top_pre_types * int
  | Predtype of ('cl, 'mem) top_pre_formulas
			   
and ('cl, 'mem) top_pre_inst_iface_type =
  | NoIface				(* no info given *)
  | CurrentIface			(* the current class, ie. Self *)
					(* some other class/adt/sig *)
  | InstIface of 'cl * ('cl, 'mem) top_pre_argument_type list

and ('cl, 'mem) top_pre_basic_expressions =
      (* 
       * TermVar is used for variables
       *)
  | TermVar of ('cl, 'mem) top_pre_identifier_record_type container

      (* Member( hosting_class_or_adt, member_or_token )
       *
       * hosting_class_or_adt is filled in the parser if the 
       *   constructor/method/groundsymbol 
       *   appears in an qualified expression id.c
       * member_or_token is not resolved in the parser
       *
       * before printing this is preprocessed in pre_printing 
       * to convert it into a record selection with the algebra/coalgebra
       *)
  | Member of 
      ('cl, 'mem) top_pre_inst_iface_type * 'mem container 

and ('cl, 'mem) top_pre_expressions = 
  | ExprLoc of ('cl,'mem) top_pre_expressions * location_type
  | BasicExpr of ('cl,'mem) top_pre_basic_expressions
  | Term of string * argument_flag * ('cl, 'mem) top_pre_argument_type list
      (* For type annotations, which get printed *)
  | TypedTerm of ('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_types

    (* TypeAnnotation keeps typing information during 
     * resolution and typecheck pass. 
     * 
     * Before the resolution pass no TypeAnnotation is allowed.
     * The resolution pass fills in the following TypeAnnotation's :
     * 
     * 	  TypeAnnotation( BasicExpr(Member _), type_of_member)
     * 	  CCSL_Case(TypeAnnotation(ex, adt_type_of_ex), variants )
     *    TypeAnnotation( InfixApplication( exl, instiface, infixop, exr ),
     *                    type_of_infixop)
     * 
     * Type derivation keeps these and adds 
     * 
     *    MethodSelection( TypeAnnotation(o, type_of_o), user_typeopt, m )
     *    Application(TypeAnnotation(fun, type_of_fun), args)
     *    Equal(TypeAnnotation(ex1, type_of_eq), ex2)
     *    Obseq(name-type-option, TypeAnnotation(ex1, type_of_eq), ex2)
     *
     * Type checking adds 
     *
     *    TypeAnnotation(Projection(i,n), type_of_proj)
     *
     * and inserts the right n.
     * the latter enables the prepretty printer to do eta expansion
     * for lonely projections
     *)
  | TypeAnnotation of ('cl, 'mem) top_pre_expressions 
      * ('cl, 'mem) top_pre_types
  | QualifiedTerm of 
      string * argument_flag *
      (('cl, 'mem) top_pre_argument_type) list * 
      string

      (* The next expression is needed in HOL for CCSL,
       * there we have expressions x.m( .... ) where 
       * x is an expression, which yields an object of some class 
       * as value, m is a method or attribute of that class 
       * 
       * Internally represented as 
       *     MethodSelection( x, Some((cl, args)), method )
       *
       * The optional field is filled in the parser, if it is a qualified 
       * method selection
       * 
       * Before printing this is preprocessed and converted into 
       * an application/record selection
       * 
       *)
  | MethodSelection of
      ('cl, 'mem) top_pre_expressions * 
      ('cl, 'mem) top_pre_inst_iface_type *
      'mem container

  | Tuple of ('cl, 'mem) top_pre_expressions list

      (* Projection(i,n) is the i-th projection for 
       * an n-ary product type. The n is needed to print 
       * projections for isabelle.
       * If n = -1, then the arity is unknown (valid in 
       * assertions before typechecking)
       * otherwise 1 < i <= n must hold
       *)
  | Projection of int * int
  | RecordTuple of (string * ('cl, 'mem) top_pre_expressions) list
  | RecordSelection of string * ('cl, 'mem) top_pre_expressions
  | RecordUpdate of 
      ('cl, 'mem) top_pre_expressions * 
      (string * ('cl, 'mem) top_pre_expressions) list
  | List of ('cl, 'mem) top_pre_expressions list
  | Abstraction of 
      (string * ('cl, 'mem) top_pre_types) list * 
      ('cl, 'mem) top_pre_expressions
  | SmartAbstraction of 
      (string * ('cl, 'mem) top_pre_types) list * 
      ('cl, 'mem) top_pre_expressions
  | Application of
      ('cl, 'mem) top_pre_expressions *
      ('cl, 'mem) top_pre_expressions
	(* Sidecondition in SmartApplication: the list must be non-empty *)
  | SmartApplication of
      ('cl, 'mem) top_pre_expressions *
      ('cl, 'mem) top_pre_expressions list
  | InfixApplication of 
      ('cl, 'mem) top_pre_expressions * 
      ('cl, 'mem) top_pre_inst_iface_type *
      'mem container *
      ('cl, 'mem) top_pre_expressions 
  | FunUpdate of
      ('cl, 'mem) top_pre_expressions * 
      (('cl, 'mem) top_pre_expressions * 
          ('cl, 'mem) top_pre_expressions) list
  | Let of
      (('cl, 'mem) top_pre_identifier_record_type * 
         ('cl, 'mem) top_pre_types option *
         ('cl, 'mem) top_pre_expressions) list * 
      ('cl, 'mem) top_pre_expressions
	  (* Sidecondition: empty 'a list raises 
           * exception *)
  | If of 
      (('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_expressions) list 
      * ('cl, 'mem) top_pre_expressions
  | Case of
      ('cl, 'mem) top_pre_expressions * 
      (('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions) list

	(* For the input language of CCSL we need a case with more 
	 * information, instead of generalizing the previous Case, 
	 * we decided to define a new one. The intended semantics
	 * is equivalent to the original case.
	 *)
  | CCSL_Case of
      ('cl, 'mem) top_pre_expressions *
      ('mem container * ('cl, 'mem) top_pre_identifier_record_type list * 
       ('cl, 'mem) top_pre_expressions) list

(* 	     (* One usually has reduce in an application
 * 	      * but we need it as a function symbol (without arguments)
 * 	      * here, since we might have it without application in our Logic
 * 	      *)
 *   | Reduce of
 * 	 'cl * argument_flag * ('cl, 'mem) top_pre_argument_type list 
 *)

      (* The Modal Operators (Box, Diamond)
       * - they are only needed while parsing the Logic
       * - they carry 
       *  + the class they belong to (with arguments)
       *  + the predicate they are talking about ('cl,'mem) top_pre_expressions
       *  + the list of considered methods
       * 
       * VALID_CLASS Sidecondition applies to first argument of Box and Diamond
       *)
  | Box of
      ('cl, 'mem) top_pre_types *
      ('cl, 'mem) top_pre_expressions * token_type list
  | Diamond of
      ('cl, 'mem) top_pre_types *
      ('cl, 'mem) top_pre_expressions * token_type list

      (* symbolic predicate lifting for adt's, classes, TypeConstants 
       *
       * in Every(typ, pred_list)
       *   typ is one of Adt, Class, Groundtype
       *   pred_list is a list of predicates, in harmony with 
       * 	    the type parameters of typ
       *   Every makes only sense, if it is wrapped in an application:
       *      Application(Expression(Every(typ, pred_list)), expr)
       *)
  | Every of ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_expressions list 

      (* symbolic relation lifting for adt's, classes, TypeConstants 
       *
       * in RelEvery(typ, rel_list)
       *   typ is one of Adt, Class, Groundtype
       *   rel_list is a list of relations, in harmony with 
       * 	    the type parameters of typ
       *   RelEvery makes only sense, if it is wrapped in an 
       *   ``smart'' application like:
       *      SmartApplication(Expression(RelEvery(typ, rel_list)), 
       *                               [expr1; expr2])
       *)
  | RelEvery of
      ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_expressions list 
  | Map of
      ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_expressions list
  | Expression of ('cl, 'mem) top_pre_formulas
	  (* Sidecondition: empty 'a list raises 
           * exception *)
  | Comment_str of string * ('cl, 'mem) top_pre_expressions
	       	(* the first expression gets commented *)
  | Comment_expr of 
      ('cl, 'mem) top_pre_expressions * 
      ('cl, 'mem) top_pre_types * 
      ('cl, 'mem) top_pre_expressions
		(* PVS specifics *)
  | Comprehension of 
      string * ('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_formulas


and ('cl, 'mem) top_pre_formulas =
  | FormLoc of ('cl,'mem) top_pre_formulas * location_type
  | True
  | False
  | Not of ('cl, 'mem) top_pre_formulas
  | And of ('cl, 'mem) top_pre_formulas list
  | Or of ('cl, 'mem) top_pre_formulas list
  | Implies of 
      ('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_formulas
  | Iff of
      ('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_formulas
  | Equal of
      ('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions
(*   | LessOrEqual of
 * 	 ('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions
 *)
  | Forall of
      (string * ('cl, 'mem) top_pre_types) list * 
      ('cl, 'mem) top_pre_formulas
  | Exists of
      (string * ('cl, 'mem) top_pre_types) list * 
      ('cl, 'mem) top_pre_formulas
  | ConstantPredicate of string
  | Formula of ('cl, 'mem) top_pre_expressions
    (* for Isabelle *)
  | MetaImplies of
      ('cl, 'mem) top_pre_formulas * ('cl, 'mem) top_pre_formulas
  | MetaForall of
      (string * ('cl, 'mem) top_pre_types) list * 
      ('cl, 'mem) top_pre_formulas
      (* bisimilarity for a class: the first argument must be a class *)
  | Bisim of
      ('cl,'mem) top_pre_types * ('cl, 'mem) top_pre_expressions * 
      ('cl, 'mem) top_pre_expressions
  | Obseq of
      (string * ('cl, 'mem) top_pre_types) option *
      ('cl, 'mem) top_pre_expressions * 
      ('cl, 'mem) top_pre_expressions


  (* the formula of an assertion can be a string or an abstract formula *)
and ('cl, 'mem) top_pre_assertion =
  | Symbolic of ('cl, 'mem) top_pre_formulas
  | Pvs_String of string
  | Isa_String of string

  (* assertion records are used for both:
   * ccsl assertions and creation conditions
   *)
and ('cl, 'mem) top_pre_assertion_record =
    { free_variables : ('cl, 'mem) top_pre_identifier_record_type list;
      self_variable : ('cl, 'mem) top_pre_identifier_record_type option;
      assertion_name : token_type;
      is_generated : bool;
      mutable assertion_formula : ('cl, 'mem) top_pre_assertion
    }


and ('cl, 'mem) top_pre_definition_record =
    { defined_method : 'mem;
      mutable variables : ('cl, 'mem) top_pre_identifier_record_type list;
      mutable definition : ('cl, 'mem) top_pre_assertion
    }


and ('cl, 'mem) top_pre_symbol_type =
					(* a ground type *)
  | CCSL_GroundTypeSymb of ('cl, 'mem) top_pre_identifier_record_type
					(* a ccsl type abbreviation *)
  | CCSL_TypeAbbrevSymb of ('cl, 'mem) top_pre_identifier_record_type
					(* a ccsl type variable *)
  | TypevarSymb of ('cl, 'mem) top_pre_identifier_record_type
					(* a ccsl variable *)
  | VarSymb of ('cl, 'mem) top_pre_identifier_record_type 
					(* a ccsl class *)
  | ClassSymb of 'cl 
					(* a instanciated ccsl class *)
  | InstClassSymb of 'cl
					(* a ccsl adt *)
  | AdtSymb of 'cl 
					(* an instanciated ccsl adt *)
  | InstAdtSymb of 'cl
					(* a ground signature *)
  | SigSymb of 'cl
					(* a method or constructor *)
  | MemberSymb of 'mem


(* 
 * Definition_parameter is used in theory declatations Defn and 
 * Defnuninterpret. Isabelle pretty printer prints all parameters
 * with full type. Pvs however only the undeclared ones.
 *)
and ('cl, 'mem) definition_parameter = 
 | Declared of string * ('cl, 'mem) top_pre_types
 | Undeclared of string * ('cl, 'mem) top_pre_types

and ('cl, 'mem) constructor = 
    string *				(* constructor name *)
    (string * ('cl, 'mem) top_pre_types) list * (* accessor * type list *)
    string				(* recognizer *)

and ('cl, 'mem) top_pre_theory_declaration =
  | Lemma of string * ('cl, 'mem) top_pre_formulas
  | Axiom of string * ('cl, 'mem) top_pre_formulas
  | Vardecl of string * ('cl, 'mem) top_pre_types
  | Typevardecl of string 
      (* Typedecl is a bit misleading, 
       * this should only be seen as type ABBREVIATION 
       *)
  | Typedecl of
      ('cl, 'mem) top_pre_parameter_type list * string * 
      ('cl, 'mem) top_pre_types
      (* the first argument is only used for isabelle *)
  | Enumdecl of string * string list
  | Defn of 
      string * 
      (('cl, 'mem) definition_parameter list list) * 
      ('cl, 'mem) top_pre_types * 
      ('cl, 'mem) top_pre_expressions
  | Defrecfn of
      string * 
      (('cl, 'mem) definition_parameter list list) * 
      ('cl, 'mem) top_pre_types * 
      ('cl, 'mem) top_pre_expressions *      
      ('cl, 'mem) top_pre_expressions      
  | Defnuninterpret of
      string * 
      (('cl, 'mem) definition_parameter list list) * 
      ('cl, 'mem) top_pre_types

	(* this is a compromize between Isabelle and PVS, 
	 * For PVS only the constructor list is used and the rest
	 * is ignored.
	 * For Isabelle the accessor and the recognizer in the 
	 * constructor structure are ignored
	 *)
  | Datatypedecl of string * ('cl, 'mem) top_pre_parameter_type list * 
      ('cl, 'mem) constructor list

        (* Isabelle primitive recursion equations 
	 * gets name, type, and a list of equations, given here as pairs
	 *)
  | Primrecdefn of
      string * ('cl, 'mem) top_pre_types * 
      (('cl, 'mem) top_pre_expressions * 
       ('cl, 'mem) top_pre_expressions) list
  | Comment of string
  | Comment_decl_expr of ('cl, 'mem) top_pre_expressions
  | Comment_decl of ('cl, 'mem) top_pre_theory_declaration
  | Import of ('cl,'mem) top_pre_importing_type list
  | LibImport of string * string * ('cl, 'mem) top_pre_argument_type list
  | Library of string * string 
  | Conversion of string * ('cl, 'mem) top_pre_types
  | Proved of ('cl, 'mem) top_pre_theory_declaration * pvs_proof

and ('cl, 'mem) top_pre_proof_file_entry =
  | Proof of
      string *                 (* name of the lemma *)
      ('cl, 'mem) top_pre_formulas *   (* goal to prove *)
      isa_tactic list          (* the proof script *)
  | Rewrite_set of
      string *                 (* name of the definition *)
      string list              (* set of rewrite rules *)

and ('cl, 'mem) top_pre_parameter_type = 
  | TypeParameter of ('cl, 'mem) top_pre_identifier_record_type


(* arguments could be defined after the pre_iface's but I think
 * I rather leave parameters, arguments and renaming and ancestors 
 * together.
 *)
and ('cl, 'mem) top_pre_argument_type =
  | TypeArgument of (('cl, 'mem) top_pre_types)


(* top_origin is used to distinguish in the kind of variable 
 * contained in some identifier_record
 *)
and ('cl, 'mem) top_pre_origin_type =
  | CCSL_TypeParameter			(* id_type irrelevant *)
  | CCSL_Var				(* id_type, id_parameters irrelevant *)
  | CCSL_TypeAbbrev			(* id_parameters irrelevant *)
  | CCSL_GroundType of 'cl
  | CCSL_Output				(* everything irrelvant 
					   except the name *)

and ('cl, 'mem) top_pre_identifier_record_type =
  { 
    id_token  : token_type;
    mutable id_type : ('cl, 'mem) top_pre_types;
    mutable id_parameters : ('cl, 'mem) top_pre_parameter_type list;
    id_origin : ('cl, 'mem) top_pre_origin_type;
    mutable id_variance : variance_type;
  }


(* type of importings
 *)

and ('cl,'mem) top_pre_importing_type =
    string * ('cl,'mem) top_pre_argument_type list



(***********************************************************************
 ***********************************************************************
 *
 * Some further types, making use of the mutually recursive types.
 *
 *)


(* this is one of the few real type abbreviations,
 * stands for old_name, type annotation, new name
 *)
type ('cl, 'mem) top_pre_renaming = 
    token_type * ('cl, 'mem) top_pre_types option * token_type


(* the type of the ancestors
 * in principle this should be refined in ccsl and java to 
 * appropriate sybtypes, but this doesn't work at the moment :-(
 *
 * 'types, expressions and members are here, because the renaming 
 * depend on them
 *)
type ('cl, 'mem) top_pre_ancestor_type =
  | Unresolved_ancestor of token_type list
  | Resolved_ancestor of 'cl
  | Unresolved_renaming of
      'cl *     (* the original class *)
      ('cl, 'mem) top_pre_argument_type list *
      ('cl, 'mem) top_pre_renaming list
  | Resolved_renaming of
      'cl *     (* the original class *)
      ('cl, 'mem) top_pre_argument_type list * 
      'mem list *
      'cl       (* the instanciated class *)

(* 
 * the type of components,
 * we do not distinguish between adt and class components as before
 * you can probe the class instead
 * The first field is the variance at which the component occurs.
 * This is needed in morphisms and map theories.
 *)

type ('cl, 'mem) top_pre_component_type = 
    variance_type * 'cl * ('cl, 'mem) top_pre_argument_type list
constraint 'cl = < get_parameters : ('cl,'mem) top_pre_parameter_type list;
                   ..>

type ('cl, 'mem) top_pre_ast = 
  | CCSL_TypeAbbrev_dec of ('cl, 'mem) top_pre_identifier_record_type
  | CCSL_class_dec of 'cl
  | CCSL_adt_dec of 'cl
  | CCSL_sig_dec of 'cl

(* type information for overloaded identifiers 
 * 'a will be instanciated with top_types, 
 * 'b with top_pre_argument_type
 * the instanciation is used for name_space InstClasses, 
 * to store instanciated classes and adts in the symboltable
 *)
type ('a, 'b) top_pre_shape_type = 
  | No_overloading
  | Instanciation of 'b


(* symbol table *)
type ('cl, 'mem) top_pre_symbol_table_type =
    (top_name_space, 
     (('cl, 'mem) top_pre_types, 
      ('cl, 'mem) top_pre_argument_type list) top_pre_shape_type, 
     ('cl, 'mem) top_pre_symbol_type) Table.global

type ('cl, 'mem) top_pre_scope_type =
    (top_name_space, 
     (('cl, 'mem) top_pre_types, 
      ('cl, 'mem) top_pre_argument_type list) top_pre_shape_type, 
     ('cl, 'mem) top_pre_symbol_type) Table.local

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)
