(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 24.8.01 by Hendrik
 *
 * Time-stamp: <Sunday 19 May 02 16:22:39 tews@ithif56.inf.tu-dresden.de>
 *
 * pretty printer for CCSL
 *
 * $Id: ccsl_pretty.ml,v 1.13 2002/05/22 13:42:38 tews Exp $
 *
 *)




open Util
open Formatter
open Top_variant_types
open Top_variant_types_util
open Classtypes
open Pretty_util
;;


(***********************************************************************
 ***********************************************************************
 *
 * printing functions 
 *
 *)

let ccsl_pp_variance = function
  | Pair(n,p) -> 
      begin
	print_string "(";
	if n = -1 
	then print_string "?"
	else print_int n;
	print_string ", ";
	if p = -1 
	then print_string "?"
	else print_int p;
	print_string ")"
      end
  | Pos -> print_string "Pos"
  | Neg -> print_string "Neg"
  | Unused -> print_string "Unused"
  | Unset -> print_string "Unset"
  | Mixed -> print_string "Mixed"

let rec ccsl_pp_type 
  = function
    | Groundtype(id,args) ->
	begin
	  print_string id.id_token.token_name;
	  if (args <> [])
	  then
	    begin
	      open_box 0;
	      print_string "[";
	      ccsl_pp_arg_list args;
	      print_string "]";
	      close_box();
	    end
	end
    | BoundTypeVariable id -> 
	print_string id.id_token.token_name
    | Self -> 
	print_string "Self"
    | Bool ->
	print_string "Bool"
    | Carrier ->
	print_string "Carrier"
    | Function(dom, codom) ->
	begin
	  open_box 0;
	  print_string "[";
	  open_box 0;
	  ccsl_pp_type dom; 
	  print_space(); 
	  print_string "-> ";
	  ccsl_pp_type codom;
	  close_box(); 
	  print_string "]";
	  close_box()
	end	
    | Product typelist ->
	begin
	  open_box 0;
	  print_string "["; 
	  pp_list ccsl_pp_type "" "" "" "," typelist;
	  print_string "]";
	  close_box()
	end
    | Class(cl, arg_list) ->
	begin
	  open_box 0;	    
      	  print_string cl#get_name;
	  if 
	    arg_list <> [] 
	  then
	    begin
	      print_string "["; 
	      ccsl_pp_arg_list arg_list; 
	      print_string "]"
	    end;
	  close_box()
	end
    | Adt(adt, flag, arg_list) ->
  	begin
	  open_box 0;
	  print_string adt#get_name;
	  if 
	    (arg_list <> [])
	  then
	    begin
              print_string "[";
              ccsl_pp_arg_list arg_list;
              print_string "]"
	    end;
	  close_box()
  	end
    | FreeTypeVariable tv -> 
	print_string (Type_variable.string_of_tv tv)
					(* not allowed *)
    | TypeConstant _
    | SmartFunction _
    | Record _ 
    | IFace _
    | Array _ 
    | Predtype _ -> 
	assert(false)

and ccsl_pp_inst_iface = function
  | NoIface -> ()
  | CurrentIface -> 
      print_string "Self::"
  | InstIface(iface, args, locopt) -> 
      begin
      	print_string iface#get_name;
	if 
	  args <> [] 
	then
	  begin
	    print_string "["; 
	    ccsl_pp_arg_list args; 
	    print_string "]"
	  end;
	print_string "::"
      end


and ccsl_pp_basic_expression = function
    | TermVar id ->
	let name = match id with
	  | Unresolved t -> t.token_name
	  | Resolved id_rec -> id_rec.id_token.token_name
	in
	print_string name
    | Member(instiface, memc)  ->
	begin
	  open_box 2;			(* 1 *)
	  ccsl_pp_inst_iface instiface;
	  (match memc with
	     | Unresolved t -> print_string t.token_name
	     | Resolved m -> print_string m#get_name);
	  close_box()
	end	      
	


and ccsl_pp_expression (* : top_expressions -> unit *)
  = function
    | ExprLoc(ex, l) -> ccsl_pp_expression ex
    | BasicExpr bexp -> 
	ccsl_pp_basic_expression bexp
    | TypedTerm(Term(t,_,_), typ) ->
	begin
	  open_box 0;
	  print_string "(";
	  print_space();
	  print_string t;
	  print_string " :";
	  print_space();
	  ccsl_pp_type typ;
	  print_string ")";
	  close_box();
	end
    | TypedTerm(ex, typ) ->
	begin
	  open_box 0;
	  print_string "(";
	  ccsl_pp_expression ex;
	  print_string " :";
	  print_space();
	  ccsl_pp_type typ;
	  print_string ")";
	  close_box();
	end
    | TypeAnnotation(ex, _) ->
	  ccsl_pp_expression ex;
    | MethodSelection(ex,instiface,memc) ->
	begin
	  open_box 0;
	  print_string "(";
	  print_string "(";
	  ccsl_pp_expression ex;
	  print_string ").";
	  ccsl_pp_inst_iface instiface;
	  (match memc with
	     | Unresolved t -> print_string t.token_name
	     | Resolved m -> print_string m#get_name);
	  print_string ")";
	  close_box()
	end
	  
    | Tuple expr_list ->
	(match expr_list with
	   | [] -> assert(false)
	   | l ->
	       begin
		 open_box 2;
		 print_string "(";
		 (pp_list ccsl_pp_expression "" "" "" "," l);
		 print_string ")";
		 close_box()
	       end)
    | Abstraction (str_typ_list, expr) ->
	(match str_typ_list with
	   | [] -> assert(false)
	   | l ->
	       begin
		 open_box 2;				(* 1 *)
		 print_string "(";
		 print_string "LAMBDA (";
		 pp_list ccsl_pp_var "" "" "" "," str_typ_list;
		 print_string ") .";
		 print_break 1 2;
		 ccsl_pp_expression expr;
		 print_string ")";
		 close_box ()
	       end)
    | Projection (ith,n) ->
	begin
 	  print_string ("PROJ_" ^ (string_of_int ith) );
    	end
	     
    | Application (func, arg) ->
	(match arg with
	   | Tuple _ ->
	       begin
		 open_box 0;
		 print_string "(";
		 ccsl_pp_expression func; 
		 print_break 0 2;
		 ccsl_pp_expression arg;
		 print_string ")";
		 close_box ()
	       end
	   | _ ->
	       begin
		 open_box 0;
		 print_string "(";
		 ccsl_pp_expression func; 
		 print_break 0 2; 
		 print_string "(";
		 open_box 0;
		 ccsl_pp_expression arg; 
		 print_string ")";
		 print_string ")";
		 close_box();
		 close_box()
	       end)
    | InfixApplication (expr1,instiface,memc,expr2) ->
	begin
	  open_box 0;			(* 1 *)
	  print_string "(";
	  ccsl_pp_expression expr1; 
	  print_break 1 2;
	  ccsl_pp_inst_iface instiface;
	  (match memc with
	     | Unresolved t -> print_string t.token_name
	     | Resolved m -> print_string m#get_name);
	  ccsl_pp_expression expr2; 
	  print_string ")";
	  close_box();			(* 0 *)
	end
    | Box(typ,pred,tok_list) ->
	begin
	  open_box 0;					(* 1 *)
	  print_string "(Always";
	  print_space();
	  open_box 2;					(* 2 *)
	  ccsl_pp_expression pred;
	  close_box();					(* 1 *)
	  print_space();
	  print_string "For ";
	  (match typ with
	     | Self -> ()
	     | t -> ccsl_pp_type t);
	  print_string "{";
	  open_box 2;					(* 2 *)
	  pp_list (fun t -> print_string t.token_name)
	    "" "" "" ", " tok_list;
	  print_string "})";
	  close_box();					(* 1 *)
	  close_box();					(* 0 *)
	end
    | Diamond(typ,pred,tok_list) ->
	begin
	  open_box 0;					(* 1 *)
	  print_string "(Eventually";
	  print_space();
	  open_box 2;					(* 2 *)
	  ccsl_pp_expression pred;
	  close_box();					(* 1 *)
	  print_space();
	  print_string "For ";
	  (match typ with
	     | Self -> ()
	     | t -> ccsl_pp_type t);
	  print_string "{";
	  open_box 2;					(* 2 *)
	  pp_list (fun t -> print_string t.token_name)
	    "" "" "" ", " tok_list;
	  print_string "})";
	  close_box();					(* 1 *)
	  close_box();					(* 0 *)
	end
    | FunUpdate (func, expr_expr_list) ->
	begin
	  open_box 0;					(* 1 *)
	  print_string "(";
	  ccsl_pp_expression func; 
	  print_break 1 2;
	  print_string "WITH ["; 
	  print_break 1 2;
	  pp_list (fun (place, expr) ->
		     begin
		       open_box 0;			(* 2 *)
		       ccsl_pp_expression place;
		       print_string ":=";
		       print_space();
		       ccsl_pp_expression expr;
		       close_box();			(* 1 *)
		     end) "" "" "" "," expr_expr_list;
	  print_string "]";
	  print_string ")";
	  close_box();					(* 0 *)
	end
    | Let (id_typ_expr_list, expr) ->
	begin
	  open_box 0;					(* 1 *)
	  assert( id_typ_expr_list <> [] );
	  print_string "(";
	  print_string "LET"; 
	  print_break 1 2;
	  pp_list
	    (fun (id, typ_opt, expr) ->
	       begin
		 open_box 0;
		 print_string id.id_token.token_name;
		 (match typ_opt with
		    | None ->
			()
		    | Some typ ->
			begin
			  print_string " : ";
			  ccsl_pp_type typ
			end);
		 print_string " =";
		 print_break 1 2;
		 ccsl_pp_expression expr;
		 close_box ()
	       end) "" "" "" "," id_typ_expr_list;
	  force_newline(); 
	  print_string "IN";
	  print_break 1 2;
	  open_box 0;					(* 2 *)
	  ccsl_pp_expression expr;
	  close_box();					(* 2 *)
	  print_string ")"; 
	  close_box()					(* 1 *)
	end
    | If (form_expr_list, expr) ->
	begin
	  assert( List.length(form_expr_list) = 1 );
	  open_box 0;					(* 1 *)
	  print_string "("; 
	  print_string " IF"; 
	  print_break 1 2;
	  pp_list (fun (cond, expr) ->
		     ccsl_pp_formula cond; 
		     print_space();
		     open_box 0;			(* 2 *)
		     print_string " THEN"; 
		     print_break 1 2;
		     ccsl_pp_expression expr;
		     close_box();			(* 1 *)
		     print_space();
		     open_box 0;			(* 2 *)
		  ) "" "" "" "ELSIF" form_expr_list;
	  open_box 0;					(* 3 *)
	  print_string " ELSE"; 
	  print_break 1 2;
	  ccsl_pp_expression expr; 
	  close_box();					(* 2 *)
	  print_space();
	  close_box();					(* 1 *)
	  print_string ")"; 
	  close_box();					(* 0 *)
	end
    | CCSL_Case (expr, match_list) ->
	(match match_list with
	   | [] -> assert false
	   | (mem1, var_list1, expr1) :: cases ->
 	       begin
		 open_box 0;		                   (* 1 *)
		 print_string "CASES"; 
		 print_break 1 2;
		 open_box 0;				   (* 2 *)
		 ccsl_pp_expression expr;
		 close_box ();				   (* 2 *)
		 print_space (); 
		 print_string "OF"; 
		 print_break 1 2;
		 open_box 0;				   (* 2a *)
		 (match mem1 with
		    | Unresolved t -> print_string t.token_name
		    | Resolved m -> print_string m#get_name);
		 if var_list1 <> [] then
		   begin
		     print_string "(";
		     pp_list 
		       (fun id -> print_string id.id_token.token_name ) 
		       "" "" "" ", " var_list1;
		     print_string ")";
		   end;
		 print_string ":"; 
		 print_break 1 2;
		 open_box 0;				   (* 3 *)
		 ccsl_pp_expression expr1; 
		 close_box ();				   (* 3 *)
		 List.iter
		   (fun (mem',var_list', expr') ->
		      begin
		   	print_string ","; 
			print_space ();
			force_newline(); (* added by BJ *)
			(match mem' with
			   | Unresolved t -> print_string t.token_name
			   | Resolved m -> print_string m#get_name);
		 	if var_list' <> [] then
			  begin
			    print_string "(";
			    pp_list 
			      (fun id -> print_string id.id_token.token_name ) 
			      "" "" "" ", " var_list';
			    print_string ")";
			  end;
			print_string ":";
		   	print_break 1 2;
		   	open_box 0;			   (* 3a *)
			ccsl_pp_expression expr';
			close_box ();			   (* 3a *)
		      end) cases;
		 close_box ();				   (* 2a *)
		 print_space ();
		 print_string "ENDCASES"; 
		 print_space();
		 close_box()				   (* 1 *)
	       end)
    | Expression form ->
	ccsl_pp_formula form
						(* not in ccsl_input_types *)
    | Term _
    | QualifiedTerm _
    | RecordTuple _
    | RecordSelection _
    | RecordUpdate _
    | List _	
    | SmartAbstraction _
    | SmartApplication _
    | Case _
(*     | Reduce _
 *)
    | Every _
    | RelEvery _
    | Map _
    | Comment_str _
    | Comment_expr _
    | Comprehension _ ->
      	assert(false)

	
and ccsl_pp_formula (* : top_formulas -> unit *)
  = function
    | FormLoc(f, l) -> ccsl_pp_formula f
    | True ->
      	print_string "TRUE"
    | False ->
	print_string "FALSE"
    | Not form ->
	begin
	  open_box 0;
	  print_string "("; 
	  print_string "NOT";
	  print_space();
	  ccsl_pp_formula form;
	  print_cut();
	  print_string ")";
	  close_box()
	end
    | And form_list ->
	begin
	  open_box 0; 
	  print_string "(";
	  pp_list ccsl_pp_formula "(" ")" "TRUE" "AND" form_list; 
	  print_cut();
	  print_string ")";
	  close_box ();
	end
    | Or form_list ->
	begin
	  open_box 0; 
	  print_string "(";
	  pp_list ccsl_pp_formula "(" ")" "FALSE" "OR" form_list; 
	  print_cut();
	  print_string ")";
	  close_box ();
	end
    | Implies (prem_expr, concl_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  ccsl_pp_formula prem_expr;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "IMPLIES";
	  print_space();
	  print_string "("; 
	  ccsl_pp_formula concl_expr;
	  print_cut();
	  print_string ")";
	  close_box(); 
	  close_box()
	end
    | Iff (prem_expr, concl_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  ccsl_pp_formula prem_expr;
	  print_string ")";
	  print_space ();
	  open_box 0;
	  print_string "IFF";
	  print_space();
	  print_string "("; 
	  ccsl_pp_formula concl_expr;
	  print_cut();
	  print_string ")";
	  close_box(); 
	  close_box()
	end
    | Equal (left_expr, right_expr) ->
	begin
	  open_box 0;
	  print_string "("; 
	  ccsl_pp_expression left_expr;
	  print_break 1 2; 
	  print_string "=";
	  print_space(); 
	  ccsl_pp_expression right_expr;
	  print_string ")"; 
	  close_box ()
	end
    | Forall (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       ccsl_pp_formula form
	   | l ->
	       begin
		 open_box 0;				(* 1 *)
		 print_string "("; 
	    	 print_string "FORALL (";
	    	 (* open_box 0;  *)
		 pp_list ccsl_pp_var "" "" "" "," str_typ_list; 
		 (* close_box(); *)
	    	 print_string ") .";
	    	 print_break 1 2;
		 ccsl_pp_formula form;
		 print_string ")"; 
		 close_box ()
	       end)
    | Exists (str_typ_list, form) ->
	(match str_typ_list with
	   | [] ->
	       ccsl_pp_formula form
	   | l ->
	       begin
		 open_box 0;
		 print_string "("; 
	    	 print_string "EXISTS (";
	    	 open_box 0; 
		 pp_list ccsl_pp_var "" "" "" "," str_typ_list; 
		 close_box();
	    	 print_string ") .";
	    	 print_break 1 2;
		 ccsl_pp_formula form;
		 print_string ")"; 
		 close_box ()
	       end)
    | Formula expr ->
	ccsl_pp_expression expr
    | Obseq (_ ,ex1,ex2) ->
	begin
	  open_box 2;					    (* 1 *)
	  print_string "(";
	  ccsl_pp_expression ex1;
	  print_string " ~";
	  print_space();
	  ccsl_pp_expression ex2;
	  print_string ")";
	  close_box()					    (* 1 *)
	end
							(* not in ccsl_input *)
(*     | LessOrEqual _
 *)
    | ConstantPredicate _
    | Bisim _
    | MetaImplies _
    | MetaForall _ ->
	assert(false)

and ccsl_pp_arg_list al =
  let pp_arg = (function
		  | TypeArgument(t) -> ccsl_pp_type t 
	       )
  in
    ((pp_list pp_arg "" "" "" "," al) : unit)
    

and ccsl_pp_var (name, typ) =
      begin
	open_box 2;
	print_string (name ^ " :");
	print_space(); 
	ccsl_pp_type typ;
	close_box ()
      end


(***********************************************************************
 ***********************************************************************
 *
 * string interface
 *
 *)

let string_of_ccsl_variance = stringwrapper ccsl_pp_variance

let string_of_ccsl_type = stringwrapper ccsl_pp_type

let string_of_ccsl_inst_iface = stringwrapper ccsl_pp_inst_iface

let string_of_ccsl_arguments args = 
  "[" ^ (stringwrapper ccsl_pp_arg_list args) ^ "]"

let string_of_ccsl_expression = stringwrapper ccsl_pp_expression

let string_of_ccsl_formula = stringwrapper ccsl_pp_formula


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

