(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Tuesday 21 May 02 15:38:03 tews@ithif51>
 *
 * Bisimilarity theories
 *
 * $Id: bisim_theory.ml,v 1.12 2002/05/22 13:42:38 tews Exp $
 *
 *)

open Util
open Global
open Top_variant_types
open Name_space
open Names
open Logic_util
open Classtypes
open Types_util
open Theory_class
open Lifting;;

(***********************************************************************
 ***********************************************************************
 *
 * general Bisimilarity
 *
 *)

class ['class_type, 'member_type] ccsl_pre_bibisim_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_coalgebra1;
	 name_of_coalgebra2;
	 name_of_private_class_rel cl;
	 name_of_public_class_rel cl;
	 name_of_private_bibisimulation cl;
	 name_of_public_bibisimulation cl;
	 name_of_private_bibisim cl;
	 name_of_public_bibisim cl
       ]

    method get_name = ccsl_bibisim_theory_name cl

    method get_parameters = self#double_self_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private self1_subst = 
      [Self, BoundTypeVariable(self1)]

    method private self2_subst = 
      [Self, BoundTypeVariable(self2)]

    method private imports =
      let do_const_anc anc args = 
	(ccsl_bibisim_theory_name anc, 
	 self#self1_argument
	 :: self#self2_argument
	 :: args) in
      let do_nonconst_anc anc args = 
	(ccsl_full_bibisim_theory_name anc, 
	 self#self1_argument
	 :: self#self2_argument
	 :: ((ccsl_substitute_arguments self#self1_subst args) 
	     @ (ccsl_substitute_arguments self#self2_subst args)))
      in
      let ancestor_imports = 
	List.map 
	(function
	   | Resolved_renaming (anc,args,_,_,_) -> 
	       if (count_self_args args) = 0 
	       then do_const_anc anc args
	       else do_nonconst_anc anc args
					(* no other stuff *)
	   | Unresolved_renaming _
(* NO JAVA ANCESTORS
 * 	      | Resolved_ancestor _ 
 * 	      | Unresolved_ancestor _ 
 *)
	     -> assert(false)
	)
	cl#get_ancestors in
      let components_import = 
	List.fold_right
	  (fun (v,c,args) accu ->
	     let subst_args1 = 
	       ccsl_substitute_arguments self#self1_subst args
	     in let subst_args2 =
	       ccsl_substitute_arguments self#self2_subst args
	     in let subst_args = subst_args1 @ subst_args2
	     in
	       match c#get_kind with
		 | Spec_adt ->
		     (ccsl_adt_rellift_theory_name c, subst_args) :: accu
		 | Spec_class ->
		     let modelarg = c#get_model_type_argument 
		     in
		       (ccsl_full_bibisim_theory_name c, 
			(modelarg subst_args1) ::
			(modelarg subst_args2) ::
			subst_args) :: accu
		 | Spec_sig -> accu
		 | Spec_Spec -> assert(false)
	  ) cl#get_components []
      in
      	Import(
 	  [((ccsl_interface_theory_name cl), 
	    self#self1_argument :: orig_arguments);
      	   ((ccsl_interface_theory_name cl), 
	      self#self2_argument :: orig_arguments)
	  ]
	  @ ancestor_imports
	  @ components_import 
	)

	  (* the workhorse for generating the relation lifting:
	   * takes a method list and generates the relation lifting for
	   * those methods and all ancestors
	  *)
    method private do_class_rel class_rel full_class_rel method_list =
  (* Debugging support *)
      let class_name = cl#get_name in
  (* *)
      let sub_ns = sub_space ns in
      let type_of_rel = 	
	Function(
	  Product([BoundTypeVariable(self1);
		   BoundTypeVariable(self2)]),
	  Bool) in
      let r = create_one_id sub_ns type_of_rel in
      let (self_var1_typ, self_var2_typ) = create_id_pairs sub_ns [Self] in
      let self_var1 = fst( List.hd self_var1_typ) in
      let self_var2 = fst( List.hd self_var2_typ) in
      let rel_expr = Term(r,Always,[]) in
      let self_lifting = (Some rel_expr, Some rel_expr) in
      let ancestor_list = 
	List.map (function
		    | Resolved_renaming(oanc, args,_,ianc,_) -> oanc,args
					(* no other stuff here *)
		    | Unresolved_renaming _
		    | _ -> assert(false)
		 ) cl#get_ancestors
      in
      let self_type_arguments = [self#self1_argument; self#self2_argument] in
      let do_const_ancestor (anc, anc_arguments) =
	Formula(
	  Application(
            Application(
              SmartApplication(
		Term(class_rel anc, Always,
                     self_type_arguments @ anc_arguments), 
		[Application(
		   Term(super_access_method anc, Always,[]), 
		   coalgebra1_term);
		 Application(
		   Term(super_access_method anc, Always,[]), 
		   coalgebra2_term)]),
              rel_expr),
            Tuple([Term(self_var1, Always,[]);
		   Term(self_var2, Always,[])]))) 
      in
      let do_nonconst_ancestor (anc, anc_arguments) =
	Formula(
          Application(

	    SmartApplication(			      (* rels *)
	      SmartApplication(			      (* parameters *)
		SmartApplication(		      (* coalgebras *)
		  Term(full_class_rel anc, Always,
		       self_type_arguments
		       @ (ccsl_substitute_arguments self#self1_subst 
			    anc_arguments) 
		       @ (ccsl_substitute_arguments self#self2_subst 
			    anc_arguments)), 
		  [Application(
		     Term(super_access_method anc, Always,[]), 
		     coalgebra1_term);
		   Application(
		     Term(super_access_method anc, Always,[]), 
		     coalgebra2_term)]),
		argument_list_fullrellift
		  eq_types
		  (ccsl_substitute_types self#self1_subst)
		  (ccsl_substitute_types self#self2_subst)
		  [] 
		  self_lifting
		  sub_ns
		  anc#get_parameters
		  anc_arguments),
	      if (make_simple anc#get_self_variance) = Mixed
	      then
		[rel_expr; rel_expr]
	      else
		[rel_expr]),
	    
            Tuple([Term(self_var1, Always,[]);
		   Term(self_var2, Always,[])])))
	
      in let do_ancestor ((_,anc_arguments) as a) =
	  if (count_self_args anc_arguments) = 0
	  then do_const_ancestor a
	  else do_nonconst_ancestor a
      in
      	Defn(class_rel cl,
	     [[Declared(name_of_coalgebra1, self#coalgebra1_type);
	       Declared(name_of_coalgebra2, self#coalgebra2_type)]],
	     Function(type_of_rel, type_of_rel) ,
	     Abstraction
	       ([r, type_of_rel],
		Abstraction(
		  [(self_var1, BoundTypeVariable(self1));
		   (self_var2, BoundTypeVariable(self2))],
		  Expression(
		    And(
		      (List.map do_ancestor ancestor_list)
		      @
		      (List.map
			 (fun m -> class_method_full_rel_lifting 
			      eq_types
			      (ccsl_substitute_types self#self1_subst)
			      (ccsl_substitute_types self#self2_subst)
			      []
			      self_lifting
			      sub_ns
			      coalgebra1_term
			      coalgebra2_term
			      m
			      self_var1 self_var2)
			 method_list)
		    )))))

	  (* generate full relation lifting *)
    method private private_class_rel = 
      self#do_class_rel
	name_of_private_class_rel name_of_full_class_rel
	cl#get_sig_actions
	
	(* generate public relation lifting *)
    method private public_class_rel =
      self#do_class_rel 
	name_of_public_class_rel name_of_full_public_class_rel
	(List.filter
	   (fun m -> m#get_visibility = Public)
	   cl#get_sig_actions)

    method private bisimulation_decl name lifting =
      let sub_ns = sub_space ns in
      let type_of_rel = 	
	Function(
	  Product([BoundTypeVariable(self1);
		   BoundTypeVariable(self2)]),
	  Bool) in
      let r = create_one_id sub_ns type_of_rel in
      let (self_var1_typ, self_var2_typ) = create_id_pairs sub_ns [Self] in
      let self_var1 = fst( List.hd self_var1_typ) in
      let self_var2 = fst( List.hd self_var2_typ) in
      	Defn( name, 
	      [[Declared(name_of_coalgebra1, self#coalgebra1_type);
	       	Declared(name_of_coalgebra2, self#coalgebra2_type)]],
	      Function(type_of_rel , Bool),
	      Abstraction(
	       	[r, type_of_rel],
	       	Expression(
		  Forall( 
		    [self_var1, BoundTypeVariable(self1); 
		     self_var2, BoundTypeVariable(self2)],
		    Implies(
		      Formula(
		       	Application(
			  Term(r,Always,[]),
			  Tuple(
			    [Term(self_var1,Always,[]); 
			     Term(self_var2,Always,[])]))),
		      Formula(
		       	Application(
			  Application(
			    SmartApplication(
			      Term(lifting, Always,[]),
			      [coalgebra1_term;coalgebra2_term]),
			    Term(r,Always,[])),
			  Tuple(
			    [Term(self_var1,Always,[]); 
			     Term(self_var2,Always,[])])
			)))))))

    method private private_bibisimulation_decl =
      self#bisimulation_decl (name_of_private_bibisimulation cl)
	(name_of_private_class_rel cl)

    method private public_bibisimulation_decl =
      self#bisimulation_decl (name_of_public_bibisimulation cl)
	(name_of_public_class_rel cl)

    method private struct_bisimulation =
      let sub_ns = sub_space ns in 
      let (decl1_list, decl2_list, c1_tuple, c2_tuple ) = 
	self#two_coalgebras_as_tuple sub_ns in
      let type_of_rel = 	
	Function(
	  Product([BoundTypeVariable(self1);
		   BoundTypeVariable(self2)]),
	  Bool) in
      let r = create_one_id sub_ns type_of_rel 
      in
	Defn(name_of_private_struct_bibisimulation cl,
	     [decl1_list; decl2_list; [ Undeclared(r, type_of_rel) ]],
	     Bool,
	     Application(
	       Application(
		 Term(name_of_private_bibisimulation cl, Always, []),
		 Tuple([c1_tuple; c2_tuple])
	       ),
	       Term(r,Always,[])
	     ))


    method private bisim_decl name bisimulation =
      let sub_ns = sub_space ns in
      let type_of_rel = 	
	Function(
	  Product([BoundTypeVariable(self1);
		   BoundTypeVariable(self2)]),
	  Bool) in
      let r = create_one_id sub_ns type_of_rel in
      let (self_var1_typ, self_var2_typ) = create_id_pairs sub_ns [Self] in
      let self_var1 = fst( List.hd self_var1_typ) in
      let self_var2 = fst( List.hd self_var2_typ) in
      let rel_form = ConstantPredicate(r) in
	Defn(name,
	      [[Declared(name_of_coalgebra1, self#coalgebra1_type);
	       	Declared(name_of_coalgebra2, self#coalgebra2_type)]],
	      type_of_rel,
	      Abstraction(
		[self_var1, BoundTypeVariable(self1); 
		 self_var2, BoundTypeVariable(self2)],
		Expression(
		  Exists(
		    [r, type_of_rel],
		    And[
		      Formula(
		      	Application(
		      	  SmartApplication(
			    Term(bisimulation, Always, []),
			    [coalgebra1_term; coalgebra2_term]),
			  Term(r,Always,[])));
		      Formula(
		      	Application(
			  Term(r,Always,[]),
			  Tuple(
			    [Term(self_var1,Always,[]); 
			     Term(self_var2,Always,[])])))
		  ]))))


    method private private_bibisim_decl =
      self#bisim_decl (name_of_private_bibisim cl) 
	(name_of_private_bibisimulation cl)

    method private public_bibisim_decl =
      self#bisim_decl (name_of_public_bibisim cl) 
	(name_of_public_bibisimulation cl)

    method make_body =
      [
	self#imports;
       	self#coalgebra1_decl;
       	self#coalgebra2_decl;
       	self#private_class_rel;
       	self#public_class_rel;
	self#private_bibisimulation_decl;
	self#struct_bisimulation;
	self#public_bibisimulation_decl
      ]
      @ ( 
	if cl#has_feature HasGreatestBisimFeature
	then
	  [
	    self#private_bibisim_decl;
	    self#public_bibisim_decl
	  ]
	else
	  []
      )

    method get_proofs = []
end

class ccsl_bibisim_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_bibisim_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Bisimilarity Rewrites
 *
 * define a generic theory, for all private/public bibisim/bisim_eq theories
 *
 * it is parametrized by private methods self1_inst, self2_inst
 * coalgebra1, coalgebra2, bisim_rel_ex
 *
 *)

class virtual ['class_type, 'member_type] 
  ccsl_pre_generic_bisim_rewrite_theory 
  (cl : 'class_type)  
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  (bisim_lemma : <get_name : string; ..> -> string) =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

    (* definined in the children classes 
     *
     * initializer reserve ns []
     *
     * method get_name = 
     *
     * method get_parameters = 
     *)

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method get_proofs = []


    method private do_bisim_method m =
      let sub_ns = sub_space ns in
      let _ = make_flat sub_ns in
					(* get names for the variables *)
      let xy_decl = create_ids sub_ns [Self;Self] in
				(* extrace the variables as expressions *)
      let x = fst (List.nth xy_decl 0) in
      let y = fst (List.nth xy_decl 1) in
		   (* constract the substitution for the relation lifting 
		    * self1/2_inst will be overriden later to get 
		    * Self1 and Self2 for Bisimilarity and Self for
		    * BisimilarityEqivalence
		    *)
      let self1_subst = (if self#self1_inst = Self 
			 then (fun x -> x)
			 else ccsl_substitute_types [Self, self#self1_inst]) 
      in
      let self2_subst = (if self#self2_inst = Self
			 then (fun x -> x)
			 else ccsl_substitute_types [Self, self#self2_inst]) 
      in
      let prem = 
	Formula(
	  Application(self#bisim_rel_ex, 
		      Tuple[Term(x,Always,[]);Term(y,Always,[])])) in
      let form = class_method_full_rel_lifting
		   eq_types
		   self1_subst self2_subst
		   []
		   (Some self#bisim_rel_ex, Some self#bisim_rel_ex)
		   sub_ns 
		   self#coalgebra1 self#coalgebra2
		   m 
		   x y
      in
	(* merge quantification over x,y in *)
      let form2 = match form with
	| Forall(vars,form) -> 
	    Forall((x,self#self1_inst) :: (y,self#self2_inst) :: vars, 
		   Implies(prem,form))
	| f -> Forall([(x,self#self1_inst); (y,self#self2_inst)], 
		      Implies(prem,form)) in
      let form3 = match !output_mode with
	| Pvs_mode -> make_pvs_rewrite_form form2
	| Isa_mode -> make_isa_rewrite_form form2
      in
(*	Proved( 
*)
	  Lemma(
	    bisim_lemma m,
	    form3)
(*	    ,
	  proof)
*)

end


(***********************************************************************
 *
 * Rewrite lemmas for private bisimulation
 *
 * instanciate the theory above
 *)

class ['class_type, 'member_type] ccsl_pre_bibisim_rewrite_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  (theory_name : string) 
  (bisim_name : string)
  (method_filter : ('class_type, 'member_type) ccsl_pre_member_type -> bool) 
  (bisim_lemma : <get_name : string; ..> -> string) =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_generic_bisim_rewrite_theory 
      cl eq_types bisim_lemma

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      [name_of_coalgebra1;name_of_coalgebra2]

    method get_name = theory_name

	(* override section *)
    method get_parameters = self#double_self_parameters

    method private self1_inst = BoundTypeVariable self1

    method private self2_inst = BoundTypeVariable self2

    method private coalgebra1 = Term(name_of_coalgebra1,Always,[])
    method private coalgebra2 = Term(name_of_coalgebra2,Always,[])

    method private bisim_rel_ex = 
      SmartApplication(
	Term(bisim_name, Always,[]),
	[self#coalgebra1;self#coalgebra2])


    method make_body =
      Import[ ccsl_bibisim_theory_name cl, self#double_self_arguments ] ::
      self#coalgebra1_decl :: 
      self#coalgebra2_decl ::
      Comment("Rewrite lemmas for methods")::
      (List.map self#do_bisim_method
	 (List.filter method_filter cl#get_all_members))

end


class ccsl_private_bibisim_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_bibisim_rewrite_theory cl
  eq_ccsl_types
  (ccsl_private_bibisim_rewrite_theory_name cl)
  (name_of_private_bibisim cl)
  (fun m -> m#is_action)
  name_of_private_bibisim_lemma

class ccsl_public_bibisim_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_bibisim_rewrite_theory cl
  eq_ccsl_types
  (ccsl_public_bibisim_rewrite_theory_name cl)
  (name_of_public_bibisim cl)
  (fun m -> (m#is_action && (m#get_visibility = Public)))
  name_of_public_bibisim_lemma


(***********************************************************************
 ***********************************************************************
 *
 * Bisimilarity on the same class
 *
 *)


class ['class_type, 'member_type] ccsl_pre_bisim_eq_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
       [ name_of_coalgebra;
	 name_of_private_bisimulation_eq cl;
	 name_of_public_bisimulation_eq cl;
	 name_of_private_bisim_eq cl;
	 name_of_public_bisim_eq cl
       ]

    method get_name = ccsl_bisim_eq_theory_name cl

    method get_parameters = self#simple_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private imports =
      Import [ccsl_bibisim_theory_name cl, 
              self_argument :: self_argument :: orig_arguments]

    method private private_bisimulation_eq_decl =
      Defn( name_of_private_bisimulation_eq cl, 
	     [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	     Function(
	       Function(
		 Product([Self; Self]),
		 Bool),
	     Bool),
	     SmartApplication(
	       Term(name_of_private_bibisimulation cl, Always, []),
	       [coalgebra_term;coalgebra_term]))

    method private private_bisim_eq_decl =
      Defn( name_of_private_bisim_eq cl, 
	     [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	     Function(
	       Product([Self; Self]),
	       Bool),
	     SmartApplication(
	       Term(name_of_private_bibisim cl, Always, []),
	       [coalgebra_term; coalgebra_term]))

    method private public_bisimulation_eq_decl =
      Defn( name_of_public_bisimulation_eq cl, 
	     [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	     Function(
	       Function(Product([Self; Self]),
		 Bool),
	     Bool),
	     SmartApplication(
	       Term(name_of_public_bibisimulation cl, Always, []),
	       [coalgebra_term; coalgebra_term]))

    method private public_bisim_eq_decl =
      Defn( name_of_public_bisim_eq cl, 
	     [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	     Function(
	       Product([Self; Self]),
	       Bool),
	     SmartApplication(
	       Term(name_of_public_bibisim cl, Always, []),
	       [coalgebra_term; coalgebra_term]))


    method make_body =
      [
	self#imports;
       	self#coalgebra_decl;
	self#private_bisimulation_eq_decl
      ]
      @ ( 
	if cl#has_feature HasGreatestBisimFeature
	then
	  [self#private_bisim_eq_decl]
	else
	  []
      ) @ [
	self#public_bisimulation_eq_decl
      ] @ (
	if cl#has_feature HasGreatestBisimFeature
	then
	  [self#public_bisim_eq_decl]
	else
	  []
      )

    method get_proofs = []
end

class ccsl_bisim_eq_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_bisim_eq_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Rewrite lemmas for bisimulation equivalence
 *
 * instanciate the generic theory ccsl_pre_generic_bisim_rewrite_theory
 *)

class ['class_type, 'member_type] ccsl_pre_bisim_eq_rewrite_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  (theory_name : string) 
  (public_version : bool)
  (method_filter : ('class_type, 'member_type) ccsl_pre_member_type -> bool) 
  (bisim_lemma : <get_name : string; ..> -> string) =
  object (self : 'self)
    
    inherit 
      ['class_type, 'member_type] ccsl_pre_generic_bisim_rewrite_theory 
      cl eq_types bisim_lemma

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      [name_of_coalgebra]

    method get_name = theory_name

	(* override section *)
    method get_parameters = self#simple_parameters

    method private self1_inst = Self

    method private self2_inst = Self

    method private coalgebra1 = coalgebra_term
    method private coalgebra2 = coalgebra_term

    method private bisim_rel_ex = 
      Application(
	Term(
	  (if public_version 
	   then name_of_public_bisim_eq cl
	   else name_of_private_bisim_eq cl), 
	  Always,[]),
	coalgebra_term)

    method private bisim_eq =
      Lemma(
	(if public_version 
	 then name_of_public_eq_bisim_lemma
	 else name_of_private_eq_bisim_lemma),
	Formula(
	  Application(
	    Application(
	      Term(
		(if public_version 
		 then name_of_public_bisimulation_eq cl
		 else name_of_private_bisimulation_eq cl)
		,Always,[]),
	      coalgebra_term),
	    Abstraction(
	      ["x1", Self; "x2", Self],
	      Expression(
		Equal(Term("x1",Always,[]), Term("x2",Always,[]))))
	    )))
	
    method private bisim_refl = 
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns Self in
      	Lemma(
	  (if public_version 
	   then name_of_public_bisim_refl_lemma
	   else name_of_private_bisim_refl_lemma),
	  Forall([x,Self],
		 Formula(Application(
			   self#bisim_rel_ex,
			   Tuple([Term(x,Always,[]);
			          Term(x,Always,[])])))))
	  
    method private bisim_symm = 
      let sub_ns = sub_space ns in
      let xy = create_ids sub_ns [Self;Self] in
      let x = fst( List.nth xy 0) in
      let y = fst( List.nth xy 1) 
      in
	Lemma(
	  (if public_version 
	   then name_of_public_bisim_sym_lemma
	   else name_of_private_bisim_sym_lemma), 
          Forall(
	    xy,
	    Implies(
	      Formula(
		Application(
		  self#bisim_rel_ex,
		  Tuple([Term(x,Always,[]); Term(y,Always,[])]))),
	      Formula(
		Application(
		  self#bisim_rel_ex,
		  Tuple([Term(y,Always,[]); Term(x,Always,[])]))))))
	  
    method private bisim_trans = 
      let sub_ns = sub_space ns in
      let xyz = create_ids sub_ns [Self;Self;Self] in
      let x = fst( List.nth xyz 0) in
      let y = fst( List.nth xyz 1) in
      let z = fst( List.nth xyz 2) in
	Lemma(
	  (if public_version 
	   then name_of_public_bisim_trans_lemma
	   else name_of_private_bisim_trans_lemma), 
	  Forall(
	    xyz,
	    Implies(
	      And(
		[Formula(
		   Application(
		     self#bisim_rel_ex,
		     Tuple([Term(x,Always,[]);Term(y,Always,[])])));
		 Formula(
		   Application(
		     self#bisim_rel_ex,
		     Tuple([Term(y,Always,[]);Term(z,Always,[])])))]),
	      Formula(
		Application(
		  self#bisim_rel_ex,
		  Tuple([Term(x,Always,[]);Term(z,Always,[])]))))))
	  
    method private bisim_equiv =
      match !output_mode with
	| Pvs_mode -> 
	    Lemma(
	      (if public_version 
	       then name_of_public_bisim_equiv_lemma
	       else name_of_private_bisim_equiv_lemma),
	      Formula(
	  	Application(Term(name_of_equivalence,Always,[]),
			    self#bisim_rel_ex)))
	| Isa_mode -> 
	    Comment("So bisimilarity is an equivalence relation.")
	
	
    method make_body =
      Import[ ccsl_bisim_eq_theory_name cl, self#simple_arguments ] ::
      self#coalgebra_decl :: 
      self#bisim_eq ::
      self#bisim_refl ::
      self#bisim_symm ::
      self#bisim_trans ::
      self#bisim_equiv ::
      Comment("Rewrite lemmas for methods")::
      (List.map self#do_bisim_method
	 (List.filter method_filter cl#get_all_members))

end


class ccsl_private_bisim_eq_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_bisim_eq_rewrite_theory cl
  eq_ccsl_types
  (ccsl_private_bisim_eq_rewrite_theory_name cl)
  false
  (fun m -> m#is_action)
  name_of_private_bisim_eq_lemma

class ccsl_public_bisim_eq_rewrite_theory cl =
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_bisim_eq_rewrite_theory cl
  eq_ccsl_types
  (ccsl_public_bisim_eq_rewrite_theory_name cl)
  true
  (fun m -> (m#is_action && (m#get_visibility = Public)))
  name_of_public_bisim_eq_lemma
      

class ['class_type, 'member_type] ccsl_pre_req_bisim_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
   -> ('class_type, 'member_type) top_pre_types -> bool)
  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)
    
    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory
	
    initializer reserve ns 
      (
	[ name_of_coalgebra ] @ 
       	(List.map (fun (t,n) -> n) 
	   cl#get_rel_lifting_requests
       	)
      )

    method get_name = ccsl_req_bisim_theory_name cl
			
    method get_parameters = self#simple_parameters 
			      
    initializer top_theory#override_file_name (ccsl_class_file_name cl)
      

    method private do_import =
      Import(
	[
	  if cl#has_feature HasGreatestBisimFeature
	  then
	    ((ccsl_private_bisim_eq_rewrite_theory_name cl), 
	     self#simple_arguments)
	  else
	    (ccsl_interface_theory_name cl,
	     self#simple_arguments)
	]
      )

    method private self1_subst = 
      [Self, BoundTypeVariable(self1)]
      
    method private self2_subst = 
      [Self, BoundTypeVariable(self2)]
      
    method private do_request_decl (typ,name) =
      let sub_ns = sub_space ns in
      let (typ_vars1, typ_vars2) = create_id_pairs sub_ns [typ] in
      let typ_var1 = fst( List.hd typ_vars1) in
      let typ_var2 = fst( List.hd typ_vars2) in
      let r = name_of_private_bisim_eq cl in
      let rel_exp = Application(Term(r,Always,[]), coalgebra_term) in
      let no_subst = (fun x -> x) in
      let self_lifting = (Some rel_exp, Some rel_exp) in
	Defn(
	  name,
	  [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	  Function(Product([typ;typ]),Bool),
	  Abstraction(
	    [typ_var1, typ;
	     typ_var2, typ],
	    Expression
	      (rellift 
		 eq_types
		 no_subst                       (* self1_subst *)   
		 no_subst			(* self2_subst *)   
		 self_lifting			(* self_lifting *)  
		 sub_ns 			(* top_name_space *)
		 typ				(* typ *)           
		 (Term(typ_var1, Always, []))	(* expr1 *)         
		 (Term(typ_var2, Always, []))	(* expr2 *)
	      )
	  )
	)
	  
	  
    method make_body =
      [
	self#do_import;

       	self#coalgebra_decl;
      ] @
      (List.map self#do_request_decl cl#get_rel_lifting_requests)
      
    method get_proofs = []
			  
end (* ccsl_pre_req_bisim_theory *)

class ccsl_req_bisim_theory cl =
  [ccsl_iface_type, ccsl_member_type] 
    ccsl_pre_req_bisim_theory cl eq_ccsl_types 

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

