/*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * adopted 25.1.99 by Hendrik
 *
 * Time-stamp: <Sunday 23 June 02 15:25:54 tews@ithif56.inf.tu-dresden.de>
 *
 * CCSL Grammar 
 *
 * $Id: grammar.mly,v 1.26 2002/07/03 12:01:14 tews Exp $
 *
 */

%{
  
  open Util
  open Global
  open Top_variant_types
  open Types_util
  open Iface_class
  open Member_class
  open Parser

  let d s =
    if debug_level _DEBUG_PARSER
    then begin
      prerr_string ("REDUCE " ^ s ^ "\n");
      flush stderr
    end;;

  let ffex expr =
    match expr with
      | ExprLoc(Expression(f),l) -> FormLoc(f,l)
      | ExprLoc(ex,l) -> FormLoc(Formula(ex), l)
      | _ -> assert(false)

  let exff form =
    match form with
      | FormLoc(Formula(e),l) -> ExprLoc(e,l)
      | FormLoc(f,l) -> ExprLoc(Expression(f),l)
      | _ -> assert(false)

  let parse_error msg = 
    Error.error_message !last_loc msg
;;
      
	    
%}

/***********************************************************************
 * Token Section
 */

%token EOF
%token EQUAL
%token OBSEQ
%token COLON
%token DOUBLECOLON
%token ASSIGN
%token SEMICOLON
%token COMMA
%token ARROW
%token DOT
%token QUESTIONMARK

%token <Top_variant_types.location_type> OBRACKET
%token <Top_variant_types.location_type> CBRACKET
%token <Top_variant_types.location_type> OPAREN
%token <Top_variant_types.location_type> CPAREN
%token <Top_variant_types.location_type> OBRACE
%token <Top_variant_types.location_type> CBRACE

%token <int * Top_variant_types.location_type> PROJ_N

%token <Top_variant_types.token_type> INFIX_EXP
%token <Top_variant_types.token_type> INFIX_MUL
%token <Top_variant_types.token_type> INFIX_ADD
%token <Top_variant_types.token_type> INFIX_SHARP
%token <Top_variant_types.token_type> INFIX_REL

%token CLASSSPEC	/* KEYWORD */
%token FINAL		/* KEYWORD */
%token END		/* KEYWORD */
%token TYPE		/* KEYWORD */
%token GROUNDTYPE	/* KEYWORD */
%token GROUNDTERM	/* KEYWORD */
%token GROUNDSIGNATURE  /* KEYWORD */
%token POS		/* KEYWORD */
%token NEG		/* KEYWORD */
%token MIXED		/* KEYWORD */
%token CONSTANT		/* KEYWORD */
%token ADT		/* KEYWORD */
%token IMPORTING        /* KEYWORD */
%token INHERIT		/* KEYWORD */
%token RENAMING		/* KEYWORD */
%token FROM		/* KEYWORD */
%token AS		/* KEYWORD */
%token CONSTRUCTOR	/* KEYWORD */
%token ATTRIBUTE	/* KEYWORD */
%token METHOD		/* KEYWORD */
%token DEFINING		/* KEYWORD */
%token SELFVAR		/* KEYWORD */
%token VAR		/* KEYWORD */
%token ASSERTION	/* KEYWORD */
%token THEOREM		/* KEYWORD */
%token PRIVATE		/* KEYWORD */
%token PUBLIC		/* KEYWORD */
%token BOOL		/* KEYWORD */
%token REQUEST          /* KEYWORD */

%token <Top_variant_types.location_type> CREATION	/* KEYWORD */
%token <Top_variant_types.location_type> BEGIN		/* KEYWORD */
%token <Top_variant_types.location_type> SELF		/* KEYWORD */
%token <Top_variant_types.location_type> CARRIER	/* KEYWORD */

/* for formulae */
%token <Top_variant_types.location_type> TRUE		/* KEYWORD */
%token <Top_variant_types.location_type> FALSE		/* KEYWORD */
%token <Top_variant_types.location_type> CASES		/* KEYWORD */
%token <Top_variant_types.location_type> OF		/* KEYWORD */
%token <Top_variant_types.location_type> ENDCASES	/* KEYWORD */
%token <Top_variant_types.location_type> IF		/* KEYWORD */
%token <Top_variant_types.location_type> THEN		/* KEYWORD */
%token <Top_variant_types.location_type> ELSE		/* KEYWORD */
%token <Top_variant_types.location_type> LAMBDA		/* KEYWORD */
%token <Top_variant_types.location_type> AND		/* KEYWORD */
%token <Top_variant_types.location_type> NOT		/* KEYWORD */        
%token <Top_variant_types.location_type> OR 		/* KEYWORD */
%token <Top_variant_types.location_type> IMPLIES	/* KEYWORD */
%token <Top_variant_types.location_type> IFF		/* KEYWORD */
%token <Top_variant_types.location_type> FORALL		/* KEYWORD */
%token <Top_variant_types.location_type> EXISTS		/* KEYWORD */
%token <Top_variant_types.location_type> ALWAYS		/* KEYWORD */
%token <Top_variant_types.location_type> EVENTUALLY	/* KEYWORD */
%token <Top_variant_types.location_type> FOR            /* KEYWORD */
%token <Top_variant_types.location_type> WITH           /* KEYWORD */
%token <Top_variant_types.location_type> LET            /* KEYWORD */
%token <Top_variant_types.location_type> IN             /* KEYWORD */

%right QUANTOR
%left IFF
%right IMPLIES 
%left OR
%left AND
%left IF THEN ELSE
%left NOT
%nonassoc OBSEQ EQUAL INFIX_REL
%left INFIX_SHARP
%left INFIX_ADD
%left INFIX_MUL
%right INFIX_EXP
%left OPAREN
%left WITH
%left CASES LAMBDA FORALL EXISTS ALWAYS EVENTUALLY LET
%left TRUE FALSE PROJ_N ID VALUE 
%left DOT
%left APPLICATION

%token <Top_variant_types.token_type> ID
%token <string * Top_variant_types.location_type> VALUE
%token <Classtypes.ccsl_assertion> PVS_FORMULA
%token <string> STRING
/* the include token is filtered and handled in the abstract_lexer */
%token <string * Top_variant_types.location_type> INCLUDE

/*****
 * End of Token Section
 ***********************************************************************/

%type <Classtypes.ccsl_input_types>	pvstype
%type <Classtypes.ccsl_input_types list>	pvstypelist
%type <Classtypes.ccsl_argument_type list>	arglist
%type <Classtypes.ccsl_parameter_type list> paramdeclarations 
%type <Classtypes.ccsl_parameter_type list> paramdeclaration
%type <Classtypes.ccsl_parameter_type list> typeparameters
%type <Top_variant_types.variance_type>		optvariance
%type <int>					numberorquestion
%type <Classtypes.ccsl_ast option list>		declarationlist 
%type <Classtypes.ccsl_ast list>		file
%type <Classtypes.ccsl_ast option>		declaration
%type <Classtypes.ccsl_ast option>		groundtermdef termdef
%type <Classtypes.ccsl_ast option>		typedef
%type <Classtypes.ccsl_iface_type>		classspec
%type <Classtypes.ccsl_iface_type>		adtspec
%type <Classtypes.ccsl_iface_type>		groundsignature
%type <token_type list>				adtaccessors
%type <Top_variant_types.token_type list>	idlist
%type <Classtypes.ccsl_importing_type list>     import_maybe
%type <Classtypes.ccsl_importing_type>		theory

%type <Classtypes.ccsl_ast option * token_type * Classtypes.ccsl_parameter_type list>
						typedefstart 

%type <Classtypes.ccsl_ast option * token_type * Classtypes.ccsl_member_sort * Classtypes.ccsl_parameter_type list>
						termdefstart

%type <token_type * Classtypes.ccsl_member_sort> termdefid

%type <Classtypes.ccsl_input_types option>	typedefeq

%type <(token_type * Classtypes.ccsl_input_types) list>
		typedvarlist varlist_maybe vardeclaration vardeclarations

%type <Classtypes.ccsl_assertion> 		formula
%type <Classtypes.ccsl_assertion option>	hol_formula_maybe

%type <(string * Classtypes.ccsl_input_types) list * Top_variant_types.location_type>
   					all_quant ex_quant lambda_quant

%type <Classtypes.ccsl_formulas>	 	hol_formula

%type <token_type option * Classtypes.ccsl_argument_type list option * location_type option * token_type>
						qualifiedid

%type <token_type>				pareninfix idorinfix

%type <Classtypes.ccsl_expressions>   	hol_base 

%type <Classtypes.ccsl_expressions * Classtypes.ccsl_expressions> update

%type <(Classtypes.ccsl_expressions * Classtypes.ccsl_expressions) list> 
						updatelist

%type <Classtypes.ccsl_expressions>   	type_annotation

%type <Classtypes.ccsl_expressions list>   hollist

%type <Classtypes.ccsl_input_types> instclass_maybe

%type <(Classtypes.ccsl_member_type container* Classtypes.ccsl_identifier_record_type list * Classtypes.ccsl_expressions) list> 
						caselist

%type <token_type * Top_variant_types.token_type list>
						casepattern

%type <Classtypes.ccsl_identifier_record_type * Classtypes.ccsl_identifier_record_type list>
						assertionsection assertionstart

%type <Top_variant_types.token_type * Classtypes.ccsl_input_types> 
						assertionselfvar

%type <Classtypes.ccsl_identifier_record_type list>
						creationsection creationstart
						theoremsection theoremstart

%type <Classtypes.ccsl_renaming list * location_type option>
					renamelist renaming

%type <Classtypes.ccsl_identifier_record_type * Classtypes.ccsl_input_types option * Classtypes.ccsl_expressions> 
					binding
%type <(Classtypes.ccsl_identifier_record_type * Classtypes.ccsl_input_types option * Classtypes.ccsl_expressions) list> 
					bindinglist

%type <Top_variant_types.location_type>		classstart datastart adtstart
%type <Top_variant_types.location_type>		signaturestart

%type <unit option> 			finalorloose

%type <unit> classbody classsection inheritsection methodsection 
%type <unit> attributesection classconstructorsection classconstructor
%type <unit> ancestor requestsection definitionsection
%type <unit> semicolon_maybe visibility colonordot semicolonorcomma 
%type <unit> semicolonorcomma_maybe typekeyword
%type <unit> adtbody adtconstructor adtconstructorlist
%type <unit> adtfield groundtermkeyword
%type <unit> signaturebody signaturesection  
%type <unit> signaturesymbolsection signaturetype signaturesymbol

%start file

%%

file:
    declarationlist EOF		{ d "File 1"; 
				  close_anon_ground_sig();
				  List.fold_left
				    (fun l astopt -> 
				       match astopt with
					 | Some x -> x :: l
					 | None -> l
				    )
				    [] $1
				}
;

declarationlist:
    /* empty */	         	{ d "declarationlist 1"; [] }
  | declarationlist declaration	{ d "declarationlist 2"; $2 :: $1 }
;


declaration:
    classspec			{ d "declaration class"; 
                                  add_class ($1)
				}
  | adtspec			{ d "declaration adt"; 
                                  add_adt $1 
				}
  | groundsignature		{ d "declaration sig";
				  add_sig $1
				}
  | typedef			{ d "declaration type"; $1 }
  | groundtermdef		{ d "declaration term"; $1 }
;


datastart:
    BEGIN ID 		{ d "Iface start";
			  do_iface_start $2;
			  $1
			}
;

classspec:
    classstart import_maybe classbody END ID
                           { d "Classdef";
			     do_class_end $1 $2 $5
			   }
;

classstart:
    datastart typeparameters COLON finalorloose CLASSSPEC 
			{ d "Classstart"; 
			  unique_type_parameter_check $2;
			  let ifa = ciface() in
			    ifa#become_class;
			    (match $4 with
			       | None -> ()
			       | Some _ -> 
				   ifa#put_feature FinalSemanticsFeature);
			    current_iface := Parse_Class( ifa );
			    $1
                        }
;

finalorloose:
    /* empty */         { None }
  | FINAL		{ Some () }
;

typeparameters:				/* true order */
    /* empty */		{ d "no typeparameters";
			  []
			}
  | OBRACKET paramdeclarations CBRACKET
  			{ d "typeparameters";
			  $2
			}
;

paramdeclarations:			/* true order */
    paramdeclaration				{ d "paramdecs 1";
						  $1
						}
  | paramdeclarations COMMA paramdeclaration	{ d "paramdecs 2";
						  $1 @ $3
						}
;

paramdeclaration:			/* true order */
    idlist COLON optvariance TYPE
				{ d "paramdec 1"; 
				  List.map (add_type_parameter $3)
				    (List.rev $1)
				}
;

optvariance:
    /* empty */ 		{ Unset }
  | POS 			{ Pos }
  | NEG 			{ Neg }
  | MIXED 			{ Mixed }
  | OPAREN numberorquestion COMMA numberorquestion CPAREN 
	 			{ do_exact_variance 
				    (new_loc $1 $5)
				    $2 $4 }
;

numberorquestion:
    QUESTIONMARK		{ -1 }
  | VALUE			{ int_of_string (fst $1) }
;



classbody:
    classsection		{ d "body 1"; () }
  | classbody classsection	{ d "Body 2"; () }
;

classsection:
    inheritsection		{ d "inherit field"; () }
  | visibility attributesection semicolon_maybe	
				{ d "attribute field"; 
				  current_visibility := NoVisibility }
  | visibility methodsection semicolon_maybe	
      				{ d "method field";
				  current_visibility := NoVisibility }
  | definitionsection		{ d "definitions"; () }
  | classconstructorsection semicolon_maybe
				{ d "classconstructor field"; () }
  | assertionsection		{ d "assertion field"; 
				}
  | creationsection		{ d "creation field"; 
				}
  | theoremsection              { d "theorem field";
				}
  | requestsection semicolon_maybe
                                { d "type request"; ()}
;

visibility:
    /* empty */		{ current_visibility := Public }
  | PUBLIC		{ current_visibility := Public }
  | PRIVATE		{ current_visibility := Private }
;

inheritsection:
    INHERIT FROM ancestor		{ () }
  | inheritsection COMMA ancestor	{ () }
;

ancestor:
    ID renaming 		{ (ccl())#add_ancestor  
				    (do_inherit $1 [] $2)   }
  | ID OBRACKET arglist CBRACKET renaming 
  				{ (ccl())#add_ancestor 
				    (do_inherit $1 (List.rev $3) $5)  }
;

renaming:				/* true order */
    /* empty */			{ ([], None) }
  | RENAMING renamelist 	{ d "Renaming"; 
				  (List.rev( fst $2 ),
				   snd $2)
				}
;

renamelist:				/* wrong order */
    ID AS ID			{ d "renamelist 1"; 
				  ([($1, None, $3)],
				   $3.loc)
				}
  | renamelist AND ID AS ID   	
				{ d "renamelist 2"; 
				  (($3, None, $5) :: (fst $1),
				   $5.loc)
				}
;

attributesection:
    ATTRIBUTE ID COLON pvstype ARROW pvstype
				{ d "attribute 1";
				  (do_attribute $2 $4 $6)
				}
  | attributesection SEMICOLON ID COLON pvstype ARROW pvstype
				{ d "attribute 2";
				  (do_attribute $3 $5 $7)
				}
;

methodsection:
    METHOD ID COLON pvstype ARROW pvstype
  			{ d "Method 1";
			  do_method $2 $4 $6 
			}
  | methodsection SEMICOLON ID COLON pvstype ARROW pvstype
  			{ d "Method 2";
			  do_method $3 $5 $7
			}
;

definitionsection:
    DEFINING ID COLON pvstype ARROW pvstype formula
			{ d "def 1";
			  do_definition $2 $4 $6 $7
			}
  | definitionsection ID COLON pvstype ARROW pvstype formula
			{ d "def 2";
			  do_definition $2 $4 $6 $7
			}
;


classconstructorsection:
    CONSTRUCTOR classconstructor	{ () }
  | classconstructorsection SEMICOLON classconstructor
				{ () }
;

classconstructor:
    ID COLON pvstype		{ d "const classconstructor";
				  (do_const_class_constructor $1 $3)  }
  | ID COLON pvstype ARROW pvstype
				{ d "var classconstructor";
				  (do_var_class_constructor $1 $3 $5) }
;

assertionsection:
    assertionstart ID COLON formula
			{ d "assertion first";
			  do_assertion $2 $1 $4;
			  $1
			}
  | assertionsection ID COLON formula
			{ d "assertion ff";
			  do_assertion $2 $1 $4;
			  $1
			}
;

assertionstart:
    ASSERTION import_maybe assertionselfvar varlist_maybe
			{
			  d "assertion start";
			  do_assertion_start $2 $3 $4;
			}
;

assertionselfvar:
    /* empty */		{ d "empty selfvar";
			  do_selfvar_default()
			}
  | SELFVAR ID COLON SELF 
			{ 
			  d "selfvar";
			  ($2, Self)
			}
;

varlist_maybe:				/* true order */
    /* empty */		{ [] }
  | varlist_maybe VAR vardeclarations 
			{ $1 @ $3 }
;

creationsection:
    creationstart ID COLON formula
			{ 
			  d "creation first";
			  (* creation and assertion imports are unified *)
			  do_creation $2 $1 $4;
			  $1
			}
  | creationsection ID COLON formula
			{ 
			  d "creation ff";
			  do_creation $2 $1 $4;
			  $1
			}
;

creationstart:
    CREATION import_maybe varlist_maybe
			{ 
			  d "creation start";
			  do_creation_start $1 $2 $3
			}
;

theoremsection:
    theoremstart ID COLON formula
			{ 
			  d "theorem first";
			  (* creation and assertion imports are unified *)
			  do_theorem $2 $1 $4;
			  $1
			}
  | theoremsection ID COLON formula
			{ 
			  d "theorem ff";
			  do_theorem $2 $1 $4;
			  $1
			}
;

theoremstart:
    THEOREM import_maybe varlist_maybe
			{ 
			  d "theorem start";
			  do_theorem_start $2 $3
			}
;

requestsection:
    REQUEST ID COLON pvstype  { d "Request 1";
				  (ccl())#add_rel_lifting $2.token_name $4 }
  | requestsection SEMICOLON ID COLON pvstype
                                { d "Request 2";
				  (ccl())#add_rel_lifting $3.token_name $5 }
;

all_quant:
    FORALL OPAREN typedvarlist CPAREN
			{ d "all_quant";
			  ((List.map (fun (tok,typ) -> 
				      tok.token_name,typ)
			     $3), 
			   (new_loc $1 $4))
			}
;

ex_quant:
    EXISTS OPAREN typedvarlist CPAREN  
  			{ d "ex-quant";
			  ((List.map (fun (tok,typ) -> 
				      tok.token_name,typ)
			     $3), 
			   (new_loc $1 $4))
			}
;


lambda_quant:
    LAMBDA OPAREN typedvarlist CPAREN 
			{ d "Lambda-quant";
			  ((List.map (fun (tok,typ) -> 
				      tok.token_name,typ)
			     $3), 
			   (new_loc $1 $4))
			}
;


hol_formula:
    all_quant colonordot hol_formula %prec QUANTOR
			{ d "Forall body";
			  FormLoc(Forall((fst $1), $3),
				  new_loc (snd $1) (get_form_loc $3))
			}
  | ex_quant colonordot hol_formula %prec QUANTOR
      			{ d "Exists body";
			  FormLoc(Exists((fst $1), $3),
				  new_loc (snd $1) (get_form_loc $3))
			}
  | lambda_quant colonordot hol_formula %prec QUANTOR
			{ d "Lambda body";
			  FormLoc(Formula(Abstraction((fst $1), (exff $3))),
				  new_loc (snd $1) (get_form_loc $3))
			}
  | LET bindinglist semicolonorcomma_maybe IN hol_formula %prec QUANTOR
      			{ d "Let";
			  FormLoc(Formula( Let( List.rev $2 , exff $5) ),
				  new_loc $1 (get_form_loc $5))
			}			    
  | hol_formula IFF hol_formula
			{ d "Iff";
			  FormLoc(Iff($1, $3),
				  new_loc (get_form_loc $1) (get_form_loc $3))
			}
  | hol_formula IMPLIES hol_formula
			{ d "Implies";
			  FormLoc(Implies($1, $3),
				  new_loc (get_form_loc $1) (get_form_loc $3))
			}
  | hol_formula OR hol_formula
			{ d "Or";
			  do_or $1 $3
			}
  | hol_formula AND hol_formula
			{ d "AND";
			  do_and $1 $3
			}
  | IF hol_formula THEN hol_formula ELSE hol_formula 
      			{ d "If";
			  FormLoc(Formula(If([$2,exff $4 ],exff $6)),
				  new_loc $1 (get_form_loc $6))
			} 
  | NOT hol_formula 
			{ d "Not";
			  FormLoc(Not($2),
				  new_loc $1 (get_form_loc $2))
			}
  | hol_formula OBSEQ hol_formula
                        { d "ObsEq";
			  FormLoc(Obseq(None, exff $1, exff $3),
				  new_loc (get_form_loc $1) (get_form_loc $3))
			}

  | hol_formula EQUAL hol_formula
      			{ d "Equality";
			  FormLoc(Equal(exff $1, exff $3),
				  new_loc (get_form_loc $1) (get_form_loc $3))
			}

  | hol_formula INFIX_REL hol_formula
      			{ d "Infix_Rel";
			    FormLoc(
			      Formula(
				InfixApplication((exff $1), NoIface, 
						 Unresolved $2, 
						 (exff $3))),
			      new_loc (get_form_loc $1) (get_form_loc $3))
			}

  | hol_formula INFIX_SHARP hol_formula
      			{ d "Infix_Sharp";
			    FormLoc(
			      Formula(
				InfixApplication((exff $1), NoIface, 
						 Unresolved $2, 
						 (exff $3))),
			      new_loc (get_form_loc $1) (get_form_loc $3))
			}
  | hol_formula INFIX_ADD hol_formula
      			{ d "Infix_Add";
			    FormLoc(
			      Formula(
				InfixApplication((exff $1), NoIface, 
						 Unresolved $2, 
						 (exff $3))),
			      new_loc (get_form_loc $1) (get_form_loc $3))
			}
  | hol_formula INFIX_MUL hol_formula
      			{ d "Infix_Mul";
			    FormLoc(
			      Formula(
				InfixApplication((exff $1), NoIface, 
						 Unresolved $2, 
						 (exff $3))),
			      new_loc (get_form_loc $1) (get_form_loc $3))
			}
  | hol_formula INFIX_EXP hol_formula
      			{ d "Infix_Exp";
			    FormLoc(
			      Formula(
				InfixApplication((exff $1), NoIface, 
						 Unresolved $2, 
						 (exff $3))),
			      new_loc (get_form_loc $1) (get_form_loc $3))
			}

  | ALWAYS hol_formula FOR instclass_maybe OBRACE idlist CBRACE 
			{ d "Always";
			  FormLoc(Formula(Box($4, (exff $2), $6)),
				  new_loc $1 $7)
			}
  | EVENTUALLY hol_formula FOR instclass_maybe OBRACE idlist CBRACE
			{ d "Eventually";
			  FormLoc(Formula(Diamond($4, (exff $2), $6)),
				  new_loc $1 $7)
			} 
  | CASES hol_formula OF caselist semicolonorcomma_maybe ENDCASES
      			{ d "Cases";
			  FormLoc(Formula(CCSL_Case(exff $2, (List.rev $4))),
				  new_loc $1 $6)
			}  
  | hol_formula WITH OBRACKET updatelist CBRACKET 
                        { d "function update";
			  FormLoc(Formula(FunUpdate(exff $1, List.rev $4)),
				  new_loc (get_form_loc $1) $5)
			}
  | hol_formula DOT qualifiedid
		 	{ d "method";
			  let f = do_method_selection (exff $1) $3 in
			  let _,_,_,lasttok = $3 
			  in 
			    FormLoc(f, new_loc (get_form_loc $1) 
				      (remove_option lasttok.loc))
			}
  | hol_formula hol_formula %prec APPLICATION
			{ d "Application";
			  FormLoc(Formula(Application(exff $1, exff $2)),
				  new_loc (get_form_loc $1) (get_form_loc $2))
			}
  | hol_base		{ d "base->fun";
			  ffex $1
			}
;

hol_base:
    TRUE 		{ d "True"; 
			  ExprLoc(Expression(True), $1)
			}
  | FALSE 		{ d "False";
			  ExprLoc(Expression(False), $1)
			}
  | PROJ_N 		{ d ("Proj_"^(string_of_int (fst $1))); 
			  ExprLoc(Projection (fst $1, -1), 
				  snd $1)
			}
  | VALUE		{ d "Value"; 
			  ExprLoc(TypedTerm(Term( (fst $1), Always,[]),
					    ground_type_nat (snd $1)),
				  snd $1)
			}
  | qualifiedid		{ d "qid";
			  let loc = match $1 with
			    | (None,None,None,id) -> remove_option id.loc
			    | (_, _,Some argloc, id2) -> 
				new_loc argloc (remove_option id2.loc)
			    | _ -> assert(false)
			  in
			    ExprLoc(do_qualified_term_id $1, loc)
			}
  | type_annotation	{ d "typeannot"; $1}
  | OPAREN hollist CPAREN
			{ d "Tuple"; 
			  match $2 with
			    | [ ExprLoc(ex,loc) ] -> ExprLoc(ex, 
							     new_loc $1 $3)
			    | [ _ ] -> assert(false)
			    | l -> ExprLoc(Tuple (List.rev $2), 
					   new_loc $1 $3)
			}
;


colonordot:
    COLON		{ () }
  | DOT			{ () }
;


vardeclarations:			/* true order */
    vardeclaration 
				{ $1 }
  | vardeclarations SEMICOLON vardeclaration  
				{ $1 @ $3 }
;

vardeclaration:				/* true order */
    idlist COLON pvstype	{ d "vardeclaration";
				  List.map
				    (fun id -> (id, $3))
				    (List.rev $1)
				}
;

qualifiedid:
    idorinfix		{ d "ID";
			  (None,None,None,$1)
			}
  | ID DOUBLECOLON idorinfix		
			{ d "qualified id";
			  (Some $1, None, Some (remove_option $1.loc), $3)
			}
  | ID OBRACKET arglist CBRACKET DOUBLECOLON idorinfix
		      	{ d "qualified_id";
			  (Some $1, Some (List.rev $3), 
			   Some (new_loc (remove_option $1.loc) $4), $6)
			}
;

idorinfix:
    pareninfix			{ $1 }
  | ID				{ $1 }
;

pareninfix:
    OPAREN INFIX_EXP CPAREN	{ $2 }
  | OPAREN INFIX_MUL CPAREN	{ $2 }
  | OPAREN INFIX_ADD CPAREN	{ $2 }
  | OPAREN INFIX_SHARP CPAREN	{ $2 }
  | OPAREN INFIX_REL CPAREN	{ $2 }
;

bindinglist:				/* wrong order */
    binding			{ d "first binding";
				  [$1]
				}
  | bindinglist semicolonorcomma binding
      				{ d "next binding";
				  $3 :: $1
				}
;

binding:
    ID EQUAL hol_formula	{ d "bind";
				  (do_id_declaration ($1,Product[]), 
				   None, exff $3)
				}
  | ID COLON pvstype EQUAL hol_formula
				{ d "bind w type";
				  (do_id_declaration ($1,Product[]), 
				   Some $3, exff $5)
				}
;

semicolonorcomma:
    COMMA		{ () }
  | SEMICOLON		{ () }
;

type_annotation:
    OPAREN hol_formula COLON pvstype CPAREN
       			{ d "Annotation";
			  ExprLoc(TypedTerm(exff $2, $4),
				  new_loc $1 $5)
			}
;
     
hollist:				/* wrong order */
    hol_formula
			{ [exff $1] 
			}
  | hollist COMMA hol_formula
                        { (exff $3) :: $1
			}
;

caselist:				/* wrong order */
    casepattern COLON hol_formula
  			{ d "firstpattern";
			  [ do_case_pattern $1 (exff $3) ]
			}
  | caselist semicolonorcomma casepattern COLON hol_formula
  			{ d "restpattern";
			  (do_case_pattern $3 (exff $5)) :: $1
			}
;

casepattern:				/* true order */
    ID			{ d "Pattern";
			  ($1, [])
			}
  | ID OPAREN idlist CPAREN
      			{ d "Pattern";
			  ($1, (List.rev $3))
			}
;

updatelist:				/* wrong order */
    update		{ d "updatelist 1";
			  [$1]
			}
  | updatelist COMMA update
                        { d "updatelist 2";
			  $3::$1
			}
;

update:
    hol_formula ASSIGN hol_formula
                        { d "single update";
			  (exff $1, exff $3)
			}
;

instclass_maybe:
    /* empty */   	{ d "Self"; Self }
  | ID DOUBLECOLON	{ d "other class 1";
			  do_instclass $1 []
			}
  | ID OBRACKET arglist CBRACKET DOUBLECOLON
			{ 
			  d "other class 2";
			  do_instclass $1 (List.rev $3)
			}
;

hol_formula_maybe:
    /* empty */				{ None }
  | hol_formula	 			{ Some(Symbolic($1)) }
;

/* hol_formula_maybe_semicolon:
 *      empty					{ None }
 *   | hol_formula semicolon_maybe		{ Some(Symbolic($1)) }
 * ;
 */

formula:
    PVS_FORMULA				{ $1 }
  | hol_formula SEMICOLON		{ Symbolic($1) }
;

typedvarlist:				/* true order */
    vardeclaration {$1}
  | typedvarlist COMMA vardeclaration {$1 @ $3}
;

adtspec:
    adtstart adtbody END ID
			{ d "Adtdef";
			  do_adt_end $1 $4
			}
;

adtstart:
    datastart typeparameters COLON ADT	
			{ d "Adtstart";
			  unique_type_parameter_check $2;
			  let ifa = ciface() in
			    ifa#become_adt;
			    current_iface := Parse_Adt( ifa );
			    $1
                        }
;

adtbody:
    adtfield		{ d "adtbody 1"; () }
  | adtbody adtfield    { d "adtbody 2"; () }
;

adtfield:
    adtconstructorlist semicolon_maybe
  				{ d "adtconstructorlist"; () }
;

adtconstructorlist:
    CONSTRUCTOR adtconstructor	{ () }
  | adtconstructorlist SEMICOLON adtconstructor { () }
;

adtconstructor:
    ID adtaccessors COLON pvstype		
  				{ d "const adtconstructor";
				  do_const_adt_constructor $1 $2 $4 
				}
  | ID adtaccessors COLON pvstype ARROW pvstype
				{ d "var adtconstructor";
				  do_var_adt_constructor $1 $2 $4 $6 }
;

adtaccessors:
    /* empty */			{ [] }
  | OPAREN idlist CPAREN	{ List.rev($2) }
;


groundsignature:
    signaturestart import_maybe signaturebody END ID
			{ d "Sigdef";
			  do_sig_end $1 $2 $5
			}
;


signaturestart:
    datastart typeparameters COLON GROUNDSIGNATURE
			{ d "Sigstart";
			  unique_type_parameter_check $2;
			  let ifa = ciface() in
			    ifa#become_sig;
			    current_iface := Parse_Sig( ifa );
			    $1
                        }
;

signaturebody:
    signaturesection			{ d "sigbody 1"; () }
  | signaturebody signaturesection	{ d "sigbody 2"; () }
;

signaturesection:
    signaturetype
  				{ d "signaturetype"; () }
  | signaturesymbolsection semicolon_maybe
  				{ d "signaturesymbolsection"; () }
;


signaturetype:
    typedef	 		{ 
				  assert($1 = None)
				}
;

typedef:
    typedefstart typedefeq	{ let (sigopt,token,parameters) = $1
				  in
				    add_groundtype token parameters $2;
				    sigopt
				}
;


typedefeq:
    /* empty */			{ None }
  | EQUAL pvstype		{ Some $2 }
;


typedefstart:
    typekeyword ID typeparameters
				{ 
				  let sigopt = start_anon_ground_sig $2 $3 
				  in
				   (sigopt, $2, $3)
				}
;

typekeyword:
    GROUNDTYPE			{ }
  | TYPE			{ }
;

signaturesymbolsection:
    CONSTANT signaturesymbol	{ () }
  | signaturesymbolsection SEMICOLON signaturesymbol  { () }
;

signaturesymbol:
    termdef			{ assert($1 = None) }
;

groundtermdef:
    groundtermkeyword termdef semicolon_maybe
				{ $2 }
;

groundtermkeyword:
    CONSTANT			{ }
  | GROUNDTERM			{ }
;


termdef:
    termdefstart COLON pvstype hol_formula_maybe
				{ let (sigopt,token,sort,parameters) = $1
				  in
				    add_groundterm token sort parameters $3 $4;
				    sigopt
				}
;

termdefstart:
    termdefid typeparameters	{ let (id, sort) = $1 in
				  let sigopt = start_anon_ground_sig id $2
				  in
				    (sigopt, id, sort, $2)
				}
;

termdefid:
    ID				{ ($1, GroundTerm) }
  | pareninfix 			{ ($1, InfixGroundTerm) }
;

/* 
 * 
 * 
 *     ID COLON pvstype hol_formula_maybe
 * 				   { d "signaturesymbol";
 * 				     do_sig_const $1 $3 $4;
 * 				   }
 *   | pareninfix COLON pvstype hol_formula_maybe
 * 				   { d "infix signaturesymbol";
 * 				     do_sig_infix_const $1 $3 $4
 * 				   }
 * ;
 */

pvstype:
    SELF		{ d "pvstype self";
			  self_action $1
			}
  | CARRIER		{ d "pvstype carrier";
			  carrier_action $1
			}
  | BOOL		{ d "bool";
			  Bool
			}
  | OBRACKET pvstypelist ARROW pvstype CBRACKET
			{ d "pvstype fun";
			  match $2 with
			    | [ t ] -> Function( t, $4 ) 
			    | tl -> 
				Function( do_type_product (List.rev tl) 
					    (new_loc $1 $5), 
					  $4)
			}
  | OBRACKET pvstypelist CBRACKET	
			{ d "pvstype prod";
			  do_type_product (List.rev $2) (new_loc $1 $3)
			}
  | qualifiedid         { d "qid";
			  do_qualified_type_id $1
			}
  | ID OBRACKET arglist CBRACKET
			{ d "pvstype class";
			  do_type_appl $1 (List.rev $3)
			}
;

pvstypelist:
    pvstype				{ [ $1 ] }
  | pvstypelist COMMA pvstype	{ $3 :: $1 }
;

arglist:				/* wrong order */
    pvstype			{ [TypeArgument $1] }
  | arglist COMMA pvstype	{ (TypeArgument $3) :: $1 }
;


import_maybe:				/* wrong order */
    /* empty */				{ [] }
  | import_maybe IMPORTING theory
  				{ d "Import";  $3 :: $1 }
;

theory:
    ID                            { ($1.token_name,[]) }
  | ID OBRACKET arglist CBRACKET  { ($1.token_name, List.rev $3) }
;

idlist:					/* reversed order */
    ID				{ d "idlist first"; [ $1 ] }
  | idlist COMMA ID		{ d "idlist"; $3 :: $1 }
;

semicolon_maybe:
    /* empty */			{ d "[;]" }
  | SEMICOLON			{ d ";" }
;

semicolonorcomma_maybe:
    /* empty */			{ }
  | semicolonorcomma		{ }
;

				
%%

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

