(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 16.03.00 by Jan
 *
 * Time-stamp: <Monday 15 July 02 16:11:32 tews@ithif51>
 *
 * predicate and relation lifting
 *
 * $Id: morphism.ml,v 1.15 2002/07/18 13:43:23 tews Exp $
 *
 *)

open Util
open Top_variant_types
open Logic_util
open Names
open Types_util
open Name_space
open Classtypes

(************************************************************************
 *
 *
 * variance separation is a special substitution:
 * The substitution type contains two substitutes, one for
 * negative occurence (first) and one for positive ones.
 * The substitution is done depending on the variances. Eg, with
 * the substitution type
 *   [ (Self,  (Self_neg, Self_pos)) ]
 * the type
 *   [Self -> Self] -> Self
 * is transformed into
 *   [Self_pos -> Self_neg] -> Self_pos
 *
 * Arguments:
 *   subst		the substitution
 *   variance		current variance true : positive, false : negative
 *   typ		the type
 *
 *)

type ccsl_variance_separation_type =
  (ccsl_input_types * (ccsl_input_types * ccsl_input_types)) list


let rec pre_separate_variances (subst : ccsl_variance_separation_type)
  variance typ =
					   (* first try to substitute *)
  try
    let contra,co = Util.assoc eq_ccsl_types typ subst
    in if variance then co else contra
  with					(* now do the recursion *)
      Not_found ->
	let recurse = pre_separate_variances subst in
	let recurse_args = pre_separate_arguments subst in
	  match typ with
      	    | Groundtype(id,arglist) ->
		Groundtype (id, recurse_args variance arglist 
			      (get_ground_type_parameters id))
      	    | Function(dom, codom) ->
		Function(
		  (recurse (not variance) dom),
		  (recurse variance codom)
		)
      	    | Product(type_list) ->
		Product(
		  List.map (recurse variance) type_list
		)
      	    | Class(cl, args) ->
		Class(cl, recurse_args variance args cl#get_parameters)
      	    | Adt(adt, flag, args) ->
		Adt(adt,flag,recurse_args variance args adt#get_parameters)
      	    | BoundTypeVariable _ 	-> typ
      	    | Self 			-> typ
	    | Bool 			-> typ

			(* not in ccsl input types  or cannot occur here: *)
      	    | TypeConstant _
      	    | Carrier
      	    | Record _
      	    | Predtype _
	    | FreeTypeVariable _
      	    | IFace _
      	    | Array _
      	    | SmartFunction _ -> assert(false)

and pre_separate_arguments (subst : ccsl_variance_separation_type)
  variance argument_list parameter_list =
  List.fold_right (fun (typ,var) accu -> match var with
		    | Unused
		    | Pos -> 
			TypeArgument(
			  pre_separate_variances subst variance typ
			) :: accu
		    | Neg -> 
			TypeArgument(
			  pre_separate_variances subst (not variance) typ
			) :: accu
		    | Mixed ->
			(* do not doubble arguments, because there
			 * no variance separated types. 
			 * The caller is responsible that all parameters
			 * with mixed variance are constant (i.e. not 
			 * substituted) or substituted in the same way. 
			 * If this holds we can procceed here either 
			 * positively or negatively
			 *)
			TypeArgument(
			  pre_separate_variances subst variance typ
			) :: accu
		    | _ -> assert(false)
		  )
				(* produce a list of type * variance pairs *)
    (List.combine
       (types_from_arguments argument_list)
       (List.map (function TypeParameter id ->
		    Logic_util.make_simple id.id_variance)
	  parameter_list)
    )
    []


let substitute_variance_types subst typ = 
  pre_separate_variances subst true typ

let substitute_variance_argument_list subst argument_list parameter_list =
  if (List.length argument_list) <> (List.length parameter_list)
  then
		(* this happens for components with local type parameters *)
    pre_separate_arguments subst true argument_list 
      (take parameter_list (List.length argument_list))
  else
    pre_separate_arguments subst true argument_list parameter_list
  

(***********************************************************************
 *
 * action of the interface functor 
 *
 * The functor comes in as type expression. What are considered as 
 * the arguments of the functor depends on other parameters.
 * 
 * ccsl_action_type describes the arguments for the functor. For 
 * each argument of the functor two functions are given (as expressions),
 * the first one for contravariant occurences, the second one for 
 * covariant onces. The domain and the codomain of the action of the 
 * functor must be given as two substitutions that generate these 
 * types.
 * 
 * For example: consider H as a type expression that involves Self. 
 * Then the two substitutions
 *   domain_subst : [Self, (Self1,Self1)]
 *   codomain_subst : [Self, (Self1, Self2)] 
 * give H(Self1,Self1) and H(Self1,Self2). Let expr be an 
 * expression of type H(Self1,Self1). To get H(id, f) (expr) you 
 * specify
 *    actions : [Self, (None, Some f)]
 * 
 * Arguments
 *    name_space    	Name space
 *    id_crea_flag	if true identifiers are created before the 
 *    		      	type substitution, otherwise after
 *    actions		the arguments of the functor as an assoc list
 *    variance	current variance, true: positive, false : negative
 *    domain_subst	substitution for getting the domain
 *    codomain_subst	substitution for getting the codomain
 *    expression	the expression to which the action is applied
 *    typ		Type describing the functor
 *    
 * There is the following invariant:
 * 
 * `expression' is an expression of domain type, ie. typ[domain_subst]
 * 
 * the result is an expression of codomain type, ie. typ[codomain_subst]
 * corresponding to  typ[actions] (expression)
 *)

type ccsl_action_type =
    (ccsl_input_types * 
     (ccsl_expressions option * ccsl_expressions option)) list

let rec pre_functor_action name_space id_crea_flag
  (actions : ccsl_action_type)
  variance domain_subst codomain_subst expression typ =
					   (* first try to substitute *)
  (try
     let contra,co = Util.assoc eq_ccsl_types typ actions
     in
       match variance,contra,co with
	 | true, _, (Some f) -> Application(f, expression)
	 | false, (Some f), _ -> Application(f, expression)
	 | _ -> expression
   with					(* now do the recursion *)
     | Not_found ->
    	 match typ with
	   | Groundtype(id,[]) when not (is_type_def id) -> expression
	   | Class(_,[])
	   | Adt(_,_,[]) -> expression
	   | Groundtype(id,arglist) ->
	       if is_type_def id 
	       then
		 pre_functor_action name_space id_crea_flag actions
		   variance domain_subst codomain_subst expression
		   (expand_type_def eq_ccsl_types typ)
	       else
		 Application(
		   Map(typ, 
		       pre_functor_action_argument_list name_space id_crea_flag
			 actions variance
			 domain_subst codomain_subst arglist 
			 (get_ground_type_parameters id)
		      ),
		   expression)
	   | Class(cl, arglist) ->
	       Application(
		 Map(typ, 
		     pre_functor_action_argument_list name_space id_crea_flag
		       actions variance
		       domain_subst codomain_subst arglist cl#get_parameters
		    ),
		 expression)
	   | Adt(adt, flag, arglist) ->
	       Application(
		 Map(typ, 
		     pre_functor_action_argument_list name_space id_crea_flag
		       actions variance
		       domain_subst codomain_subst arglist adt#get_parameters
		    ),
		 expression)
	   | BoundTypeVariable _ 	-> expression
	   | Self 			-> expression
	   | Carrier 			-> expression
	   | Bool 			-> expression
	   | Function(dom, codom) ->
	       (* H(A,B)     =	H1(B,A) -- expression --> H2(A,B)
		*    |   	  /\                        |
		*    | H(f,g) 	  | H1(g,f)    	       	    | H2(f,g)
		*    \/	    	  |                         \/
		* H(C,D)     =	H1(D,C) -- ? result ? --> H2(C,D)
		*)
	       let sub_ns = sub_space name_space in
	       let subst_dom =		(* H1(D,C) *)
	    	 pre_separate_variances codomain_subst (not variance) dom
	       in let subst_dom_list = 
		   match subst_dom with
		     | Product tl -> tl
		     | t -> [t]
	       in let dom_arg_list =	(* x \in H1(D,C) *)
		   if id_crea_flag 
		   then
		     let decltypelist = 
		       match dom with
			 | Product tl -> tl
			 | t -> [t]
		     in
		       List.map2 (fun (n,_) t -> (n,t))
			 (create_ids sub_ns decltypelist)
			 subst_dom_list			 
		   else
		       create_ids sub_ns subst_dom_list
	       in
	       let dom_term = match subst_dom with
		 | Product tl -> 
		     Tuple(
		       List.map (fun (t,_) -> Term(t,Always,[]))
			     dom_arg_list)
		 | t -> 
		     Term(
		       fst(List.hd dom_arg_list), Always, [])
	       in

	       let dom_expr =		(* H1(g,f)(x) *)
		 pre_functor_action sub_ns id_crea_flag
		   actions (not variance) 
		   codomain_subst domain_subst
		   dom_term dom 
	       in
	       let codom_expr =		(* H2(f,g) ( expr (H1(g,f)(x)) ) *)
		 pre_functor_action sub_ns id_crea_flag
		   actions variance 
		   domain_subst codomain_subst
		   (Application(expression, dom_expr))
		   codom 
	       in
	    	 Abstraction(
		   dom_arg_list,
		   codom_expr
		 )
	   | Product(type_list) ->
	       let tuple_len = List.length type_list in
	       let do_i_pos typ (i,accu) = 
		 (i-1,
		  (pre_functor_action name_space id_crea_flag
		     actions variance
		     domain_subst codomain_subst
		     (Application(Projection(i, tuple_len), 
				  expression))
		     typ 
		  ) :: accu)
	       in let _, nexprs = 
		   List.fold_right do_i_pos
		     type_list
		     (List.length type_list, [])
	       in
		 Tuple nexprs

		  (* cannot occur here *)
	   | TypeConstant _
	   | Record _
	   | Predtype _
	   | FreeTypeVariable _
	   | IFace _
	   | Array _
	   | SmartFunction _ -> 
	       assert(false)
  )

            (* work through a whole argument list of an Adt/Class/Groundtype 
	     * taking variances into account
	     *)
and pre_functor_action_argument_list name_space id_crea_flag actions variance
  domain_subst codomain_subst argument_list parameter_list =

  List.fold_right (fun (typ,var) accu -> match var with
		    | Unused
		    | Pos -> 
			(pre_functor_action_argument
			   name_space id_crea_flag actions variance
			   domain_subst codomain_subst 
			   typ
			) :: accu
		    | Neg -> 
			(pre_functor_action_argument
			   name_space id_crea_flag actions (not variance)
			   codomain_subst domain_subst 
			   typ
			) :: accu
		    | Mixed ->
			(pre_functor_action_argument
			   name_space id_crea_flag actions (not variance)
			   codomain_subst domain_subst 
			   typ
			) :: 
			(pre_functor_action_argument
			   name_space id_crea_flag actions variance
			   domain_subst codomain_subst 
			   typ
			) :: accu
		    | _ -> assert(false)
		  )
				(* produce a list of type * variance pairs *)
    (List.combine
       (types_from_arguments argument_list)
       (List.map (function TypeParameter id ->
		    Logic_util.make_simple id.id_variance)
	  parameter_list)
    )
    []

and pre_functor_action_argument name_space id_crea_flag
  actions variance domain_subst 
  codomain_subst typ 
  =
  let sub_ns = sub_space name_space in
  let subst_dom =
    pre_separate_variances domain_subst variance typ
  in
  let subst_dom_list = 
    match subst_dom with
      | Product tl -> tl
      | t -> [t]
  in let dom_arg_list =	(* x \in H1(D,C) *)
      if id_crea_flag 
      then
	let decltypelist = 
	  match typ with
	    | Product tl -> tl
	    | t -> [t]
	in
	  List.map2 (fun (n,_) t -> (n,t))
	    (create_ids sub_ns decltypelist)
	    subst_dom_list			 
      else
	create_ids sub_ns subst_dom_list
  in
  let dom_term = match subst_dom with
    | Product tl -> 
	Tuple(
	  List.map (fun (t,_) -> Term(t,Always,[]))
		dom_arg_list)
    | t -> 
	Term(
	  fst(List.hd dom_arg_list), Always, [])
  in
  let expr =		(* H1(g,f)(x) *)
    pre_functor_action sub_ns id_crea_flag
      actions variance domain_subst codomain_subst
      dom_term typ
  in
    Abstraction(
      dom_arg_list,
      expr
    )

	    

let functor_action name_space id_crea_flag actions 
  variance domain_subst codomain_subst expression typ 
  =
  opt_expression(
    pre_functor_action name_space id_crea_flag actions 
		   variance domain_subst codomain_subst expression typ
  )


let functor_action_argument_list name_space id_crea_flag actions 
  variance domain_subst codomain_subst argument_list parameter_list 
  =
  List.map
    opt_expression
    (pre_functor_action_argument_list name_space id_crea_flag actions 
       variance domain_subst codomain_subst argument_list parameter_list)



(* the main part of the condition that has to hold for coalgebra
 * morphisms:
 *
 *    X ---c---->H(X,X)
 *    |               \
 *    |               H(X,f)
 *    |                 \
 *    |                 \/
 *    f                 H(X,Y)
 *    |                 /\
 *    |                 /
 *    |               H(f,Y)
 *    \/             /
 *    Y --d---->H(Y,Y)
 *
 *    generates the equation(s) for the commuting diagram above, with respect
 *    to a given list of methods. This is done by calling the REAL morphism
 *    action function after massaging the domain of the type of each function.
 *    For ancestors the ancestor morphism predicate is inserted. This
 *    is only correct, if the ancestor is instanciated with constant 
 *    types only. Therefore this function raises an assertion, if one of 
 *    the ancestor arguments is not constant.
 *
 * parameters
 *     morphism            -> the morphism between the state spaces of the
 *                            coalgebras that we check (as TERM)
 *     name_space          -> name space for name creation
 *     coalgebra1          -> the domain coalgebra   (as TERM)
 *     coalgebra2          -> the codomain coalgebra (as TERM)
 *     ancestor_list       -> list of ancestors, they are in the form of a
 *                            ('class * 'class top_pre_argument list ) list
 *     anc_morphism        -> Predicate for for checking ancestors
 *     method_list         -> the list of methods    (as 'mem list)
 *     selfvar             -> the variable of Self, to which we apply
 *                            the equation (as TERM)
 *     self_type_argument  -> the types of Self1 and Self2, as an argument, to
 *                            instanciate the ancestors right
 *     self1_typ           -> typesubst. for covariant Self
 *     self2_typ           -> typesubst. for cantravariant Self
 *
 *)

let coalgebra_morphism
  morphism
  name_space
    coalgebra1
    coalgebra2
    ancestor_list
    anc_morphism
    method_list
    self_var
    self_type_arguments
    self1_typ
    self2_typ
    =

  let do_ancestor (anc, anc_arguments) =
    assert(count_self_args anc_arguments = 0);
    Formula(
      Application(
      	SmartApplication(
	  Term(anc_morphism anc, Always, self_type_arguments @ anc_arguments),
	    [
	      Application(Term(super_access_method anc, Always, []),
			  coalgebra1);
	      Application(Term(super_access_method anc, Always, []),
			  coalgebra2)
	    ]
	),
      	morphism
      )
    )
  in
  let do_method m =
    let name  = m#get_name in
    let sub_ns = sub_space name_space in
    let left_action = [ (Self, (None, Some morphism)) ] in
    let right_action = [ (Self, (Some  morphism, None)) ] in
    let left_domain_subst = [ (Self, (self1_typ,self1_typ)) ] in
    let right_domain_subst = [ (Self, (self2_typ, self2_typ)) ] in
    let codomain_subst = [ (Self, (self1_typ, self2_typ)) ] in
    let left_expression = RecordSelection( m#get_name, coalgebra1) in
    let right_expression = RecordSelection( m#get_name, coalgebra2) in
      Equal(
	pre_functor_action sub_ns true left_action true left_domain_subst
	      codomain_subst left_expression m#get_full_type,
	pre_functor_action sub_ns true right_action true right_domain_subst
	      codomain_subst right_expression m#get_full_type
      )
  in
    opt_formula(
      And( (List.map do_ancestor ancestor_list) @
	   (List.map do_method method_list)
	 )
    )



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

