(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 17.6.99 by Jan
 *
 * Time-stamp: <Tuesday 16 July 02 10:15:44 tews@ithif51>
 *
 * Liftings and invariants
 *
 * $Id: methodinv_theory.ml,v 1.13 2002/07/18 13:43:22 tews Exp $
 *
 *)

open Util
open Global
open Top_variant_types
open Top_classtypes
open Name_space
open Names
open Top_names
open Classtypes
open Types_util
open Pvs_proof_util
open Lifting
open Theory_class;;

  
(***********************************************************************
 ***********************************************************************
 *
 * method identifier datatype
 *
 *)

class ['class_type, 'member_type] ccsl_pre_method_idtype_adt
  (cl : 'class_type) 
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)
    
    constraint 'class_type = ccsl_iface_type
    constraint 'member_type = ccsl_member_type

    inherit ccsl_method_inv_helper_class cl
      as helper

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

      (* This is a datatype declaration ! *)
    method kind = Datatype

    method get_name = ccsl_method_idtype_adt_name cl

    method get_parameters = []

(* ISAR
 *     method get_proofs = []
 *)

    initializer top_theory#override_file_name (ccsl_class_file_name cl)



    method private method_id_enum =
      let methods = self#get_lifting_methods
      in
        if (List.length methods) > 0 then
	  Datatypedecl(self#get_name, [],
		       List.map
			 (fun m -> (name_of_method_tag m, [], 
				    name_of_method_tag_recognizer m)
			 )
			 methods
		      )
        else
	  Datatypedecl(self#get_name, [],
		       [("no_method", [], "no_method?")]
		      )

    method make_body =
      [self#method_id_enum]
end


class ccsl_method_idtype_adt cl = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_method_idtype_adt cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * method liftings
 *
 *)

class ['class_type, 'member_type] ccsl_pre_method_lift_theory 
  (cl : 'class_type) 
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)
    
    constraint 'class_type = ccsl_iface_type
    constraint 'member_type = ccsl_member_type

    inherit ccsl_method_inv_helper_class cl
      as helper

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      ((if cl#has_constructors then
	  [name_of_algebra]
        else [])
       @
       [ name_of_coalgebra;
         name_of_method_class_pred cl;
         name_of_initially cl;
         name_of_method_invariance cl;
       ]
      )
      
    initializer name_of_list <- create_one_id ns self#list_type

    method get_name = ccsl_method_lift_theory_name cl

    method get_parameters = self#simple_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

(* 	  (* compute list of methods that need lifting *)
 *     val mutable lifting_methods = None
 * 
 *     method private get_lifting_methods =
 * 	 match lifting_methods with
 * 	   | Some ms -> ms
 * 	   | None -> 
 * 	       let methods = 
 * 		 List.filter (fun m -> m#needs_lifting && m#is_method) 
 * 		   cl#get_all_members
 * 	       in
 * 		 lifting_methods <- Some methods;
 * 		 methods
 * 
 *)
(*     method private enum_type_decl =
 * 	 let methods = self#get_lifting_methods
 * 	 in
 * 	   if (List.length methods) > 0 then
 * 	     Enumdecl(name_of_method_enum_type cl,
 * 		      List.map
 * 			(fun m -> name_of_method_tag m)
 * 			methods
 * 		     )
 * (* HENDRIK : this should not produce an anonymous type *)
 * 	   else
 * 	     Typevardecl(name_of_method_enum_type cl)
 * 	 
 *)
    method private pred_type_decl =
      Typedecl([self_parameter],name_of_method_pred_type cl,
               Function(
                 Function(Self,Bool),
                 Function(Self,Bool)
               )
              )

    (* DO NOT RELY ON SUPERCLASSES:
     *
     * Do not use the methodinv theory of one of the superclasses!
     * This would yield completely wrong liftings, if one
     * of the ancestors is instanciated with self
     *
     *     method private super_imports =
     * 	 List.map 
     * 	   (function
     * 	      | Resolved_renaming (_,args,_,iifa) -> 
     * 		  ccsl_method_lift_theory_name iifa, self_argument::args
     * 		    (* no other stuff *)
     * 	      | Unresolved_renaming _
     * 	      | Resolved_ancestor _ 
     * 	      | Unresolved_ancestor _ -> assert(false)
     * 	   )
     * 	   cl#get_ancestors
     *)

    method private component_imports = 
      List.fold_right
	(fun (v,comp,args) accu ->
	   match comp#get_kind with
	     | Spec_adt -> 
		 if comp#has_feature NeedsMapFeature
		 then (ccsl_adt_every_theory_name comp, args) :: accu
		 else accu
	     | Spec_class ->
		 (ccsl_greatest_invariance_theory_name comp, 
		  (comp#get_model_type_argument args) :: args) :: accu
	     | Spec_sig -> accu
	     | Spec_Spec -> assert(false)
	)
	cl#get_components
	[]
      
          
    method private do_method_liftings =
  (* Debugging support
      let name = cl#get_name 
      in
  *)
      List.map (class_method_pred_definition ns
		  (TypeConstant( 
		     (name_of_method_pred_type cl), 
		     Isabelle_only, [self_argument]))
		  name_of_coalgebra
		  self#coalgebra_type
               )
	 (* DO NOT RELY ON SUPERCLASSES (see above or below) *)
	self#get_lifting_methods

    method private do_class_lifting =
      let sub_ns = sub_space ns in
      let rec_type = IFace(cl, Isabelle_only, self#simple_arguments) in
      let p = create_one_id sub_ns (Predtype True) in
      let m = create_one_id sub_ns self#method_enum_type in 
      let l = create_one_id sub_ns self#list_type in
      let re = create_one_id sub_ns rec_type in
      let self_var = create_one_id sub_ns Self in
      let pred_form = ConstantPredicate(p) in
      let list_expr = Term(l,Always,[]) in
      let re_expr = Term(re,Always,[]) in
        Defn(
	  name_of_method_class_pred cl,
	  [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	  (Function(
	     self#list_type,
	     TypeConstant(name_of_method_pred_type cl, 
			  Isabelle_only, [self_argument]))),
	  Abstraction(
            [l,self#list_type],
            Abstraction(
              [p, Function(Self,Bool)],
	      Abstraction(
		[self_var, Self],
              	Expression(
               	  class_real_method_pred_lifting
                             cl
			     self#simple_arguments
                             sub_ns
                             list_expr
                             pred_form
                             coalgebra_term
                             m
                             self#method_enum_type
                             self_var
              	)))))  
      
    method private initially =
      let sub_ns = sub_space ns in
      let p = create_one_id sub_ns (Function(Self,Bool)) in
      let self_lifting = 
	(None,
	 Some(Expression(ConstantPredicate(p))))
      in
        if cl#has_constructors then
          [Defn(name_of_initially cl,
		[[Declared(name_of_algebra, self#algebra_type)]],
		Function(Function(Self,Bool), 
			 Bool),
		Abstraction(
		  [p, Function(Self, Bool)],
		  Expression(
		    predlift eq_types self_lifting sub_ns 
			       self#get_constructor_functor_type
			       (Term(name_of_algebra,Always,[])))))
	  ]
	else
	  (* do nothing if user didn't supply constructors *)
	  []
    
    method private method_invariance =
      let sub_ns = sub_space ns in
      let l = create_one_id sub_ns self#list_type in
      let p = create_one_id sub_ns (Predtype True) in
      let self_var = create_one_id sub_ns Self in
      let pred_ex = Term(p,Always,[]) in
        Defn(
          name_of_method_invariance cl,
          [[Declared(name_of_coalgebra, self#coalgebra_type)];
           [Declared(name_of_list, self#list_type)]],
          Function(Function(Self,Bool), Bool),
            Abstraction(
              [p,Function(Self,Bool)],
              Expression(
                Forall(
                  [self_var,Self],
                  Implies(
                    Formula(Application(pred_ex,
                                              Term(self_var,Always,[]))),
                    Formula(
                      Application(
                        Application(
                          Application(
                            Application(
			      Term(name_of_method_class_pred cl,Always,[]),
                              coalgebra_term),
                            self#list_term),
                          pred_ex),
                        Term(self_var,Always,[])
                      )))))))

    method private invariance =
      let sub_ns = sub_space ns in
      let p = create_one_id sub_ns (Function(Self,Bool)) in
      let self_var = create_one_id sub_ns Self in
      let pred_ex = Term(p,Always,[]) in
      let methods = self#get_lifting_methods
      in let full_list = 
	  List( List.map (fun m -> Term(name_of_method_tag m,Always,[]))
		  methods)
      in
        Defn(
          name_of_private_invariance cl,
          [[Declared(name_of_coalgebra, self#coalgebra_type)]],
          Function(Function(Self,Bool), Bool),
            Abstraction(
              [p,Function(Self,Bool)],
	      Application(
		Application(
		  Application(
		    Term(name_of_method_invariance cl,Always,[]),
		    coalgebra_term),
		  full_list),
		Term(p,Always,[]))))


    method private method_struct_invariance =
      let sub_ns = sub_space ns in 
      let (decl_list, coalgebra_from_tuple ) = 
	self#coalgebra_as_tuple sub_ns in
      let pred_type = Function(Self,Bool) in
      let p = create_one_id sub_ns pred_type 
      in
	Defn(name_of_private_struct_invariance cl,
	     [decl_list; [ Undeclared(p, pred_type) ]],
	     Bool,
	     Application(
	       Application(
		 Term(name_of_private_invariance cl, Always, []),
		 coalgebra_from_tuple
	       ),
	       Term(p,Always,[])
	     ))


    method private complete_list =
      let sub_ns = sub_space ns in
      let m = create_one_id sub_ns self#method_enum_type 
      in 
	Defn(
	  name_of_complete_inv_id_list cl,
          [[Declared(name_of_list, self#list_type)]],
	  Bool,
	  Expression(
	    Forall([m, self#method_enum_type],
		   Formula(
		     SmartApplication(
		       Term(name_of_list_member (), Always, []),
		       [Term(m,Always,[]);
			self#list_term]
		     )))))
       
                          
(* ISAR
 *     method get_proofs = []
 *)

    method make_body =
      [ Import(
          (ccsl_interface_theory_name cl, self#simple_arguments) ::
	  (ccsl_method_idtype_adt_name cl, []) ::
	  self#component_imports
        );
        (self#coalgebra_decl)
      ]
      @
      (if cl#has_constructors then
         [(self#algebra_decl)]
       else [])
      @
(* 	 [Comment("Enumeration type for methods");
 * 	  self#enum_type_decl;
 *)
      [Comment("Type of the lifting wrt. a single  method");
       self#pred_type_decl;
       Comment("Definitions of the liftings for methods")
      ]
      @
      self#do_method_liftings
      @
      [(self#do_class_lifting)
      ]
      @
      [(self#list_decl)]
      @
      self#initially
      @
      [
        self#method_invariance;
	self#invariance;
	self#method_struct_invariance;
	self#complete_list
      ]
        
  end (* ccsl_pre_method_lift_theory *)

class ccsl_method_lift_theory cl = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_method_lift_theory cl eq_ccsl_types

(***********************************************************************
 ***********************************************************************
 *
 * Rewrite lemmas for invariance
 *
 *)

class ['class_type, 'member_type] ccsl_pre_method_invariance_rewrite_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type
  = 
  object (self : 'self)

     inherit ccsl_method_inv_helper_class cl
       as helper
     inherit 
       ['class_type, 'member_type] ccsl_pre_class_theory_class 
       cl eq_types    as top_theory

        (* reserve all names, that we want to declare *)
     initializer reserve ns 
       ((if cl#has_constructors then
	   [name_of_algebra]
         else [])
        @
        [
	  name_of_coalgebra;
	  "P"; "Q";
          name_of_inv_true_lemma;
	  name_of_inv_and_lemma;
	  name_of_inv_list_lemma;
	  name_of_inv_linear_cons_lemma;
	  name_of_inv_linear_append_lemma;
	  name_of_method_id;
        ])

(*     initializer name_of_list <- create_one_id ns self#list_type
*)
     method get_parameters = self#simple_parameters
                               
     initializer top_theory#override_file_name (ccsl_class_file_name cl)
       
(* ISAR
 * 	method get_proofs = []
 *)
                           
     (* this gets overriden in derived classes to specialize
      * this class for private or public invariance
      *)
                           
     method private invariant_pred =
       Expression(ConstantPredicate(name_of_method_invariance cl))
         

       (* compute list of methods that get a rewrite lemma *)
    val mutable rewrite_methods = None

    method private get_rewrite_methods =
      match rewrite_methods with
	| Some ms -> ms
	| None -> 
	    let methods = 
	      List.filter (fun m -> m#is_action && m#needs_lifting) 
		cl#get_all_members
	    in
	      rewrite_methods <- Some methods;
	      methods


      (* 
       *   inv_add : LEMMA
       *     (FORALL ( x: Self , p1: Name , p2: Data) :
       *     (invariant?(c)(l)(P) AND list_member(add,l))
       *     IMPLIES
       * 	  ((P(x))) IMPLIES (every(P)(add(c)(x , p1 , p2))))
       *)

    method private do_method m =
      let inner_expr = (Application(
			 Term(m#get_name,Always,[]), 
			 coalgebra_term)) in
      let sub_ns = sub_space ns in
      let _ = make_flat sub_ns in
      let self_var = create_one_id sub_ns Self in
      let l = create_one_id sub_ns self#list_type in
      let self_lifting = 
	(None, 
	 Some(Expression(
		ConstantPredicate("P")
	      )))
      in
      let param_lifting = [] in
      let member_prem =
	if m#is_defined 
	then
	  Formula(
	    Application(
	      Term(name_of_complete_inv_id_list cl, Always,[]),
	      Term(l,Always,[])))
	else
	  Formula(
	    SmartApplication(
	      Term(name_of_list_member(), Always, []),
	      [
		Term(name_of_method_tag m, Always, []);
		Term(l,Always,[])]
	    ))
      in
      let prem = And([
		       Formula(
		      	 Application(
			   Application(
		      	     Application(self#invariant_pred,
					 coalgebra_term),
			     Term(l,Always,[])),
		      	   Term("P",Always,[]))
		       );
		       member_prem;
		       Formula(Application
				 (Expression(ConstantPredicate("P")),
				  Term(self_var,Always,[])))
		     ]) in
      let form = 
	Forall([(self_var,Self); (l, self#list_type)],
	       (Implies
		  (prem,
		   class_pred_lifting
		     eq_types
		     param_lifting
		     self_lifting		       
		     sub_ns 
		     coalgebra_term
		     m 
		     self_var 
		  )))
(* 
 * 		      predlift self_lifting
 * 		      sub_ns 
 * 		      (Function(m#get_domain,m#get_codomain))
 * 		      inner_expr 
 * 
 *)


      in
      let form2 = match !output_mode with
	| Pvs_mode -> make_pvs_rewrite_form form
	| Isa_mode -> make_isa_rewrite_form form
      in 
	Lemma(
	  name_of_inv_lemma m,
	  form2)

     (*************************************************** 
      * 
      * standard lemmas for truth and finite conjunction 
      *) 

     val p_pred = Expression(ConstantPredicate("P")) 
     val q_pred = Expression(ConstantPredicate("Q")) 

     method private pqlm_declaration = 
        [Vardecl("P", Function(Self,Bool)); 
         Vardecl("Q", Function(Self,Bool)); 
(*         (helper#list_decl);
*)	 (helper#method_decl)
        ] 

     (*   
      *  inv_true : LEMMA invariant?(c)(l)((LAMBDA(x: Self) : TRUE)) 
      *) 
     method private inv_true =  
       let sub_ns = sub_space ns in 
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in 
	 Proved(
	   Lemma(
             name_of_inv_true_lemma, 
	     Forall([l, self#list_type],
             Formula( 
               Application( 
		 Application( 
                   Application( 
                     self#invariant_pred, 
                     coalgebra_term), 
                   Term(l,Always,[]) 
		 ), 
		 Abstraction( 
                   [x,Self], 
                   Expression(True) 
		 ) 
               ) 
             )) 
           ),
	   PvsProof(
	     Anon_proof( make_simple_proof(
			 [ grind ])
		     ))
	 )   
	  
     (*   
      *   inv_and : LEMMA
      *     ((invariant?(c)(l)(P)) AND (invariant?(c)(l)(Q)))
      *     IMPLIES (invariant?(c)(l)((LAMBDA(x: Self) : (P(x)) AND (Q(x)))))
      *)
  
     method private inv_and = 
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
	 Lemma(
	   name_of_inv_and_lemma,
	   Forall([(l, self#list_type)],
	   Implies(
	     And(
	       [Formula(
		  Application(
		    Application(
		      Application(self#invariant_pred,coalgebra_term),
		      Term(l,Always,[])),
		    p_pred));
	       	Formula(
		  Application(
		    Application(
		      Application(self#invariant_pred,coalgebra_term),
		      Term(l,Always,[])),
		    q_pred))]),
	     Formula(
	       Application(
		 Application(
	      	   Application(self#invariant_pred,coalgebra_term),
		   Term(l,Always,[])),
	      	 Abstraction(
		   [x,Self],
		   Expression(
		     And(
		       [Formula(Application(p_pred,Term(x,Always,[])));
		       	Formula(Application(q_pred,Term(x,Always,[])))
		       ]))))))))
	   
      (* 
       *   inv_list : LEMMA
       *     FORALL ( z: list[[Self -> bool]]) :
       * 	 (every(invariant?(c))(z))
       * 	 IMPLIES
       * 	 (invariant?(c)
       * 	    (reduce
       * 	       ((LAMBDA(x: Self) : TRUE) ,
       * 		(LAMBDA(P: [Self -> bool] , Q: [Self -> bool]) :
       * 		  (LAMBDA(x: Self) : (P(x)) AND (Q(x)))))
       * 	       (z)))
       *)

     method private inv_list =
       match !output_mode with
	 | Pvs_mode -> 
	     let sub_ns = sub_space ns in
	     let l = create_one_id sub_ns self#list_type in
	     let x = create_one_id sub_ns Self in
	     let pred_list_type = Adt(self#list_adt, Always,
				 [TypeArgument(
				    Function(Self,Bool))]) in
	     let z = create_one_id sub_ns pred_list_type in
	       Lemma(
		 name_of_inv_list_lemma,
		 Forall(
		   [(z, pred_list_type); (l, self#list_type) ],
		   Implies(
		     Formula(
		       Application(
			 Application(
	      		   Term(name_of_adt_every self#list_adt,Always,[]),
			   Application(
			     Application(self#invariant_pred,coalgebra_term),
			     Term(l,Always,[]))),
			 Term(z,Always,[]))),
		     Formula(
		       Application(
			 Application(
			   Application(self#invariant_pred,coalgebra_term),
			   Term(l,Always,[])),
			 Application(
			   Application(
			     Term(name_of_adt_reduce self#list_adt, Always,[]),
			     Tuple(
		      	       [Abstraction([x,Self], Expression(True));
			      	Abstraction(
			    	  ["P",Function(Self,Bool);
				   "Q",Function(Self,Bool)],
			    	  Abstraction(
				    [x,Self],
				    Expression(
				      And(
				      	[Formula(Application(
						   p_pred,
						   Term(x,Always,[])));
					 Formula(Application(
						   q_pred,
						   Term(x,Always,[])))
				      	]))))
		      	       ])),
			   Term(z,Always,[])))))))
	 | Isa_mode -> 
	     Comment("Invarinats are closed under finite conjunction. \n"
		       ^ "To be implemented.")


     method private inv_linear_cons =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       Lemma(
	 name_of_inv_linear_cons_lemma,
	 Forall([l, self#list_type],
	 Implies(
	   Formula(
	     Application(
	       Application(
		 Application(self#invariant_pred,coalgebra_term),
		 Application(Term("cons",Always, []),
			     Tuple([self#method_term;Term(l,Always,[])]))
	       ),
	       p_pred
	     )
	   ),
	   Formula(
	     Application(
	       Application(
		 Application(self#invariant_pred,coalgebra_term),
		 Term(l,Always,[])),
	       p_pred
	     )
	   )
	 ))
       )

     method private inv_linear_append =
       let sub_ns = sub_space ns in 
       let (l1,l2) = create_id_pairs sub_ns [self#list_type] in
       let l1_term = Term(fst (List.hd l1), Always, []) in
       let l2_term = Term(fst (List.hd l2), Always, []) in
	 Lemma(
	   name_of_inv_linear_append_lemma,
	   Forall(
	     l1@l2, 
	     Iff(
	       And[
		 Formula(
		   Application(
		     Application(
		       Application(self#invariant_pred,coalgebra_term),
		       l1_term),
		     p_pred));
		 Formula(
		   Application(
		     Application(
		       Application(self#invariant_pred,coalgebra_term),
		       l2_term),
		     p_pred))
	       ],
	       Formula(
		 Application(
		   Application(
		     Application(self#invariant_pred,coalgebra_term),
		     Application(
		       Term("append",Always,[]),
		       Tuple[l1_term; l2_term])
		   ),
		   p_pred))
	     )))

	      
     method get_name = ccsl_method_invariance_rewrite_name cl
			 
     method make_body =
       [ 
         Import([ccsl_method_lift_theory_name cl, self#simple_arguments]); 
         self#coalgebra_decl; 
       ] 
       @ 
       (if cl#has_constructors then 
          [(self#algebra_decl)] 
        else []) 
       @ 
       self#pqlm_declaration 
       @  
       [
	 Comment("General results about invariants"); 
         self#inv_true;
         self#inv_and; 
         self#inv_list;
	 self#inv_linear_cons;
	 self#inv_linear_append
       ]  
       @
       Comment("Rewrite lemmas for methods"):: 
       (List.map self#do_method self#get_rewrite_methods) 

  end (* ccsl_pre_method_invariance_rewrite_theory *)

class ccsl_method_invariance_rewrite_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_method_invariance_rewrite_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * inherit methodwise invariance
 *
 *
 * this theory is reused below to generate the lemmas for inheriting Box
 *
 *)


class ['class_type, 'member_type] ccsl_pre_method_invariance_inherit_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  (* without type restriction -- we want to inherit from this one *)
  = 
  object (self : 'self)

     inherit 
       ['class_type, 'member_type] ccsl_pre_class_theory_class 
       cl eq_types    as top_theory

     method get_parameters = self#simple_parameters
                               
     initializer top_theory#override_file_name (ccsl_class_file_name cl)
       
     method get_name = ccsl_method_invariance_inherit_name cl
			 
(* ISAR
 * 	method get_proofs = []
 *)
                           
     (* this gets overriden in derived classes to specialize
      * this class for private or public invariance
      *)
                           
     method private invariant_pred use_this_cl =
       Expression(ConstantPredicate(name_of_method_invariance use_this_cl))
         

(* Apparently not needed
 *
 * 	method private super_imports = 
 * 	  List.map 
 * 	    (fun ianc -> 
 * 	       ccsl_basic_theory_name ianc, self_argument:: ianc#get_arguments)
 * 	    cl#get_resolved_ancestors
 * 	    
 *)

	(**********************************************************
	 *
	 * override section: the following methods are adopted 
	 * for the Box case
	 *)
	 
     method private describe = "methodwise invariants"

     method private lemma_name acl = name_of_inv_inherit_lemma acl

	 (* there is no x here, but one in Box *)
     method private get_x_decl_list = []

     method private this_class_pred l_term p_term = 
       Application(
	 Application(
	   Application(self#invariant_pred cl, coalgebra_term),
	   l_term),
	 p_term)       

     method private super_class_pred acl l_term p_term = 
       Application(
	 Application(
	   Application(
	     self#invariant_pred acl, 
	     Application(Term(super_access_method(acl),
			      Always,[]), 
			 coalgebra_term)),
	   l_term),
	 p_term)


	(*
	 * end override section
	 *
	 **************************************************************)


     method private do_inv_inherit acl amethods = 
       let sub_ns = sub_space ns in 
       let p_type = Function(Self, Bool) in
       let p = create_one_id sub_ns p_type in
       let p_pred = Term(p, Always,[]) in
       let mlist = 
	 List(List.map (fun m -> Term(name_of_method_tag m, Always, []))
		amethods)
       in let alist =			(* anticipate change here *)
	 List(List.map 
		(fun m -> 
		   if !output_mode = Isa_mode
		   then
		     QualifiedTerm(isar_theory_name acl, Always, [], 
				   name_of_method_tag_string m#last_name)
		   else
		     Term(name_of_method_tag_string m#last_name, Always, [])
		)
		amethods)
       in
	 Lemma(
	   self#lemma_name acl,
	   Forall(
	     [(p,p_type)] @ self#get_x_decl_list,
	     Iff(
	       Formula(
		 self#this_class_pred mlist p_pred),
	       Formula(
		 self#super_class_pred acl alist p_pred)
	     )))


     method private do_inv_inherit_maybe acl =
       if count_self_args acl#get_arguments <> 0
       then
	 [Comment 
	    ("No lemma for inherited " ^ self#describe ^ " from " ^
	     acl#get_name ^ ":\n" ^
	     "instanciation for " ^ acl#get_name ^ " is not constant")
	 ]
       else
	 let amethods = 
	   List.filter (fun m -> m#is_method && m#needs_lifting ) 
	     acl#get_all_members
	 in
	   if amethods = [] 
	   then
	     [Comment("No inherited methods from " ^ acl#get_name ^ 
		      "have a successor state")
	     ]
	   else
	     [Comment
		("Inherit " ^ self#describe ^ " from " ^ acl#get_name );
	      self#do_inv_inherit acl amethods
	     ]
	     
     method make_body =
       [ 
         Import([ccsl_method_lift_theory_name cl, self#simple_arguments]); 
         self#coalgebra_decl; 
       ] 
       @
       [Comment "Link methodwise invariance with ancestor classes"]
       @
       (List.flatten
	  (List.map
	     self#do_inv_inherit_maybe
	     cl#get_resolved_ancestors)
       )

  end (* ccsl_pre_method_invariance_rewrite_theory *)

class ccsl_method_invariance_inherit_theory cl = 
  ([ccsl_iface_type, ccsl_member_type] 
     ccsl_pre_method_invariance_inherit_theory cl eq_ccsl_types
     : [ccsl_iface_type, ccsl_member_type] ccsl_pre_class_theory_type)




(***********************************************************************
 ***********************************************************************
 *
 * Greatest Invariance 
 *
 *)

(* 
 * class ['class_type, 'member_type] ccsl_pre_greatest_invariance_theory
 *   (cl : 'class_type) 
 *   (* the equality relation for initializing the name space *)
 *   (eq_types : ('class_type, 'member_type) top_pre_types
 * 	-> ('class_type, 'member_type) top_pre_types -> bool)
 * 
 *   : ['class_type, 'member_type] ccsl_pre_class_theory_type
 *   = 
 *   object (self : 'self)
 * 
 * 	inherit ccsl_method_inv_helper_class cl
 * 	  as helper
 * 	inherit 
 * 	  ['class_type, 'member_type] ccsl_pre_class_theory_class 
 * 	  cl eq_types   as top_theory
 * 
 * 	   (* reserve all names, that we want to declare *)
 * 	initializer reserve ns 
 * 	  ((if cl#has_constructors then
 * 	      [name_of_algebra]
 * 	     else [])
 * 	   @
 * 	   [
 * 	     name_of_coalgebra;
 * 	     "P"; "Q";
 * 	     name_of_method_id;
 * 	     name_of_greatest_invariance cl;
 * 	     name_of_h_gi;
 * 	     name_of_monotonic;
 * 	     name_of_nonempty
 * 	   ])
 * 
 * 	initializer name_of_list <- create_one_id ns self#list_type
 * 
 * 	val p_pred = Expression(ConstantPredicate("P")) 
 * 	val q_pred = Expression(ConstantPredicate("Q")) 
 * 	val gfp = Term(name_of_gfp,Always,[])
 * 	val pred_type = Function(Self, Bool)
 * 
 * 	method private greatest_inv_appl = 
 * 	    Application(
 * 		Term(name_of_greatest_invariance cl, Always, []),
 * 		Tuple(
 * 		  [coalgebra_term;
 * 		   self#list_term;
 * 		   p_pred
 * 		  ]))	 
 * 
 * 	method private pqlm_declaration = 
 * 	   [Vardecl("P", pred_type); 
 * 	    Vardecl("Q", pred_type); 
 * 	    (helper#list_decl);
 * 	    (helper#method_decl)
 * 	   ] 
 * 
 * 
 * 	method private greatest_inv_def = 
 * 	  let sub_ns = sub_space ns in	 
 * 	  let q1 = create_one_id sub_ns pred_type in
 * 	  let q1_term = Term(q1, Always, []) in
 * 	  let gip_term = Term("gi_P", Always, []) in
 * 	  let method_inv_appl = 
 * 	    Application(
 * 	      Application(
 * 		Term(name_of_method_invariance cl, Always, []),
 * 		coalgebra_term),
 * 	      self#list_term) in
 * 	    Defn(
 * 	      name_of_greatest_invariance cl,
 * 	      [[Declared(name_of_coalgebra, self#coalgebra_type);
 * 		Declared(name_of_list, self#list_type);
 * 		Declared("P", pred_type)]],
 * 	      Function(
 * 		pred_type,
 * 		Bool),
 * 	      Abstraction(
 * 		[ "gi_P", pred_type],
 * 		Expression(
 * 		  And([
 * 			LessOrEqual( gip_term, p_pred);
 * 
 * 			Formula(
 * 			  Application(
 * 			    method_inv_appl,
 * 			    gip_term));
 * 
 * 			Forall(
 * 			  [q1, pred_type],
 * 			  Implies(
 * 			    And([
 * 				  LessOrEqual(q1_term,p_pred);
 * 
 * 				  Formula(
 * 				    Application(
 * 				      method_inv_appl,
 * 				      q1_term))]),
 * 			    LessOrEqual(q1_term, gip_term)
 * 			  )
 * 			)
 * 		      ])
 * 		)
 * 	      )
 * 	    )
 * 
 * 	method private greatest_inv_unique =
 * 	  let sub_ns = sub_space ns in
 * 	  let q1 = create_one_id sub_ns pred_type in
 * 	  let q2 = create_one_id sub_ns pred_type in
 * 	  let q_term = fun q -> Term(q, Always, []) in
 * 	    Lemma(
 * 	      name_of_greatest_inv_unique_lemma,
 * 	      Forall(
 * 		[q1, pred_type;
 * 		 q2, pred_type],
 * 		Implies(
 * 		  And([
 * 			Formula(
 * 			  Application(
 * 			    self#greatest_inv_appl,
 * 			    q_term(q1)));
 * 			Formula(
 * 			  Application(
 * 			    self#greatest_inv_appl,
 * 			    q_term(q2)))]
 * 		     ),
 * 		  Equal(q_term(q1), q_term(q2))
 * 		)
 * 	      )
 * 	    )
 * 
 * 	method private h_gi_def =
 * 	  let sub_ns = sub_space ns in
 * 	  let x = create_one_id sub_ns Self in
 * 	  let x_term = Term(x, Always, []) in
 * 	  let q = create_one_id sub_ns pred_type in
 * 	  let q_term = Term(q, Always, []) in
 * 	    Defn(
 * 	      name_of_h_gi,
 * 	      [[Declared(name_of_coalgebra, self#coalgebra_type);
 * 		Declared(name_of_list, self#list_type);
 * 		Declared("P", pred_type)]],
 * 	      Function(pred_type, pred_type),
 * 	      Abstraction(
 * 		[q, pred_type],
 * 		Abstraction(
 * 		  [x, Self],
 * 		  Expression(
 * 		    And([
 * 			  Formula(
 * 			    Application(p_pred, x_term));
 * 			  Formula(
 * 			    Application(
 * 			      Application(
 * 				Application(
 * 				  Application(
 * 				    Term(name_of_method_class_pred cl, Always,[]),
 * 				    coalgebra_term),
 * 				  self#list_term),
 * 				Term(q,Always,[])),
 * 			      x_term)
 * 			  )
 * 			])
 * 		  )
 * 		)
 * 	      )
 * 	    )
 * 
 * 	method private h_gi_monotonic =
 * 	  Lemma(
 * 	    name_of_h_gi_monotonic_lemma,
 * 	    Formula(
 * 	      Application(
 * 		Term(name_of_monotonic, Always, []),
 * 		Application(
 * 		  Term(name_of_h_gi, Always, []),
 * 		  Tuple([coalgebra_term;
 * 			 self#list_term;
 * 			 p_pred])
 * 		)
 * 	      )
 * 	    )
 * 	  )
 * 	  
 * 	method private greatest_inv_char =
 * 	  Lemma(
 * 	    name_of_greatest_inv_char_lemma,
 * 	    Formula(
 * 	      Application(
 * 		self#greatest_inv_appl,
 * 		Application(
 * 		  gfp,
 * 		  Application(
 * 		    Term(name_of_h_gi, Always, []),
 * 		    Tuple([coalgebra_term;
 * 			   self#list_term;
 * 			   p_pred])
 * 		  )
 * 		)
 * 	      )
 * 	    )
 * 	  )
 * 	  
 * 	method private greatest_inv_exist =
 * 	  Lemma(
 * 	    name_of_greatest_inv_exist_lemma,
 * 	    Formula(
 * 	      Application(
 * 		Term(name_of_nonempty, Always, []),
 * 		self#greatest_inv_appl)
 * 	    )
 * 	  )
 * 	method get_name = ccsl_greatest_invariance_theory_name cl
 * 
 * 	method get_parameters = self#simple_parameters
 * 				  
 * 	initializer top_theory#override_file_name (ccsl_box_file_name cl)
 * 
 * 	method get_proofs = []
 * 			    
 * 	method make_body =
 * 	  [ 
 * 	    Import([ccsl_method_invariance_rewrite_name cl,
 * 		    self#simple_arguments;
 * 		    "fixedpoints",[self_argument]]); 
 * 	    self#coalgebra_decl; 
 * 	  ] 
 * 	  @
 * 	  self#pqlm_declaration 
 * 	  @ 
 * 	  [
 * 	    Comment("Definition of greatest invariant: ");
 * 	    self#greatest_inv_def;
 * 	    Comment("Greatest invariants are unique: ");
 * 	    self#greatest_inv_unique;
 * 	    Comment("Define a helper function that constructs from a predicate");
 * 	    Comment("a predicate that is ''closer'' to the greatest invariant");
 * 	    self#h_gi_def;
 * 	    Comment("Helper lemma to solve occuring TCCs");
 * 	    self#h_gi_monotonic;
 * 	    Comment("Now we can specify how a greatest invariant is computed:");
 * 	    self#greatest_inv_char;
 * 	    Comment("Another helper lemma for TCCs");
 * 	    self#greatest_inv_exist;
 * 	  ]
 *   end (* ccsl_pre_greatest_invariance_theory *)
 * 
 * class ccsl_greatest_invariance_theory cl = 
 *   [ccsl_iface_type, ccsl_member_type] ccsl_pre_greatest_invariance_theory cl eq_ccsl_types
 * 
 *)



(***********************************************************************
 ***********************************************************************
 *
 * BOX
 *
 *)

class  ['class_type, 'member_type] ccsl_pre_box_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type 
  = 
  object (self : 'self)

     inherit ccsl_method_inv_helper_class cl
       as helper
     inherit 
       ['class_type, 'member_type] ccsl_pre_class_theory_class 
       cl eq_types   as top_theory

        (* reserve all names, that we want to declare *)
     initializer reserve ns 
       ((if cl#has_constructors then
	   [name_of_algebra]
          else [])
        @
        [
	  name_of_coalgebra;
	  "P"; "Q";
	  name_of_method_id;
	  name_of_greatest_invariance cl;
	  name_of_h_gi;
	  name_of_box cl;
	  name_of_diamond cl;
	]
       )

(* work with item wise list declarations
 * 	initializer name_of_list <- create_one_id ns self#list_type
 *)

     method private diamond_term = Term(name_of_diamond cl, Always, []) 

     val p_pred = Expression(ConstantPredicate("P")) 
     val q_pred = Expression(ConstantPredicate("Q")) 
     val gfp = Term(name_of_gfp,Always,[])
     val pred_type = Function(Self, Bool)

     method private box_csp s p=
       Application(
	 Application(
	   Application(
	     Term(name_of_box cl, Always, []),
	     coalgebra_term
	   ),
	   s
	 ),
	 p
       )

     method private diamond_csp s p=
       Application(
	 Application(
	   Application(
	     Term(name_of_diamond cl, Always, []),
	     coalgebra_term
	   ),
	   s
	 ),
	 p
       )

	 
     method private pqlm_declaration = 
        [Vardecl("P", pred_type); 
         Vardecl("Q", pred_type); 
(* work with itemwise list declarations
 * 	    (helper#list_decl);
 *)
	 (helper#method_decl)
        ] 

     method private box_def =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let y = create_one_id sub_ns Self in
       let y_term = Term(y, Always, []) in
	 Defn(
	   name_of_box cl,
	   [[Declared(name_of_coalgebra, self#coalgebra_type)];
	    [Undeclared(l, self#list_type)];
	    [Declared("P",pred_type)]],
	   pred_type,
	   Abstraction(
	     [x,Self],
	     Expression(
	       Exists(
		 ["Q",pred_type],
		 And([
		       Formula(
			 Application(
			   q_pred,
			   Term(x, Always, [])));
		       Formula(
			 Application(
			   Application(
			     Application(
			       Term(name_of_method_invariance cl,
				    Isabelle_only,
				    self#simple_arguments),
			       coalgebra_term),
			     Term(l,Always,[])),
			   q_pred));
		       Forall(
			 [y, Self],
			 Implies(
			   Formula(
			     Application(
			       q_pred,
			       y_term)),
			   Formula(
			     Application(
			       Term("P", Always, []),
			       y_term))
			 )
		       )
		     ])
	       )
	     )
	   )
	 )


     method private diamond_def =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let vars = create_ids sub_ns [Self;Self] in
       let x : string = fst(List.nth vars 0) in
       let y : string = fst(List.nth vars 1) in
	 Defn(
	   name_of_diamond cl,
	   [[Declared(name_of_coalgebra, self#coalgebra_type)];
	    [Undeclared(l, self#list_type)];
	    [Declared("P",pred_type)]],
	   pred_type,
	   Abstraction(
	     [y,Self],
	     Expression(
	       Not(
		 Formula(
		   Application(
		     self#box_csp (Term(l,Always,[]))
				 (Abstraction(
				    [x, Self],
				    Expression(
				      Not(
					Formula(
					  Application(
					    p_pred, 
					    Term(x,Always,[]))))))),
				 Term(y,Always,[])))))))



     method private box_K =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
         Lemma(
	   name_of_box_K_lemma cl,
	   Forall([l, self#list_type],
	   Implies(
	     Forall(
	       [x,Self],
	       Implies(
		 Formula(
		   Application(p_pred,x_term)
		 ),
		 Formula(
		   Application(q_pred,x_term)
		 )
	       )
	     ),
	     Forall(
	       [x,Self],
	       Implies(
		 Formula(
		   Application(
		     self#box_csp (Term(l,Always,[])) p_pred,
		     x_term)
		 ),
		 Formula(
		   Application(
		     self#box_csp (Term(l,Always,[])) q_pred,
		     x_term)
		 )
	       )
	     )
	   ))
	 )

	   
     method private box_T =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
	 Lemma(
	   name_of_box_T_lemma cl,
	   Forall(
	     [(l, self#list_type); (x,Self)],
	     Implies(
	       Formula(
		 Application(
		   self#box_csp (Term(l,Always,[])) p_pred,
		   x_term
		 )
	       ),
	       Formula(
		 Application(
		   p_pred,
		   x_term
		 )
	       )
	     )
	   )
	 )

     method private diamond_T =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
	 Lemma(
	   name_of_diamond_T_lemma cl,
	   Forall(
	     [(l, self#list_type); (x,Self)],
	     Implies(
	       Formula(
		 Application(
		   p_pred,
		   x_term
		 )
	       ),
	       Formula(
		 Application(
		   self#diamond_csp (Term(l,Always,[])) p_pred,
		   x_term
		 )
	       )
	     )
	   )
	 )
	   
     method private box_4 =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
	 Lemma(
	   name_of_box_4_lemma cl,
	   Forall(
	     [(l, self#list_type); (x,Self)],
	     Implies(
	       Formula(
		 Application(
		   self#box_csp (Term(l,Always,[])) p_pred,
		   x_term)
	       ),
	       Formula(
		 Application(
		   self#box_csp (Term(l,Always,[]))
			       (self#box_csp (Term(l,Always,[])) p_pred),
			       x_term
		 )
	       )
	     )
	   )
	 )
	   
     method private diamond_4 =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
	 Lemma(
	   name_of_diamond_4_lemma cl,
	   Forall(
	     [(l, self#list_type); (x,Self)],
	     Implies(
	       Formula(
		 Application(
		   self#diamond_csp (Term(l,Always,[]))
			       (self#diamond_csp (Term(l,Always,[])) p_pred),
			       x_term
		 )
	       ),
	       Formula(
		 Application(
		   self#diamond_csp (Term(l,Always,[])) p_pred,
		   x_term)
	       )
	     )
	   )
	 )

     method private box_inv =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       Lemma(
	 name_of_box_inv_lemma cl,
	 Forall([l, self#list_type],
	 Formula(
	   Application(
	     Application(
	       Application(
		 Term(name_of_method_invariance cl, Always, []),
		 coalgebra_term),
	       Term(l,Always,[])),
	     (self#box_csp (Term(l,Always,[])) p_pred)
	   )
	 ))
       )

(* This lemma is not true :-(
 *
 * 	method private diamond_inv =
 * 	  let sub_ns = sub_space ns in
 * 	  let x = create_one_id sub_ns Self in
 * 	    Lemma(
 * 	      name_of_diamond_inv_lemma cl,
 * 	      Not(
 * 		Formula(
 * 		  Application(
 * 		    Application(
 * 		      Application(
 * 			Term(name_of_method_invariance cl, Always, []),
 * 			coalgebra_term),
 * 		      Term(l,Always,[])),
 * 		    Abstraction(
 * 		      [x,Self],
 * 		      Expression(
 * 			Not(
 * 			  Formula(
 * 			    Application(
 * 			      self#diamond_csp (Term(l,Always,[])) p_pred,
 * 			      Term(x,Always,[]))))))))))
 *)
	 

     method private box_greatest_inv =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let y = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
       let y_term = Term(y, Always, []) in
	 Lemma(
	   name_of_box_greatest_inv_lemma cl,
	   Forall(
	     [(l, self#list_type); (y,Self)],
	     Implies(
	       And([
		     Formula(
		       Application(
			 Application(
			   Application(
			     Term(name_of_method_invariance cl, Always, []),
			     coalgebra_term),
			   Term(l,Always,[])),
			 p_pred)
		     );
		     Forall(
		       [x, Self],
		       Implies(
			 Formula(
			   Application(p_pred, x_term)
			 ),
			 Formula(
			   Application(q_pred, x_term)
			 )
		       )
		     );
		     Formula(
		       Application(p_pred, y_term)
		     )
		   ]),
	       Formula(
		 Application(
		   (self#box_csp (Term(l,Always,[])) q_pred),
		   y_term)
	       )
	     )
	   )
	 )
			       
     method private box_linear_cons =
       let sub_ns = sub_space ns in
       let l = create_one_id sub_ns self#list_type in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
       Lemma(
	 name_of_box_linear_cons_lemma cl,
	 Forall(
	   [(l, self#list_type); (x,Self)],
	   Implies(
	     Formula(
	       Application(
		 self#box_csp  
		   (Application(Term("cons", Always, []),
			       	Tuple([
				       	self#method_term;
				       	Term(l,Always,[])])
			       ))
		   p_pred,
		 x_term)
	     ),
	     Formula(
	       Application(
		 self#box_csp (Term(l,Always,[])) p_pred,
		 x_term)
	     )
	   )
	 )
       )

     method private box_linear_append =
       let sub_ns = sub_space ns in
       let (l1,l2) = create_id_pairs sub_ns [self#list_type] in
       let l1_term = Term(fst (List.hd l1), Always, []) in
       let l2_term = Term(fst (List.hd l2), Always, []) in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x, Always, []) in
       Lemma(
	 name_of_box_linear_append_lemma cl,
	 Forall(
	   l1 @ l2 @ [(x,Self)],
	   Iff(
	     And[
	       Formula(
		 Application(
		   self#box_csp l1_term p_pred,
		   x_term)
	       );
	       Formula(
		 Application(
		   self#box_csp l2_term p_pred,
		   x_term)
	       )],
	     Formula(
	       Application(
		 self#box_csp 
		    (Application(
		       Term("append", Always, []),
		       Tuple([l1_term; l2_term])))
		    p_pred,
		 x_term))
	   )))


     method get_name = ccsl_box_theory_name cl

     method get_parameters = self#simple_parameters
                               
     initializer top_theory#override_file_name (ccsl_class_file_name cl)

(* ISAR
 * 	method get_proofs = []
 *)
			 
     method make_body =
       [ 
	 Import([ccsl_method_invariance_rewrite_name cl, 
		 self#simple_arguments;
		]); 
         self#coalgebra_decl; 
       ] 
       @
       self#pqlm_declaration 
       @ 
       [
	 Comment("Defining Box");
	 self#box_def;
	 Comment("Defining Diamond");
	 self#diamond_def;
	 Comment(" Rules for Box and Diamond ");
	 Comment(" Box is a kind-of normal Operator ");  
	 self#box_K;

	 Comment(" Rule for Reflexivity ");
	 self#box_T;
	 Comment(" Dual Diamond Rule ");
	 self#diamond_T;
	 Comment(" Rule for Transitivity (with Dual Diamond Rule) ");
	 self#box_4;
         self#diamond_4;

	 Comment(" Box is an Invariant ");
	 self#box_inv;
(* This lemma ist not true :-(
 *
 * 	    Comment(" Dual Diamond Rule: Diamond holds  `in one step' ");
 * 	    self#diamond_inv;
 *)
	 self#box_greatest_inv;
	 self#box_linear_cons;
	 self#box_linear_append
       ]
  end (* ccsl_pre_box_theory *)

class ccsl_box_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_box_theory cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * inherit Box
 *
 * reuse the inherit method wise theory
 *)

class ['class_type, 'member_type] ccsl_pre_box_inherit_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type
  = 
  object (self : 'self)

     inherit 
       ['class_type, 'member_type] ccsl_pre_method_invariance_inherit_theory
       cl eq_types    as top_theory

     method get_name = ccsl_box_inherit_theory_name cl
			 

	(**********************************************************
	 *
	 * override section: the following methods are adopted 
	 *)

         (* term for "x", this is changed before things get hot *)
     val mutable x_term = Term("x",Always,[])

	 (* this is changed, once x is allocated *)
     val mutable x_decl_list = []

     method private get_x_decl_list = x_decl_list

     method private describe = "Box"

     method private lemma_name acl = name_of_box_inherit_lemma acl

     method private this_class_pred l_term p_term = 
       Application(
	 Application(
	   Application(
	     Application(
	       Term(name_of_box cl,Always,[]), 
	       coalgebra_term),
	     l_term),
	   p_term),
	 x_term)

     method private super_class_pred acl l_term p_term = 
       Application(
	 Application(
	   Application(
	     Application(
	       Term(name_of_box acl,Always,[]),
	       Application(Term(super_access_method(acl),
				Always,[]), 
			   coalgebra_term)),
	     l_term),
	   p_term),
	 x_term)

	(*
	 * end override section
	 *
	 **************************************************************)

     method make_body =
       let x = create_one_id ns Self in
       let _ = x_term <- Term(x, Always, []) in
       let _ = x_decl_list <- [(x, Self)] 
       in
	 [ 
           Import
	     [(ccsl_box_theory_name cl, self#simple_arguments);
	      (ccsl_method_invariance_inherit_name cl, self#simple_arguments)
	     ]; 
           self#coalgebra_decl; 
	 ] 
	 @
	 [Comment "Link Box with ancestor classes"]
	 @
	 (List.flatten
	    (List.map
	       self#do_inv_inherit_maybe
	       cl#get_resolved_ancestors)
	 )

  end (* ccsl_pre_box_inherit_theory *)

class ccsl_box_inherit_theory cl = 
  [ccsl_iface_type, ccsl_member_type] 
  ccsl_pre_box_inherit_theory cl eq_ccsl_types





(***********************************************************************
 ***********************************************************************
 *
 * DIAMOND (by paths)
 *
 *)

(* Jan: Need something to generate the adt for Computation Steps *)

class ['class_type, 'member_type] ccsl_pre_step_adt 
  (adt : 'class_type)
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  (cl : 'class_type)
  : ['class_type, 'member_type] ccsl_pre_adt_theory_type
  = 
object (self : 'self)
  inherit ['class_type, 'member_type] ccsl_pre_adt_theory_class 
    adt eq_types as top_theory

  method kind = Datatype

  method get_name = ccsl_step_theory_name cl

  method get_parameters = cl#get_parameters

  initializer top_theory#override_file_name (ccsl_class_file_name cl)
       
  method make_body = 
    [
     (* Import([ccsl_box_theory_name cl, cl#simple_arguments;]); *)
     (* Vardecl("P", TypeConstant("TYP",Always,[])); *)
      step_constructor_declarations ns cl;
    ]
  
(* ISAR
 *   method get_proofs = []
 *)

end (* ccsl_pre_step_adt *)

class ccsl_step_adt cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_step_adt 
  cl eq_ccsl_types cl

(***********************************************************************
 ***********************************************************************
 *
 * 
 *
 *)

class ['class_type, 'member_type] ccsl_pre_path_theory
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type 
  = 
  object (self : 'self)

     inherit ccsl_method_inv_helper_class cl
       as helper
     inherit 
       ['class_type, 'member_type] ccsl_pre_class_theory_class 
       cl eq_types   as top_theory

        (* reserve all names, that we want to declare *)
     initializer reserve ns 
       ((if cl#has_constructors then
	   [name_of_algebra]
          else [])
        @
        [
	  name_of_coalgebra;
	  "P";
	  name_of_method_id;
	  name_of_box cl;
	  name_of_diamond cl;
	]
       )

     method private step_type = 
       TypeConstant(ccsl_step_theory_name cl,Always,orig_arguments)

     method private step_list_type = 
       Adt(self#list_adt,Always,[TypeArgument self#step_type])

     initializer name_of_list <- create_one_id ns self#list_type
     initializer name_of_step <- create_one_id ns self#step_type
     initializer name_of_path1 <- create_one_id ns self#step_list_type
     initializer name_of_path2 <- create_one_id ns self#step_list_type

     method private path1_term = Term(name_of_path1,Never,[])

     method private path2_term = Term(name_of_path2,Never,[])

     method private step_term = Term(name_of_step, Never,[])

     method private adm_term = Term(name_of_admissible,Never,[])

     method private trans_term = Term(name_of_transition,Never,[])

     method private diamond_term = Term(name_of_diamond cl, Always, []) 

     val q_pred = Expression(ConstantPredicate("Q")) 
     val pred_type = Function(Self, Bool)

     method private box_csp s p=
       Application(
	 Application(
	   Application(
	     Term(name_of_box cl, Always, []),
	     coalgebra_term
	   ),
	   s
	 ),
	 p
       )

     method private diamond_csp s p=
       Application(
	 Application(
	   Application(
	     self#diamond_term,
	     coalgebra_term
	   ),
	   s
	 ),
	 p
       )

	 
     method private pqlm_declaration = 
        [Vardecl("P", pred_type); 
         Vardecl("Q", pred_type); 
         (helper#list_decl);
	 (helper#method_decl)
        ]
       
     method get_name = ccsl_path_theory_name cl
			 
     method get_parameters = self#simple_parameters
                               
     method private declarations =
       [Vardecl("P", pred_type);
	self#coalgebra_decl;
        Vardecl(name_of_path1, self#step_list_type);
	Vardecl(name_of_path2, self#step_list_type);
	Vardecl(name_of_step, self#step_type);
	self#list_decl
       ]
	
     initializer top_theory#override_file_name (ccsl_class_file_name cl)

     method private admissible_def =
       let sub_ns = sub_space ns in
       let name_of_m1 = create_one_id ns self#step_type in
       let name_of_l1 = create_one_id ns self#step_list_type in
 	 Defrecfn(name_of_admissible,
 	      [[Declared(name_of_list, self#list_type);
		Declared(name_of_path1, self#step_list_type)]],
 	      Bool,
	      Application(
		Term("length",Never,[]),
		self#path1_term
	      ),
	      Case(self#path1_term,
		    [(* null: true *)
		      Term("null",Never,[]),Expression(True);
		      (* cons(m1,l1): ... *)
		      Application(
			Term("cons",Never,[]),
			Tuple([
				Term(name_of_m1,Never,[]);
				Term(name_of_l1,Never,[])
			      ])
		      ), (* cases for steps *)
		      Case(Term(name_of_m1,Never,[]),
			   List.fold_right
			     (fun m l ->
				(admissible_cases_for_method 
				   sub_ns 
				   m 
				   self#list_term)@l)
			     cl#get_all_methods
			     []
			  )
		    ]
		   )
	     )
	   

     method private admissible_null_lemma =
       Lemma(
	 name_of_admissible_null_lemma,
	 Formula(
	   Application(
	     self#adm_term,
	     Tuple([ self#list_term; Term("null",Never,[])])
	   )
	 )
       )

     method private admissible_linear_lemma =
       let sub_ns = sub_space ns in
       let mc = create_one_id sub_ns self#method_enum_type in
       Lemma(
	 name_of_admissible_linear_lemma,
	 Implies(
	   Formula(
	     Application(
	       self#adm_term,
	       Tuple([self#list_term;
		      Application(
			Term("cons",Never,[]),
			Tuple([ 
				self#step_term;
				self#path1_term
			      ])
		      )])
	     )
	   ),
	   Formula(
	     Application(
	       self#adm_term,
	       Tuple([self#list_term;self#path1_term])
	     )
	   )
	 )
       )
       
			   
     method private admissible_append_lemma =
       Lemma(
	 name_of_admissible_append_lemma,
	 Iff(
	   Formula(
	     Application(
	       self#adm_term,
	       Tuple([self#list_term;
		      Application(
			Term("append",Never,[]),
			Tuple([self#path1_term;
			       self#path2_term])
		      )
		     ]
		    )
	     )
	   ),
	   And(
	     [ Formula(
		 Application(
		   self#adm_term,
		   Tuple([self#list_term;
			  self#path1_term])
		 )
	       );
	       Formula(
		 Application(
		   self#adm_term,
		   Tuple([self#list_term;
			  self#path2_term])
		 )
	       )
	     ]
	   )
	 )
       )


     method private transition_def =
       let sub_ns = sub_space ns in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x,Never,[]) in
	 Defrecfn(
	   name_of_transition,
	   [[Declared(name_of_coalgebra,self#coalgebra_type)];
	    [Declared(name_of_path1,self#step_list_type)]],
	   Function(Self,Self),
	   Application(
	     Term("length",Never,[]),
	     self#path1_term
	   ),
	   Abstraction(
	     [x,Self],
	     Case(
	       self#path1_term,
	       [
		 Term("null",Never,[]), x_term;
		 Application(
		   Term("cons",Never,[]),
		   Tuple([
			   self#step_term;
			   self#path2_term
			 ])
		 ),
		 Let(
		   [ id_record_from_string "prev_trans",
		     None,
		     Application(
		       Application(
			 Application(
			   self#trans_term,
			   coalgebra_term
			 ),
			 self#path2_term
		       ),
		       x_term
		     )
		   ],
		   Case(
		     self#step_term,
		     List.flatten
		       (List.map
			  (fun mem ->
			     (transition_cases_for_method 
				sub_ns mem
				coalgebra_term
				(Term("prev_trans",Never,[])))
			  )
			  cl#get_all_methods)
		   )
		 )
	       ]
	     )
	   )
	 )
	   

     method private transition_append_lemma =
       let sub_ns = sub_space ns in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x,Never,[]) in
       let transition_term = Term(name_of_transition,Never,[]) in
	 Lemma(
	   name_of_transition_append_lemma,
	   Forall(
	     [ x, Self ],
	     Equal(
	       Application(
		 Application(
		   Application(
		     transition_term,
		     coalgebra_term
		   ),
		   Application(
		     Term("append",Never,[]),
		     Tuple(
		       [self#path1_term;self#path2_term]
		     )
		   )
		 ),
		 x_term
	       ),
	       Application(
		 Application(
		   Application(
		     transition_term,
		     coalgebra_term
		   ),
		   self#path1_term
		 ),
		 Application(
		   Application(
		     Application(
		       transition_term,
		       coalgebra_term
		     ),
		     self#path2_term
		   ),
		   x_term
		 )
	       )
	     )
	   )
	 )

     method private transition_at_end_lemma =
       let sub_ns = sub_space ns in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x,Never,[]) in
       let transition_term = Term(name_of_transition,Never,[]) in
       let reverse_term = Term("reverse",Never,[]) in
       let cons_term = Term("cons",Never,[]) in
	 Lemma( 
	   name_of_transition_at_end_lemma,
	   Forall(
	     [x,Self],
	     Equal(
	       Application(
		 Application(
		   Application(
		     transition_term,
		     coalgebra_term
		   ),
		   Application(
		     reverse_term,
		     Application(
		       cons_term,
		       Tuple(
			 [self#step_term;self#path1_term]
		       )
		     )
		   )
		 ),
		 x_term
	       ),
	       Application(
		 Application(
		   Application(
		     transition_term,
		     coalgebra_term
		   ),
		   Application(
		     reverse_term,
		     self#path1_term
		   )
		 ),
		 Application(
		   Application(
		     Application(
		       transition_term,
		       coalgebra_term
		     ),
		     Application(
		       cons_term,
		       Tuple(
			 [
			   self#step_term;
			   Term("null",Never,[])
			 ])
		     )
		   ),
		   x_term
		 )
	       )
	     )
	   )
	 )


     method private inv_char_lemma =
       let sub_ns = sub_space ns in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x,Never,[]) in
       let p_term = Term("P",Never,[]) in
       let transition_term = Term(name_of_transition,Never,[]) in
	 Lemma(
	   name_of_invariant_char_lemma,
	   Iff(
	     Formula(
	       Application(
		 Application(
		   Application(
		     Term(name_of_method_invariance cl,Never,[]),
		     coalgebra_term
		   ),
		   self#list_term
		 ),
		 p_term
	       )
	     ),
	     Forall(
	       [ x,Self;
		 name_of_path1, self#step_list_type],
	       Implies(
		 And([
		       Formula(
			 Application(p_term,x_term)
		       );
		       Formula(
			 Application(
			   self#adm_term,
			   Tuple(
			     [self#list_term;
			      self#path1_term]
			   )
			 )
		       )
		     ]),
		 Formula(
		   Application(
		     p_term,
		     Application(
		       Application(
			 Application(
			   transition_term,
			   coalgebra_term
			 ),
			 self#path1_term
		       ),
		       x_term
		     )
		   )
		 )
	       )
	     )
	   )
	 )

     method private diamond_transition_lemma =
       let sub_ns = sub_space ns in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x,Never,[]) in
       let p_term = Term("P",Never,[]) in
       let transition_term = Term(name_of_transition,Never,[]) in
	 Lemma(
	   name_of_diamond_transition_lemma,
	   Forall(
	     [x,Self],
	     Implies(
	       Formula(
		 Application(
		   self#diamond_csp self#list_term p_term,
		   x_term
		 )
	       ),
	       Exists(
		 [name_of_path1,self#step_list_type],
		 And(
		   [
		     Formula(
		       Application(
			 self#adm_term,
			 Tuple(
			   [self#list_term;self#path1_term]
			 )
		       )
		     );
		     Formula(
		       Application(
			 p_term,
			 Application(
			   Application(
			     Application(
			       transition_term,
			       coalgebra_term
			     ),
			     self#path1_term
			   ),
			   x_term
			 )
		       )
		     )
		   ]
		 )
	       )
	     )
	   )
	 )


     method private transition_diamond_lemma =
       let sub_ns = sub_space ns in
       let x = create_one_id sub_ns Self in
       let x_term = Term(x,Never,[]) in
       let p_term = Term("P",Never,[]) in
       let transition_term = Term(name_of_transition,Never,[]) in
	 Lemma(
	   name_of_transition_diamond_lemma,
	   Forall(
	     [x,Self],
	     Implies(
	       Not(
		 Forall(
		   [name_of_path1, self#step_list_type],
		   Not(
		     And(
		       [
			 Formula(
			   Application(
			     self#adm_term,
			     Tuple(
			       [self#list_term;self#path1_term]
			     )
			   )
			 );
			 Formula(
			   Application(
			     p_term,
			     Application(
			       Application(
				 Application(
				   transition_term,
				   coalgebra_term
				 ),
				 self#path1_term
			       ),
			       x_term
			     )
			   )
			 )
		       ]
		     )
		   )
		 )
	       ),
	       Formula(
		 Application(
		   self#diamond_csp self#list_term p_term,
		   x_term
		 )
	       )
	     )
	   )
	 )
		       
			 
	       
(* ISAR
 * 	method get_proofs = []
 *)
			 
     method make_body =
       [	    
	 Import([(ccsl_box_theory_name cl, 
		  self#simple_arguments);
		 (ccsl_step_theory_name cl,
		  orig_arguments);
		])
       ]
       @
       self#declarations
       @
       [
	 self#admissible_def;
         self#admissible_null_lemma;
	 self#admissible_linear_lemma;
	 self#transition_def;
	 self#transition_append_lemma;
	 self#transition_at_end_lemma;
	 self#inv_char_lemma;
	 self#diamond_transition_lemma;
	 Comment("The following Lemma might look strange due to lots of (redundant NOTs");
	 Comment("However, these are usefull when using the lemma for rewriting");
	 self#transition_diamond_lemma
       ]
  end

class ccsl_path_theory cl = 
  [ccsl_iface_type,ccsl_member_type] ccsl_pre_path_theory cl eq_ccsl_types

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)



