(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Monday 28 October 02 14:52:55 tews@ithif51>
 *
 * Typechecking with unification
 *
 * $Id: newtypecheck.ml,v 1.19 2002/10/28 16:33:19 tews Exp $
 *
 *
 * the algorithm is homegrown and works in two passes
 * 1. bottem up type derivation
 * 2. top down type checking
 * Both passes are performed in sequence for every assertion. They 
 * share a (per assertion) persistent store of bindings for free 
 * type variables.
 * 
 * Type derivation derives types for the expressions and introduces 
 * free type variables when neccessary. Certain nodes (such as application) 
 * perform type unification. If the unification fails an error is reported. 
 * Otherwise the unification may bind free type variables to types. 
 * At places where type information is lost in the derivation (for 
 * instance Application, Equal) some types are stored in the ast 
 * for the following type check.
 * 
 * Type checking works top down and checks if all free type variables 
 * are bound to a ground type. For Members it checks if a complete 
 * instanciation for the class/adt is available. If so the instanciation 
 * is stored in the ast.
 * For projections their type is checked, for Obseq an appropiate 
 * lifting is requested via add_rel_lifting.
 * 
 * Type unification is straightforward. It keeps a store of bindings for 
 * free type variables. For the occur check to work there is an 
 * important side condition (see below in the type variable 
 * management section)
 *)

open Util
open Error
open Global
open Top_variant_types
open Ccsl_pretty
open Classtypes
open Types_util
;;


(***********************************************************************
 ***********************************************************************
 *
 * Utility functions
 *
 *)

let is_form_loc = function
  | FormLoc _ -> true
  | _ -> false

let is_expr_loc = function
  | ExprLoc _ -> true
  | _ -> false


(***********************************************************************
 ***********************************************************************
 *
 * Error messages
 *
 *)

exception Typecheck_error

let undefined_method cl token loc = begin
  error_message loc ("Method " ^ token.token_name ^
		       " not defined in class " ^
		       cl#get_name ^ ".");
  if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE then begin
    prerr_string( "Symboltable: " );
    print_verbose (Symbol.symbol_table_dump 
		     Top_variant_types_util.dump_symbol_brief);
  end;
  raise Typecheck_error
end;;


let types_dont_match loc typ constrainttyp =
  error_message loc
    ("This expression has type " ^
     (string_of_ccsl_type typ) ^
     "\nbut is expected to have type " ^
     (string_of_ccsl_type constrainttyp) ^ "."
    )
	  
let eq_not_equal loc typeleft typeright =
  error_message loc
    ("Both sides of the equation must have equal type. Type\n" ^
     (string_of_ccsl_type typeleft) ^
     " is not compatible with type\n" ^
     (string_of_ccsl_type typeright)
    )

let is_not_fun end_message loc typ constr =
  error_message loc
    ("This expression has type " ^
     (string_of_ccsl_type typ) ^ ". " ^
     end_message ^ "."
    )

let dom_mismatch loc domtype argtype =
  error_message loc
    ("Type mismatch in application. Function domain type\n" ^
     (string_of_ccsl_type domtype) ^
     " is not compatible with type of argument\n" ^
     (string_of_ccsl_type argtype) ^ "."
    )

let proj_codom_mismatch projtype i loc ith_typ codom =
  error_message loc
    ("Type error. Derived\n" ^
     (string_of_ccsl_type projtype) ^
     " for " ^ 
     (match i with 
	| 1 -> "1st"
	| 2 -> "2nd"
	| 3 -> "3rd"
	| _ -> (string_of_int i) ^ "th"
     ) ^ " projection. Type " ^
     (string_of_ccsl_type ith_typ) ^ "\n" ^
     "is not compatible with type " ^
     (string_of_ccsl_type codom) ^ "."
    )

let mismatch loc type1 type2 =
  error_message loc
    ("Type mismatch. Type " ^
     (string_of_ccsl_type type1) ^
     "\nis not compatible with type " ^
     (string_of_ccsl_type type2) ^ "."
    )

let if_mismatch if_or_case loc type1 type2 =
  error_message loc
    ("Branches of " ^
     if_or_case ^ 
     " have different types. Type\n" ^
     (string_of_ccsl_type type1) ^
     "\nis not compatible with type\n" ^
     (string_of_ccsl_type type2)
    )

let case_mismatch loc typex adttype =
  error_message loc
    ("This case expression matches values of " ^
     (match adttype with
	| Adt(adt,_,_) -> adt#get_name
	| _ -> assert(false)
     ) ^
     ". Type\n" ^
     (string_of_ccsl_type typex) ^
     " is not compatible with\n" ^
     (string_of_ccsl_type adttype) ^ "."
    )

       
let type_constr_error loc typ constr =
  error_message loc
    ("Type constraint violated. Derived type\n" ^
     (string_of_ccsl_type typ) ^
     " is not compatible with constraint\n" ^
     (string_of_ccsl_type constr)
    )

let wrong_object_type loc objtype quali =
  error_message loc
    ("Type mismatch in qualified method selection. Derived type\n" ^
     (string_of_ccsl_type objtype) ^
     " is not compatible with " ^
     (string_of_ccsl_type quali) ^ "."
    )

let proj_i_error projexpr loc producttype =
  error_message loc
  ("Type mismatch. Derived " ^
   (string_of_ccsl_type producttype) ^ "\n" ^
   "for the domain of " ^
   (string_of_ccsl_expression projexpr) ^ "."
  )

let proj_dom_error loc domtype =
  error_message loc
    ("Type mismatch. Derived " ^
     (string_of_ccsl_type domtype) ^
     " for the domain of a projection.\n" ^
     "This is not compatible with a product type."
    )


let invalid_definition loc construct =
  error_message loc
    (construct ^ " is not allowed in definitional extension.");
  raise Typecheck_error

let d s = 
  if debug_level _DEBUG_TYPECHECK then
    print_verbose s

let dr s = d( "    TC " ^ s)

let du s = 
  if debug_level _DEBUG_UNIFICATION then
    print_verbose ( "    Unify " ^ s)


let dc s = 
  if debug_level _DEBUG_TYPECHECK then
    print_verbose ( "    C " ^ s)


(***********************************************************************
 ***********************************************************************
 *
 * type variable management
 *
 *)

exception Occur_check

type tv_binding_type = (Type_variable.t * ccsl_output_types) list ref

let new_tv_binding () : tv_binding_type = ref []

(* the following invariant is kept on new_tv_binding:
 * type variables that have a binding must not occur in the types 
 * for bindings of other type variables. 
 * the function add_new_binding keeps this assertion by 
 * performing various substitutions before adding a binding.
 *)

let string_of_tv_bindings (tv_bindings : tv_binding_type) = 
  List.fold_right (fun (tv,typ) res -> 
		     (Type_variable.string_of_tv tv) ^ " => " ^
		     (string_of_ccsl_type typ) ^ "; " ^ res)
    !tv_bindings ""

let find_tv (tv_bindings : tv_binding_type) tv = 
  Util.assoc Type_variable.eq tv !tv_bindings

let normalize_type (tv_bindings : tv_binding_type) t = 
  ccsl_substitute_types 
    (List.map (fun (tv,t) -> (FreeTypeVariable tv, t))
       !tv_bindings)
    t

let normalize_args (tv_bindings : tv_binding_type) arglist = 
  ccsl_substitute_arguments
    (List.map (fun (tv,t) -> (FreeTypeVariable tv, t))
       !tv_bindings)
    arglist


let add_new_binding (tv_bindings : tv_binding_type) tv t = 
  let nt = normalize_type tv_bindings t in
  let _ = dr ("bind " ^
	     (Type_variable.string_of_tv tv) ^
	     " to " ^ (string_of_ccsl_type t)) 
  in
  let _ = if occur_check tv nt then raise Occur_check 
  in let subst = [ FreeTypeVariable tv, nt] in
  let ntv_bindings = 
    List.map (fun (tv,t) -> (tv, ccsl_substitute_types subst t))
      !tv_bindings
  in
    tv_bindings := (tv,nt) :: ntv_bindings


(***********************************************************************
 ***********************************************************************
 *
 * Type Unification
 *
 *)

exception Unify

let rec unify tv_bindings t1 t2 = 
  let recurse = unify tv_bindings in
  let recurse_arg = unify_arg tv_bindings 
  in
    (match t1,t2 with
       | Groundtype(id1,args1), Groundtype(id2,args2) when
	   id1.id_token.token_name = id2.id_token.token_name ->
	     begin
(* HENDRIK: this assertion can be triggered with overloading groundtypes *)
	       assert((List.length args1) = (List.length args2));
	       Groundtype(id1, List.map2 recurse_arg args1 args2)
	     end
       | BoundTypeVariable id1, BoundTypeVariable id2 ->
	   if id1.id_token.token_name = id2.id_token.token_name 
	   then
	     BoundTypeVariable id1
	   else
	     raise Unify
							  (* we are in unify *)
       | FreeTypeVariable t1, FreeTypeVariable t2
	   when Type_variable.eq t1 t2 
	     ->
	        FreeTypeVariable t1
       | Self, Self -> Self
       | Carrier, Carrier -> Carrier
       | Bool, Bool -> Bool
       | Function(dom1,codom1), Function(dom2,codom2) -> 
	   let ndom = recurse dom1 dom2 in
	   let ncodom = recurse codom1 codom2 
	   in
	     Function(ndom, ncodom)
       | Product(type_list1),Product(type_list2) ->
	   if (List.length type_list1) = (List.length type_list2) 
	   then
	     Product( List.map2 recurse type_list1 type_list2 )
	   else
	     raise Unify
							  (* we are in unify *)
       | Class(cl1, args1),Class(cl2, args2) ->
	   if cl1#get_name = cl2#get_name 
	   then
	     begin
	       assert((List.length args1) = (List.length args2));
	       Class(cl1, List.map2 recurse_arg args1 args2)
	     end
	   else
	     raise Unify
       | Adt(adt1,flag1,args1),Adt(adt2,flag2,args2) ->
	   if adt1#get_name = adt2#get_name 
	   then
	     begin
	       assert((List.length args1) = (List.length args2));
	       Adt(adt1, flag1, List.map2 recurse_arg args1 args2)
	     end
	   else
	     raise Unify
							  (* we are in unify *)

					(* new bindings *)
       | FreeTypeVariable tv1, FreeTypeVariable tv2 ->
	   (try
	      let binding = find_tv tv_bindings tv1
	      in
		recurse binding t2
	    with
	      | Not_found ->
		  try
		    let binding = find_tv tv_bindings tv2
		    in
		      recurse t1 binding
		  with
		    | Not_found ->
			(try
			   add_new_binding tv_bindings tv2 t1;
			   t1
			 with
			   | Occur_check -> raise Unify
			)
	   )		  

							  (* we are in unify *)
       | FreeTypeVariable tv, t ->
	   (try
	      let binding = find_tv tv_bindings tv
	      in 
		recurse binding t
	    with
	      | Not_found -> 
		  (try
		     add_new_binding tv_bindings tv t;
		     t
		   with 
		     | Occur_check -> raise Unify
		  )
	   )
       | t, FreeTypeVariable tv ->
	   (try
	      let binding = find_tv tv_bindings tv
	      in 
		recurse t binding
	    with
	      | Not_found -> 
		  (try
		     add_new_binding tv_bindings tv t;
		     t
		   with 
		     | Occur_check -> raise Unify
		  )
	   )
							  (* we are in unify *)
       | Groundtype(id1,args1), t2 when is_type_def id1 ->
	   let subst = 
	     make_substitution (get_ground_type_parameters id1) args1
	   in
	     recurse (ccsl_substitute_types subst id1.id_type) t2

       | t1, Groundtype(id2,args2) when is_type_def id2 ->
	   let subst = 
	     make_substitution (get_ground_type_parameters id2) args2
	   in
	     recurse t1 (ccsl_substitute_types subst id2.id_type)

	   
					(* not allowed cases *)
       | SmartFunction _,_
       | TypeConstant _,_
       | Predtype _,_
       | Record _,_
       | IFace _,_
       | Array _,_ 
	   
       | _, SmartFunction _
       | _, TypeConstant _
       | _, Predtype _
       | _, Record _
       | _, IFace _
       | _, Array _          -> assert(false)
	   
	   
				(* cases where the constructors mismatch *)
       | Groundtype _, _
       | BoundTypeVariable _ ,_
       | Self ,_
       | Carrier ,_
       | Bool, _
       | Function _ ,_
       | Product _ ,_
       | Class _ ,_
       | Adt _ ,_
							  (* we are in unify *)
(* double cases
 * 	  | _, Groundtype _
 * 	  | _, BoundTypeVariable _
 * 	  | _, Self
 * 	  | _, Carrier 
 * 	  | _, Bool
 * 	  | _, Function _
 * 	  | _, Product _
 * 	  | _, Class _
 * 	  | _, Adt _ 
 *)
          -> raise Unify
    )
    
and unify_arg tv_bindings arg1 arg2 = 
  (match arg1,arg2 with
     | TypeArgument t1, TypeArgument t2 ->
	 TypeArgument( unify tv_bindings t1 t2 )
  )


let unify_types tv_bindings t1 t2 error_msg_fun =
  try
    let _ = du ((string_of_ccsl_type t1) ^ " & " ^
		(string_of_ccsl_type t2)) in
    let nt = unify tv_bindings t1 t2 in

    let _ = du ("==> " ^ (string_of_ccsl_type nt))
    in let _ = du ("bound vars: " ^(string_of_tv_bindings tv_bindings))
    in
      nt
	       
  with
    | Unify ->
	let nt1 = (normalize_type tv_bindings t1) in
	let nt2 = (normalize_type tv_bindings t2) in
	  error_msg_fun nt1 nt2;
	  raise Typecheck_error


let types_must_unify t1 t2 =
  let tv_bindings = new_tv_binding () 
  in
    try
      ignore(unify tv_bindings t1 t2);
      !tv_bindings = []
    with
      | Unify -> false


(***********************************************************************
 ***********************************************************************
 *
 * Type Checking
 *
 *)

let check_ground loc tv_bindings typ message =
  let ntyp = normalize_type tv_bindings typ in
  let _ = 
    if type_is_nonground ntyp then
      begin
	error_message loc
	  ("Type not uniquely determined.\n" ^
	   "Derived " ^
	   (string_of_ccsl_type ntyp) ^
	   " for " ^ message ^ ".\n" ^
	   "Please add appropriate type annotations."
	  );
	raise Typecheck_error
      end
  in
    ntyp

let check_ground_arglist loc tv_bindings iface args message =
  let nargs = normalize_args tv_bindings args in
  let _ = 
    if args_are_nonground nargs then
      begin
	error_message loc
	  ("Type not uniquely determined.\n" ^
	   "Derived " ^
	   (iface#get_name) ^ "::" ^
	   (string_of_ccsl_arguments nargs) ^
	   " for " ^ message ^ ".\n" ^
	   "Please add appropriate type annotations."
	  );
	raise Typecheck_error
      end
  in
    nargs
    

let generate_lifting_name typ =
  let s = string_of_ccsl_type typ in
  let r = String.make (String.length s) 'X' in
  let rec copy i j under =
    if i = String.length s 
    then j
    else
      if (s.[i] = ' ') 
	  || (s.[i] = '[') 
	  || (s.[i] = ']') 
	  || (s.[i] = ',') 
	  || (s.[i] = '-') 
	  || (s.[i] = '>') 
      then (if under
	    then 
	      (r.[j] <- '_';
	       copy (i + 1) (j + 1) false
	      )
	    else
	       copy (i + 1) j false
	   )
      else
	(r.[j] <- s.[i];
	 copy (i + 1) (j + 1) true
	)
  in let n = (copy 0 0 false) in
  let n_ = if (n > 0) && (r.[n-1] = '_') then n-1 else n
  in
    Names.lifting_name_prefix ^ (String.sub r 0 n_)


let rec check_expression ccl loc tv_bindings expr_typ expr = 
  let recurse_exp = check_expression ccl loc tv_bindings in
  let recurse_form = check_formula ccl loc tv_bindings in
  let _ = assert(not (type_is_nonground expr_typ)) in
  let nexpression = 
    match expr with
      | ExprLoc(ex,l) -> 
	  ExprLoc(check_expression ccl l tv_bindings expr_typ ex, l)
					       (* we are in check_expression *)
      | TypeAnnotation(BasicExpr(Member(instiface, 
					Resolved(m))), typ) ->
	  let ntyp = 
	      check_ground loc tv_bindings typ 
		("for member " ^ m#get_name)
	  in
	  let ninstiface = match instiface with
	    | CurrentIface -> CurrentIface
	    | InstIface(iface, args, instloc) -> 
		let nargs = check_ground_arglist loc tv_bindings 
			      iface args
			      ("hosting class of " ^
			       m#get_name)
		in begin
			(* iterate components unless ccl is a signature 
			 * and m is a ground term from it
			 *)
		    if ccl#is_sig && m#hosting_class#get_name = ccl#get_name
		    then
		      ()
		    else begin
		      iter_component_arglist_add_component ccl
			nargs (get_member_parameters m);
		      ccl#add_component (Pos,iface,nargs) 
		    end;
				(* add a special import if m is coreduce *)
		    (if m#get_sort = Class_Coreduce 
		     then
		       let self1,self2 = match ntyp with
			 | Function(_,Function(self1,self2)) -> self1,self2
			 | _ -> assert(false)
		       in let _ = assert(iface#is_class) 
		       in
			 ccl#add_iface_import
			   (Names.ccsl_finality_theory_name iface,
			    TypeArgument(self1) :: 
			      TypeArgument(self2) :: 
			      (normalize_args tv_bindings nargs))
		    );
		    InstIface(iface,nargs,instloc)
		  end
	    | NoIface -> 
		assert(false)
	  in 
	  let _ = dc ("derive " ^
		      (string_of_ccsl_inst_iface ninstiface) ^
		      " for member " ^
		      m#get_name) 
	  in
	  let _ = assert(types_must_unify expr_typ ntyp) 
	  in
	    TypeAnnotation(
	      BasicExpr(Member(ninstiface, Resolved(m))),
	      expr_typ)
	      
					       (* we are in check_expression *)
      | BasicExpr(TermVar(Resolved(id_rec))) -> 
	  (* if the type of id_rec was not determined completely, 
	   * then this is checked in Let or CCSL_Case
	   *)
	  let _ = assert(types_must_unify expr_typ id_rec.id_type)
	  in
	    expr
				(* other BasicExpr cannot appear here *)
      | BasicExpr(TermVar(Unresolved _))
      | BasicExpr(Member _) -> 
	  assert(false)

					       (* we are in check_expression *)
      | TypedTerm(Term _, typ) -> 
	  begin
	    assert(types_must_unify expr_typ typ);
	    expr
	  end

      | TypedTerm(ex, typ) -> 
	  begin
	    assert(types_must_unify expr_typ typ);
	    TypedTerm( recurse_exp expr_typ ex, typ)
	  end
	  
			(* ex_typ is normalized, see derive_expression *)
      | MethodSelection(TypeAnnotation(ex,ex_typ), instiface, m) ->
	  let _ = 
	    iter_components_add_component ccl ex_typ
	  in 
	    MethodSelection(
	      TypeAnnotation(recurse_exp ex_typ ex, ex_typ), 
	      instiface, m) 

				(* MethodSelection needs TypeAnnotation *)
      | MethodSelection _ ->
	  assert(false)

					       (* we are in check_expression *)
      | Tuple(ex_list) -> 
	  let tl = match expand_type_def eq_ccsl_types expr_typ with
	    | Product tl -> tl
	    | _ -> assert(false)
	  in
	    Tuple(List.map2 recurse_exp tl ex_list)

      | Projection(i,_) -> 
	  let domtl, codom = 
	    match expand_type_def eq_ccsl_types expr_typ with
	      | Function(Product tl, codom) -> tl, codom
	      | Function(domtyp,_) -> 
		  begin
		    proj_dom_error loc domtyp;
		    raise Typecheck_error
		  end
	      | _ -> assert(false)
	  in
	  let tuple_len = List.length domtl in
	  let _ = if i > tuple_len then
	    begin
	      proj_i_error expr loc (Product domtl);
	      raise Typecheck_error
	    end
	  in let _ = 
			(* all involved types are ground, 
			 * therefore type context is not changed here *)
	      unify_types tv_bindings (List.nth domtl (i - 1)) codom
		(proj_codom_mismatch expr_typ i loc)
	  in let _ = dc ("derive " ^ (string_of_ccsl_type expr_typ) ^
			 " for " ^ (string_of_ccsl_expression expr))
	  in
	    TypeAnnotation(Projection(i,tuple_len), expr_typ)

					       (* we are in check_expression *)
      | Abstraction(decl_list,ex) ->
	  let codom = match expand_type_def eq_ccsl_types expr_typ with
	    | Function(_, codom) -> codom
	    | _ -> assert(false)
	  in 
	    Abstraction(decl_list, recurse_exp codom ex)

      | Application( TypeAnnotation(ex1,ex1_typ), ex2) -> 
	  let nex1_typ = check_ground (get_ex_loc ex1) tv_bindings 
			   ex1_typ "function"
	  in
	  let _ = dc ("derive " ^ (string_of_ccsl_type nex1_typ) ^
		      " for fun " ^ (string_of_ccsl_expression ex1))
	  in let dom,codom = match expand_type_def eq_ccsl_types nex1_typ with
	    | Function(dom,codom) -> dom,codom
	    | _ -> assert(false)
	  in let _ = assert(types_must_unify codom expr_typ)
	  in
	    Application( 
	      TypeAnnotation(recurse_exp nex1_typ ex1, nex1_typ),
	      recurse_exp dom ex2)

					(* Application needs TypeAnnotation *)
      | Application _ ->
	  assert(false)

					       (* we are in check_expression *)
      | TypeAnnotation(InfixApplication(ex1, instiface, (Resolved m), ex2), 
		       memtype) ->
	  let ninstiface = match instiface with
	    | InstIface(iface, args, instloc) -> 
		let nargs = check_ground_arglist loc tv_bindings 
			      iface args
			      ("infix operator " ^
			       m#get_name)
		in 
		  begin
			(* iterate components unless ccl is a signature 
			 * and m is a ground term from it
			 *)
		    if ccl#is_sig && m#hosting_class#get_name = ccl#get_name
		    then
		      ()
		    else begin
		      iter_component_arglist_add_component ccl
			nargs (get_member_parameters m);
		      ccl#add_component (Pos,iface,nargs) 
		    end;
		    InstIface(iface,nargs,instloc)
		  end
				(* may happen in a top level definition *)
	    | CurrentIface -> CurrentIface

	    | NoIface -> 
		assert(false)
	  in 
	  let nmemtype = check_ground loc tv_bindings memtype 
			   "infix operator"
	  in
	  let _ = dc ("derive " ^ (string_of_ccsl_type nmemtype) ^
		      " for infix " ^ m#get_name )
	  in
	  let lefttype, righttype, resulttype =
	    (match expand_type_def eq_ccsl_types nmemtype with
	       | Function(Product [t1;t2], t3) -> t1, t2, t3
	       | Function(t1, Function(t2,t3)) -> t1, t2, t3
	       | _ -> assert(false)
	    )
	  in let _ = assert(types_must_unify resulttype expr_typ)
	  in
	    TypeAnnotation(
	      InfixApplication(
		recurse_exp lefttype ex1,
		ninstiface, Resolved m,
		recurse_exp righttype ex2),
	      nmemtype)
					       (* we are in check_expression *)

				(* InfixApplication needs TypeAnnotation *)
      | InfixApplication _ ->
	  assert(false)

      | FunUpdate(fex, changes) -> 
	  let dom_type, codom_type = 
	    match expand_type_def eq_ccsl_types expr_typ with
	      | Function(dom,codom) -> dom,codom
	      | _ -> assert(false)
	  in 
	    FunUpdate(
	      recurse_exp expr_typ fex,
	      List.map (fun (arg,nval) ->
			  (recurse_exp dom_type arg, 
			   recurse_exp codom_type nval))
		changes)

      | Let(decl_list, ex) ->
	  let ndecl_list = 
	    List.map (fun (id_rec, typeopt, ex) ->
			let ntyp_ex = 
			  check_ground (get_ex_loc ex) tv_bindings 
			    id_rec.id_type "expression"
			in let _ = dc ("derive " ^
				       (string_of_ccsl_type ntyp_ex) ^
				       " for let var " ^
				       id_rec.id_token.token_name)
			in let _ = id_rec.id_type <- ntyp_ex 
			in
			  (id_rec, typeopt, recurse_exp ntyp_ex ex)
		     )
	      decl_list
	  in
	    Let(ndecl_list,
		recurse_exp expr_typ ex)

					       (* we are in check_expression *)
      | If(conds,else_ex) -> 
	  let nconds = 
	    List.map 
	      (fun (cond,ex) -> (recurse_form Bool cond,
				 recurse_exp expr_typ ex)
	      )
	      conds
	  in
	    If(nconds, 
	       recurse_exp expr_typ else_ex)

					       (* we are in check_expression *)
      | CCSL_Case(TypeAnnotation(ex,ex_typ), variants) -> 
	  let adt_type = check_ground (get_ex_loc ex) tv_bindings
			   ex_typ "expression"
	  in let _ = dc ("derive " ^
			 (string_of_ccsl_type adt_type) ^
			 " for case expr " ^
			 (string_of_ccsl_expression ex))
	  in let nex = recurse_exp adt_type ex in
	  let do_variant (const,args,ex) = 
	    let _ = List.iter
		      (fun id_rec ->
			 let nid_typ = (try
					  check_ground loc tv_bindings 
					    id_rec.id_type "INTERNAL ERROR"
					with
					  | Typecheck_error -> 
					      assert(false)
				       )
			 in 
			   id_rec.id_type <- nid_typ
		      )
		      args
	    in
	      (const,args,
	       recurse_exp expr_typ ex)
	  in
	  let nvariants = List.map do_variant variants 
	  in
	    CCSL_Case(
	      TypeAnnotation(nex, adt_type),
	      nvariants)

					       (* we are in check_expression *)

					(* CCSL_Case needs a TypeAnnotation *)
      | CCSL_Case _ -> 
	  assert(false)

      | Box(typ,pred,tlist) ->
	  let _ = assert( types_must_unify expr_typ (Function(typ,Bool)) )
	  in
	    Box(typ,
		recurse_exp expr_typ pred,
		tlist)
    

      | Diamond(typ,pred,tlist) ->
	  let _ = assert( types_must_unify expr_typ (Function(typ,Bool)) )
	  in
	    Diamond(typ,
		    recurse_exp expr_typ pred,
		    tlist)

      | Expression form ->
	  Expression(recurse_form expr_typ form)

					       (* we are in check_expression *)

			(* type annotations are already catched above *)
      | TypeAnnotation _
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
(* 	 | Reduce _
 *)
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
	  in 
	    nexpression


and check_formula ccl loc tv_bindings formula_type formula =
  let recurse_exp = check_expression ccl loc tv_bindings in
  let recurse_form = check_formula ccl loc tv_bindings in
  let _ = assert(not (type_is_nonground formula_type)) in
  let nformula = 
    match formula with
      | FormLoc(f,l)    -> 
	  FormLoc(check_formula ccl l tv_bindings formula_type f, l)
      | True 		-> assert(types_must_unify formula_type Bool); formula
      | False 		-> assert(types_must_unify formula_type Bool); formula
      | Not f 		-> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Not(recurse_form formula_type f)
	  end
      | And f_list 	-> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    And(List.map (recurse_form formula_type) f_list)
	  end
      | Or f_list 	-> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Or(List.map (recurse_form formula_type) f_list)
	  end
						  (* we are in check_formula *)
      | Implies(assum,concl) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Implies( recurse_form formula_type assum, 
		     recurse_form formula_type concl)
	  end
      | Iff(assum,concl) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Iff( recurse_form formula_type assum, 
		 recurse_form formula_type concl)
	  end
      | Equal( TypeAnnotation(ex_a,eq_typ), ex_b) -> 
	  let _ = assert(types_must_unify formula_type Bool) in
	  let neq_typ = check_ground loc tv_bindings eq_typ 
			  "both sides of equality"
	  in let _ = dc ("derive " ^
			 (string_of_ccsl_type neq_typ) ^
			 " for equality")
	  in
	    Equal( TypeAnnotation(recurse_exp neq_typ ex_a, eq_typ),
		   recurse_exp neq_typ ex_b)

					(* Equal needs TypeAnnotation *)
      | Equal _ -> assert(false)

      | Forall(quant_list, f) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Forall(quant_list, recurse_form formula_type f)
	  end
						  (* we are in check_formula *)
      | Exists(quant_list, f) -> 
	  begin
	    assert(types_must_unify formula_type Bool);	    
	    Exists(quant_list, recurse_form formula_type f)
	  end
      | Formula ex -> 
	  Formula(recurse_exp formula_type ex)

						  (* we are in check_formula *)

					(* t equals None *)
      | Obseq(t, TypeAnnotation(ex_a,eq_typ), ex_b) ->		
	  let _ = assert(types_must_unify formula_type Bool) in
	  let neq_typ = check_ground loc tv_bindings eq_typ 
			  "both sides of equality" 
	  in let _ = dc ("derive " ^
			 (string_of_ccsl_type neq_typ) ^
			 " for beh equality")
					(* check for greatest bisim *)
	  in let violation = ref false in
	  let _ = iter_components Pos 
		    (fun _ typ -> 
		       let args = match typ with
			 | Class(_,args) -> args
			 | Adt(_,_,args) -> args
			 | Groundtype(_,args) -> args
			 | _ -> assert(false)
		       in
			 if (not (type_has_feature HasRelLiftFeature typ)) &&
			   (count_self_args args <> 0)
			 then violation := true
		    ) neq_typ
	  in let _ = (if (count_self neq_typ) <> 0 && 
			(not (ccl#has_feature HasGreatestBisimFeature))
		      then violation := true)
	  in let _ = 
	      if !violation then
		if !expert_mode then
		  warning_message loc 
		    ("Observational equivalence for type " ^
		     (string_of_ccsl_type neq_typ) ^
		     " is not well defined.")
		else
		  begin
		    error_message loc 
		      ("Observational equivalence for type " ^
		       (string_of_ccsl_type neq_typ) ^
		       " is not well defined.");
		    raise Typecheck_error
		  end

						  (* we are in check_formula *)
	  in let obsname =
	      (match neq_typ with
					(* Self gets Bisim later *)
		 | Self -> ""
					(* gets bisim for that class *)
		 | Class(cl,[]) -> ""
		 | neq_typ ->
		     try
		       ccl#find_rel_lifting_for_type neq_typ
		     with
		       | Not_found ->
			   let tname = generate_lifting_name neq_typ in
			   let _ = ccl#add_rel_lifting tname neq_typ 
			   in
			     tname
	      )
	  in
	    Obseq( Some(obsname, neq_typ), 
		   recurse_exp neq_typ ex_a, 
		   recurse_exp neq_typ ex_b)

						  (* we are in check_formula *)

					(* Obseq needs TypeAnnotation *)
      | Obseq _ -> assert(false)

					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | MetaForall _
      | Bisim _
      | ConstantPredicate _
(* 	 | LessOrEqual _
 *)
	-> assert(false)
  in 
    nformula
    


let check_assertion_formula ccl loc tv_bindings formula =
  let nformula = check_formula ccl loc tv_bindings Bool formula 
  in
    nformula


let check_definition_expression ccl loc tv_bindings typ expr =
  let nex = check_expression ccl loc tv_bindings typ expr
  in
    nex



(***********************************************************************
 ***********************************************************************
 *
 * Type derivation
 *
 *)

let rec derive_expression ccl loc tv_bindings expr = 
  let recurse_exp = derive_expression ccl loc tv_bindings in
  let recurse_form = derive_formula ccl loc tv_bindings in
  let do_unify = unify_types tv_bindings in
  let nexpression, ntype = 
    match expr with
      | ExprLoc(ex,l) -> 
	  let _ = dr ("ExprLoc " ^ (string_of_loc l)) in
	  let nex, typ_ex = derive_expression ccl l tv_bindings ex 
	  in
	    (ExprLoc(nex, l),
	     typ_ex)
      | TypeAnnotation(BasicExpr(Member _), typ) ->
	  expr, typ
      | BasicExpr(TermVar(Resolved(id_rec))) -> 
	  expr, id_rec.id_type
				(* other BasicExpr cannot appear here *)
      | BasicExpr(TermVar(Unresolved _))
      | BasicExpr(Member _) -> 
	  assert(false)

      | TypedTerm(Term _, typ) -> 
	  let _ = dr "typed ground term" 
	  in
	    (expr, typ)

      | TypedTerm(ex, typ) -> 
	  let _ = dr "TypedTerm" in
	  let nex, typ_ex = recurse_exp ex in
	  let utyp_ex = do_unify typ_ex typ (mismatch loc)
	  in
	    (TypedTerm(nex, utyp_ex),
	     utyp_ex)

					      (* we are in derive_expression *)
      | MethodSelection(ex, instiface, m) ->
	  let _ = dr "MethodSelection" in
	  let nex, typ_ex = recurse_exp ex in
	  let utyp = match instiface with
	    | InstIface (iface,args,instloc) -> 
		assert(iface#is_class);
		do_unify typ_ex (Class(iface,args)) (wrong_object_type loc)
	    | CurrentIface ->
		do_unify typ_ex Self (wrong_object_type loc)
	    | NoIface -> typ_ex 
	  in
	  let nutyp = normalize_type tv_bindings utyp in
	  let mtoken = match m with
	    | Unresolved t -> t
	    | Resolved _ -> assert(false) in
	  let _ = match nutyp with
	    | Self			
	    | Class _			(* all ok *)
	    | FreeTypeVariable _ -> ()
					(* incompatible *)
	    | _ ->
		begin
		  error_message loc
		    ("Subject of method selection has type\n" ^
		     (string_of_ccsl_type nutyp) ^
		     "\nThis type is not compatible with any object type."
		    );
		  raise Typecheck_error
		end
					      (* we are in derive_expression *)
	  in let _ = 
	      if type_is_nonground nutyp 
	      then
		begin
		  error_message loc(
		    "Cannot determine type for subject of method selection. "^
		    "Derived " ^
		    (string_of_ccsl_type nutyp) ^
		    "\nPlease use a qualified method name or " ^
		    "add type constraints."
		  );
		  raise Typecheck_error
		end
	  in let cl = match nutyp with
	    | Self -> ccl
	    | Class(cl,_) -> cl 
	    | _ -> assert(false)
	  in let m = 
	    try cl#find_member mtoken.token_name
	    with | Member_not_found -> undefined_method cl mtoken 
		     (remove_option mtoken.loc) in
	  let _ = (if not m#is_action 
		   then undefined_method cl mtoken
		     (remove_option mtoken.loc)) in
					      (* we are in derive_expression *)
	  let typ,ninstiface = match nutyp with
	    | Self -> (m#get_curried_type, CurrentIface)
	    | Class(cl,args) -> 
		(snd( export_member_with_args cl m args m#get_curried_type ),
		 InstIface(cl, args, None))
	    | _ -> assert(false)
	  in let _ = dr ("Resolve " ^ m#get_name ^ ": " ^
			 m#hosting_class#get_name)
	  in
	    (MethodSelection(
	       TypeAnnotation(nex, nutyp),
	       ninstiface, Resolved m),
	     typ)

      | Tuple(ex_list) -> 
	  let _ = dr "Tuple" in
	  let nex_list, typ_list = 
	    List.split (List.map recurse_exp ex_list)
	  in
	    (Tuple(nex_list), Product(typ_list))

		(* hack for projections, 
		 * this branch is taken when the projection appears 
		 * in an application 
		 *)
					      (* we are in derive_expression *)
      | Application(ExprLoc(Projection(i,_), ploc) as proj_ex, ex2) ->
	  let _ = dr "ProjAppl" in
	  let nex2, typ_ex2 = recurse_exp ex2 in
	  let ntyp_ex2 = normalize_type tv_bindings typ_ex2 
	  in
	    (match ntyp_ex2 with
	       | FreeTypeVariable _ -> 
		   let res_type = FreeTypeVariable(Type_variable.fresh()) in
		   let proj_type = Function(ntyp_ex2, res_type) 
		   in
		     (Application(
			TypeAnnotation( proj_ex, proj_type ),
			nex2),
		      res_type)
	       | Product domtlist -> 
		   let _ = if i > List.length domtlist then
		     begin
		       proj_i_error proj_ex loc ntyp_ex2;
		       raise Typecheck_error
		     end
		   in let res_type = (List.nth domtlist (i - 1)) in
		   let proj_type = Function(ntyp_ex2, res_type) 
		   in
		     (Application(
			TypeAnnotation( proj_ex, proj_type ),
			nex2),
		      res_type)
	       | _ -> 
		   begin
		     proj_dom_error loc ntyp_ex2;
		     raise Typecheck_error
		   end
	    )

					      (* we are in derive_expression *)
      | Projection(i,_) -> 
	  let _ = dr "Projection" in
	  let typ = 
	    Function(
	      FreeTypeVariable(Type_variable.fresh()),
	      FreeTypeVariable(Type_variable.fresh()))
	  in
	    (expr,
	     typ)	    
      | Abstraction(decl_list,ex) ->
	  let _ = dr "Abstraction" in
	  let nex,typ_ex = recurse_exp ex in
	  let decl_types = List.map snd decl_list in
	  let dom_typ = match decl_types with
	    | [t] -> t
	    | tl -> Product(tl) 
	  in
	    (Abstraction(decl_list, nex),
	     Function(dom_typ, typ_ex))
					      (* we are in derive_expression *)
      | Application(ex1,ex2) -> 
	  let _ = dr "Application" in
	  let nex1, typ_ex1 = recurse_exp ex1 in
	  let nex2, typ_ex2 = recurse_exp ex2 in
	  let ntyp_ex1 = do_unify typ_ex1
			   (Function(
			      FreeTypeVariable(Type_variable.fresh()),
			      FreeTypeVariable(Type_variable.fresh())))
			   (is_not_fun "It cannot be applied" 
			      (get_ex_loc ex1))
	  in let dom_type, codom_type = match ntyp_ex1 with
	    | Function(dt,ct) -> dt,ct
	    | _ -> assert(false)
	  in let ntyp_ex2 = do_unify dom_type typ_ex2 (dom_mismatch loc)
	  in
	    (Application(
	       TypeAnnotation(nex1, ntyp_ex1),
	       nex2),
	     codom_type)
					      (* we are in derive_expression *)
      | TypeAnnotation(InfixApplication(ex1,instiface,mem,ex2), memtype) ->
	  let _ = dr "Infix" in
	  let nex1, typ_ex1 = recurse_exp ex1 in
	  let nex2, typ_ex2 = recurse_exp ex2 in
	  let lefttype, righttype, resulttype =
	    (match memtype with
	       | Function(Product [t1;t2], t3) -> t1, t2, t3
	       | Function(t1, Function(t2,t3)) -> t1, t2, t3
	       | _ -> assert(false)
	    )
	  in let nlefttype = do_unify typ_ex1 lefttype 
			       (mismatch (get_ex_loc ex1))
	  in let nrighttype = do_unify typ_ex2 righttype 
			       (mismatch (get_ex_loc ex2))
	  in
	    (TypeAnnotation(
	       InfixApplication(nex1,instiface,mem,nex2),
	       memtype),
	     resulttype)

			(* InfixApplication is always under TypeAnnotation *)
      | InfixApplication _ -> assert(false)

					      (* we are in derive_expression *)
      | FunUpdate(fex, changes) -> 
	  let _ = dr "FunUpdate" in
	  let nfex, typ_f = recurse_exp fex in
	  let ntyp_f = do_unify typ_f
			 (Function(
			    FreeTypeVariable(Type_variable.fresh()),
			    FreeTypeVariable(Type_variable.fresh())))
			 (is_not_fun "It cannot occur in an update expression"
			    (get_ex_loc fex))
	  in let dom_type, codom_type = match ntyp_f with
	    | Function(dt,ct) -> dt,ct
	    | _ -> assert(false)
	  in let nchanges = 
	      List.map (fun (ex1,ex2) ->
			  let nex1,typ_ex1 = recurse_exp ex1 in
			  let nex2,typ_ex2 = recurse_exp ex2 in
			  let ntyp_ex1 = do_unify typ_ex1 dom_type
					   (mismatch (get_ex_loc ex1)) in
			  let ntyp_ex2 = do_unify typ_ex2 codom_type
					   (mismatch (get_ex_loc ex2)) in
			    (nex1, nex2)
		       ) changes 
	  in
	    (FunUpdate( nfex, nchanges), 
	     ntyp_f)

					      (* we are in derive_expression *)
      | Let(decl_list, ex) ->
	  let _ = dr "Let" in
					(* Let binds sequentially *)
	  let ndecl_list =
	    List.map (fun (id_rec, typopt, ex) ->
			let nex, typ_ex = recurse_exp ex in
			let ntyp_ex = do_unify typ_ex id_rec.id_type
					(mismatch (get_ex_loc ex)) in
			let _ = id_rec.id_type <- ntyp_ex
			in
			  (id_rec, typopt, nex)
		     ) decl_list 
	  in let nex,typ_ex = recurse_exp ex 
	  in
	    (Let(ndecl_list, nex),
	     typ_ex)

					      (* we are in derive_expression *)
      | If(conds,else_ex) -> 
	  let _ = dr "If" in
	  let res_type = FreeTypeVariable(Type_variable.fresh()) in
	  let nconds = 
	    List.map 
	      (fun (cond,ex) -> 
		 let ncond,typ_cond = recurse_form cond in
		 let nex, typ_ex = recurse_exp ex in
		 let ntyp_cond = do_unify typ_cond Bool 
				   (types_dont_match (get_form_loc ncond)) in
		 let ntyp_ex = do_unify typ_ex res_type
				 (if_mismatch "if" (get_ex_loc nex))
		 in
		   (ncond, nex)
	      ) conds
	  in
	  let nelse_ex, typ_else_ex = recurse_exp else_ex in
	  let ntyp_else_ex = do_unify typ_else_ex res_type
			       (if_mismatch "if" (get_ex_loc nelse_ex)) 
	  in
	    (If(nconds, nelse_ex), ntyp_else_ex)

					      (* we are in derive_expression *)
      | CCSL_Case(TypeAnnotation(ex,ex_typ), variants) -> 
	  let _ = dr "CCSL_Case" in
	  let nex, nex_typ = recurse_exp ex in
	  let nnex_typ = do_unify nex_typ ex_typ 
			   (case_mismatch loc) in
	  let res_type = FreeTypeVariable(Type_variable.fresh()) in
	  let nvariants = 
	    List.map (function 
			| (Resolved(m) as const, args, cex) ->
			    let _ = dr ("case " ^ m#get_name) in
			    let ncex, typ_cex = recurse_exp cex in
			    let ntyp_cex = do_unify res_type typ_cex 
					(if_mismatch "case" (get_ex_loc cex))
			    in
			      (const, args, ncex)
			| _ -> assert(false)
		     ) variants
	  in
	    (CCSL_Case(TypeAnnotation(nex,nnex_typ), nvariants),
	     res_type)

					(* CCSL_Case needs a TypeAnnotation *)
      | CCSL_Case _ -> assert(false)

					      (* we are in derive_expression *)
      | Box(typ,pred,tlist) ->
	  let _ = dr "Box" in
	  let npred, typ_pred = recurse_exp pred in
	  let ntyp_pred = do_unify typ_pred (Function(typ,Bool)) 
			    (types_dont_match (get_ex_loc npred))
	  in
	    (Box(typ,npred,tlist), ntyp_pred)

      | Diamond(typ,pred,tlist) ->
	  let _ = dr "Diamond" in
	  let npred, typ_pred = recurse_exp pred in
	  let ntyp_pred = do_unify (Function(typ,Bool)) typ_pred 
			    (types_dont_match (get_ex_loc npred))
	  in
	    (Diamond(typ,npred,tlist), ntyp_pred)

      | Expression form -> 
	  let _ = dr "Expression" in
	  let nform, typ_form = recurse_form form 
	  in
	    (Expression(nform), typ_form) 

					      (* we are in derive_expression *)

			(* type annotations are already catched above *)
      | TypeAnnotation _
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
(* 	 | Reduce _
 *)
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
  in 
  let nntype = (* normalize_type tv_bindings *) ntype 
  in let _ = 
      if not (is_expr_loc nexpression) then
	dr ("derive " ^ (string_of_ccsl_type nntype) ^ " for " ^
	    (string_of_ccsl_expression nexpression))
  in
    (nexpression, nntype)


and derive_formula ccl loc tv_bindings formula =
  let recurse_exp = derive_expression ccl loc tv_bindings in
  let recurse_form = derive_formula ccl loc tv_bindings in
  let do_unify = unify_types tv_bindings in
  let nformula,ntype = 
    match formula with
      | FormLoc(f,l)    -> 
	  let _ = dr ("FormLoc " ^ (string_of_loc l)) in
	  let nf,typ_f = derive_formula ccl l tv_bindings f 
	  in
	    (FormLoc(nf, l),
	     typ_f)
      | True 		-> (True, Bool)
      | False 		-> (False, Bool)
      | Not f -> 
	  let _ = dr "Not" in
	  let nf,typ_f = recurse_form f in
	  let utyp_f = do_unify typ_f Bool (types_dont_match (get_form_loc nf))
	  in
	    (Not nf, Bool)

						 (* we are in derive_formula *)
      | And f_list -> 
	  let _ = dr "And" in
	  let nftype_list = List.map recurse_form f_list in
	  let utyp_f_list = 
	    List.map 
	      (fun (f,t) -> do_unify t Bool (types_dont_match (get_form_loc f))) 
	      nftype_list 
	  in
	  let nf_list, typ_f_list = List.split nftype_list 
	  in
	    (And nf_list, Bool)
      | Or f_list -> 
	  let _ = dr "Or" in
	  let nftype_list = List.map recurse_form f_list in
	  let utyp_f_list = 
	    List.map 
	      (fun (f,t) -> do_unify t Bool (types_dont_match (get_form_loc f))) 
	      nftype_list 
	  in
	  let nf_list, typ_f_list = List.split nftype_list 
	  in
	    (Or nf_list, Bool)

						 (* we are in derive_formula *)
      | Implies(assum,concl) -> 
	  let _ = dr "Implies" in
	  let nassum, typ_assum = recurse_form assum in
	  let nconcl, typ_concl = recurse_form concl in
	  let utyp_assum = do_unify typ_assum Bool
			     (types_dont_match (get_form_loc assum)) in
	  let utyp_concl = do_unify typ_concl Bool 
			     (types_dont_match (get_form_loc concl))
	  in
	    (Implies( nassum, nconcl), Bool)
      | Iff(assum,concl) -> 
	  let _ = dr "Iff" in
	  let nassum, typ_assum = recurse_form assum in
	  let nconcl, typ_concl = recurse_form concl in
	  let utyp_assum = do_unify typ_assum Bool 
			     (types_dont_match (get_form_loc assum)) in
	  let utyp_concl = do_unify typ_concl Bool 
			     (types_dont_match (get_form_loc concl))
	  in
	    (Iff(nassum, nconcl), Bool)
      | Equal(ex_a,ex_b) -> 
	  let _ = dr "Equal" in
	  let nex_a, typ_a = recurse_exp ex_a in
	  let nex_b, typ_b = recurse_exp ex_b in
	  let utyp_ab = do_unify typ_a typ_b (eq_not_equal loc)
	  in
	    (Equal(
	       TypeAnnotation(nex_a, utyp_ab), nex_b),
	     Bool)
						 (* we are in derive_formula *)
      | Forall(quant_list, f) ->
	  let _ = dr "Forall" in
	  let nf,typ_f = recurse_form f in
	  let utyp_f = do_unify typ_f Bool (types_dont_match (get_form_loc f))
	  in
	    (Forall( quant_list, nf), Bool)
      | Exists(quant_list, f) -> 
	  let _ = dr "Exists" in
	  let nf,typ_f = recurse_form f in
	  let utyp_f = do_unify typ_f Bool (types_dont_match (get_form_loc f))
	  in
	    (Exists( quant_list, nf), Bool)
      | Formula ex -> 
	  let _ = dr "Formula" in
	  let nex, typ_ex = recurse_exp ex
	  in
	    (Formula nex, typ_ex)
						 (* we are in derive_formula *)
      | Obseq(t,ex_a,ex_b) ->		(* t equals None *)
	  let _ = dr "Obseq" in
	  let nex_a, typ_a = recurse_exp ex_a in
	  let nex_b, typ_b = recurse_exp ex_b in
	  let utyp_ab = do_unify typ_a typ_b (eq_not_equal loc)
	  in
	    (Obseq(t, TypeAnnotation(nex_a, utyp_ab), nex_b),
	     Bool)
					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | MetaForall _
      | Bisim _
      | ConstantPredicate _
(* 	 | LessOrEqual _
 *)
	-> assert(false)
  in 
  let nntype = (* normalize_type tv_bindings *) ntype 
  in let _ = 
      if not (is_form_loc nformula) then
	dr ("derive " ^ (string_of_ccsl_type nntype) ^ " for " ^
	    (string_of_ccsl_formula nformula))
  in
    (nformula, nntype)
    


let derive_assertion_formula ccl loc tv_bindings formula =
  let nformula, typ_f = derive_formula ccl loc tv_bindings formula in
  let utyp_f = unify_types tv_bindings typ_f Bool (types_dont_match loc)
  in
    nformula


let derive_definition_expression ccl loc tv_bindings typ expr =
  let nex, typ_ex = derive_expression ccl loc tv_bindings expr in
  let utyp_ex = unify_types tv_bindings typ_ex typ (types_dont_match loc) in
  let _ = assert(types_must_unify utyp_ex typ)
  in
    nex


(***********************************************************************
 ***********************************************************************
 *
 * definition checking
 *
 *)

let rec check_valid_definition_expr loc expr = 
  let recurse_exp = check_valid_definition_expr loc in
  let recurse_form = check_valid_definition_form loc in
    match expr with
      | ExprLoc(ex,loc) -> check_valid_definition_expr loc ex
      | BasicExpr bexpr -> ()
      | TypedTerm(Term _, typ) -> ()
      | TypedTerm(ex, typ) -> recurse_exp ex
      | MethodSelection(ex, typeopt, m) -> recurse_exp ex
      | Tuple(ex_list) -> List.iter recurse_exp ex_list
      | Projection(i,_) -> ()
      | Abstraction(decl_list,ex) -> recurse_exp ex
      | Application(ex1,ex2) -> recurse_exp ex1; recurse_exp ex2
      | InfixApplication(ex1, iface, tokcontainer, ex2) ->
	  recurse_exp ex1; recurse_exp ex2 
      | FunUpdate(fex, changes) -> 
	  recurse_exp fex;
	  List.iter (fun (ex1,ex2) -> (recurse_exp ex1; recurse_exp ex2))
	    changes
      | Let(decl_list, ex) ->
	  List.iter (fun (id_rec, typopt, ex) -> recurse_exp ex)
	    decl_list;
	  recurse_exp ex
      | If(conds,ex) -> 
	  List.iter (fun (f,ex) -> (recurse_form f; recurse_exp ex)) conds;
	  recurse_exp ex
      | CCSL_Case(ex,variants) -> 
	  recurse_exp ex;
	  List.iter (fun (mem,ids,exp) -> recurse_exp exp) variants
      | Box(typ, pred, tlist) ->
	  (match typ with
	     | Self -> invalid_definition loc "Modal operator"
	     | _ -> recurse_exp pred
	  )

      | Diamond(typ, pred, tlist) -> 
	  (match typ with
	     | Self -> invalid_definition loc "Modal operator"
	     | _ -> recurse_exp pred
	  )

      | Expression form -> recurse_form form
      | TypeAnnotation(exp,typ) -> recurse_exp exp
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
	
and check_valid_definition_form loc formula =
  let recurse_exp = check_valid_definition_expr loc in
  let recurse_form = check_valid_definition_form loc in
    match formula with
      | FormLoc(f,loc) -> check_valid_definition_form loc f
      | True -> ()
      | False -> ()
      | Not f -> recurse_form f
      | And f_list -> List.iter recurse_form f_list
      | Or f_list -> List.iter recurse_form f_list
      | Implies(assum,concl) -> 
	  recurse_form assum; 
	  recurse_form concl
      | Iff(assum,concl) -> 
	  recurse_form assum;
	  recurse_form concl
      | Equal(ex_a,ex_b) ->
	  recurse_exp ex_a; 
	  recurse_exp ex_b
      | Forall(decll, f) -> recurse_form f
      | Exists(decll, f) -> recurse_form f
      | Formula ex -> recurse_exp ex
      | Obseq(Some(obsname, eq_typ), ex1,ex2) -> 
	  if count_self eq_typ <> 0 
	  then
	    invalid_definition loc "Observational equality involving Self"

      | Obseq _ 
					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | MetaForall _
      | Bisim _
      | ConstantPredicate _
	-> assert(false)







(***********************************************************************
 ***********************************************************************
 *
 * Assertions and Classes
 *
 *)


let typecheck_definition ccl def =
  match def.definition with 
    | Symbolic(FormLoc(Formula(TypeAnnotation(defex, righttyp)), loc)) ->
	let mem = def.defined_method in
	let _ = d ("  * TC definition " ^ mem#get_name) in
	let tv_bindings = new_tv_binding () in
	let ex1 = derive_definition_expression ccl loc tv_bindings 
		    righttyp defex in
	let ex2 = check_definition_expression ccl loc tv_bindings 
		    righttyp ex1
	in
	let _ = (if not !expert_mode 
		 then check_valid_definition_expr loc ex2
		 else ())
	in
	  def.definition <- 
	  Symbolic(FormLoc(Formula(TypeAnnotation(ex2, righttyp)), loc))
	      
    | Symbolic _ -> assert false
	  
    | Pvs_String _
    | Isa_String _ -> ()



let typecheck_assertion ccl assertion =
  if not assertion.is_generated 
  then
    match assertion.assertion_formula with
      | Symbolic(FormLoc(formula,loc)) -> 
	  let _ = d ("  * TC assertion " ^ assertion.assertion_name.token_name)
	  in let tv_bindings = new_tv_binding () in
	  let form1 = derive_assertion_formula ccl loc tv_bindings formula in
	  let form2 = check_assertion_formula ccl loc tv_bindings form1
	  in
	    assertion.assertion_formula <- Symbolic(FormLoc(form2,loc))
	      
      | Symbolic _ -> assert(false)
	  
      | Pvs_String _
      | Isa_String _ -> ()


let typecheck_class ccl =
  d (" ** TC " ^ ccl#get_name);
  List.iter (typecheck_definition ccl) ccl#get_definitions;
  List.iter (typecheck_assertion ccl) ccl#get_assertions;
  List.iter (typecheck_assertion ccl) ccl#get_creations;
  List.iter (typecheck_assertion ccl) ccl#get_theorems

let typecheck_sig si =
  d (" ** TC " ^ si#get_name);
  List.iter (typecheck_definition si) si#get_definitions


let typecheck_ast = function
  | CCSL_class_dec cl -> typecheck_class cl
  | CCSL_adt_dec adt -> ()
  | CCSL_sig_dec si -> typecheck_sig si


let ccsl_typecheck_class_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter typecheck_ast ast;;



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

