(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: assoc.v,v 1.11 2013/04/10 12:06:16 tews Exp $
 *)

(** ** association lists 

      This module defines the well-known [assoc] and [rassoc], which
      associates from right to left.
*)

Require Export list_support.

Section Assoc.
  Variable A B : Type.
  Variable aeq : eq_type A.
  Variable beq : eq_type B.

  (***************************************************************************)
  (** *** Basic definitions and lemmas *)
  (***************************************************************************)

  Fixpoint assoc(l : list (A * B))(a : A) : option B := 
    match l with
      | [] => None
      | ab :: l => if aeq a (fst ab) then Some (snd ab) else assoc l a
    end.

  Fixpoint rassoc(l : list (A * B))(b : B) : option A := 
    match l with
      | [] => None
      | ab :: l => if beq b (snd ab) then Some (fst ab) else rassoc l b
    end.

  Lemma assoc_cons_split :
    forall(P : option B -> Prop)(a1 a2 : A)(b : B)(l : list (A * B)),
      (a1 = a2 -> P (Some b)) ->
      (a1 <> a2 -> P (assoc l a2)) ->
        P (assoc ((a1, b) :: l) a2).
  Proof.
    intros P a1 a2 b l H H0.
    simpl.
    destruct (aeq a2 a1).
      apply H.
      auto.
    apply H0.
    auto.
  Qed.

  Lemma assoc_cons_first :
    forall(a : A)(b : B)(l : list (A * B)),
      assoc ((a, b) :: l) a = Some b.
  Proof.
    intros a b l.
    apply assoc_cons_split.
      trivial.
    intros H.
    exfalso.
    apply H.
    trivial.
  Qed.

  Lemma assoc_cons_tail :
    forall(a1 a2 : A)(b : B)(l : list (A * B)),
      a1 <> a2 -> 
        assoc ((a1, b) :: l) a2 = assoc l a2.
  Proof.
    intros a1 a2 b l H.
    apply assoc_cons_split.
      contradiction.
    trivial.
  Qed.


  Lemma rassoc_cons_split :
    forall(P : option A -> Prop)(a : A)(b1 b2 : B)(l : list (A * B)),
      (b1 = b2 -> P (Some a)) ->
      (b1 <> b2 -> P (rassoc l b2)) ->
        P (rassoc ((a, b1) :: l) b2).
  Proof.
    intros P a b1 b2 l H H0.
    simpl.
    destruct (beq b2 b1).
      apply H.
      auto.
    apply H0.
    auto.
  Qed.

  Lemma rassoc_cons_first :
    forall(a : A)(b : B)(l : list (A * B)),
      rassoc ((a, b) :: l) b = Some a.
  Proof.
    intros a b l.
    apply rassoc_cons_split.
      trivial.
    intros H.
    exfalso.
    auto.
  Qed.

  Lemma rassoc_cons_tail :
    forall(a : A)(b1 b2 : B)(l : list (A * B)),
      b1 <> b2 -> 
        rassoc ((a, b1) :: l) b2 = rassoc l b2.
  Proof.
    intros a b1 b2 l H.
    apply rassoc_cons_split.
      contradiction.
    trivial.
  Qed.

  Lemma rassoc_assoc_none : 
    forall(l : list (A * B))(a : A)(b : B),
      rassoc l b = None ->
        assoc l a <> Some b.
  Proof.
    induction l.
      intros a b H H0.
      simpl in *.
      discriminate.
    destruct a as [a1 b1].
    intros a2 b2 H.
    apply assoc_cons_split.
      intros H0 H1.
      inversion H1; clear H1.
      revert H.
      subst a2 b2.
      rewrite rassoc_cons_first.
      discriminate.
    intros H0.
    apply IHl.
    revert H.
    apply rassoc_cons_split.
      discriminate.
    trivial.
  Qed.

  Lemma assoc_rassoc_some :
    forall(l : list (A * B))(a : A)(b : B),
      assoc l a = Some b ->
        is_some (rassoc l b).
  Proof.
    intros l a b H.
    destruct (rassoc l b) eqn:?.
      simpl.
      trivial.
    exfalso.
    eapply rassoc_assoc_none in Heqo.
    apply Heqo.
    eexact H.
  Qed.

  Lemma assoc_append_split :
    forall(P : option B -> Prop)(l1 l2 : list (A * B))(a : A),
      (is_some (assoc l1 a) -> P (assoc l1 a)) ->
      (is_none (assoc l1 a) -> P (assoc l2 a)) ->
        P (assoc (l1 ++ l2) a).
  Proof.
    induction l1.
      intros l2 a H H0.
      simpl in *.
      apply H0.
      trivial.
    destruct a as [a1 b1].
    intros l2 a2 H H0.
    simpl (((a1, b1) :: l1) ++ l2).
    apply assoc_cons_split.
      intros H1.
      subst a2.
      rewrite assoc_cons_first in H.
      apply H.
      simpl.
      trivial.
    intros H1.
    apply IHl1.
      intros H2.
      rewrite assoc_cons_tail in H.
        auto.
      trivial.
    intros H2.
    apply H0.
    rewrite assoc_cons_tail.
      trivial.
    trivial.
  Qed.

  Lemma assoc_map_split :
    forall(C : Type)(f : C -> A * B)(l : list C)(a : A),
      (assoc (map f l) a = None /\ forall(c : C), In c l -> fst (f c) <> a)
      \/
      exists(c : C), In c l /\ 
        fst (f c) = a /\ assoc (map f l) a = Some (snd (f c)).
  Proof.
    induction l.
      intros a.
      left.
      split.
        trivial.
      intros c H.
      contradiction.
    rename a into c.
    intros a.
    simpl (map f (c :: l)).
    destruct (f c) as [fca fcb] eqn:?.
    apply assoc_cons_split.
      intros H.
      clear IHl.
      right.
      exists c.
      rewrite Heqp.
      repeat split; trivial.
      left.
      trivial.
    intros H.
    specialize (IHl a).
    decompose [ex and or] IHl; clear IHl.
      left.
      split.
        trivial.
      intros c0 H0.
      destruct H0.
        subst c0.
        rewrite Heqp.
        trivial.
      auto.
    right.
    exists x.
    repeat split; trivial.
    right.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Injective association lists *)
  (***************************************************************************)

  Definition injective_assoc(l : list (A * B)) : Prop :=
    forall(a1 a2 : A),
      assoc l a1 = assoc l a2 ->
      is_some (assoc l a1) ->
        a1 = a2.

  Lemma injective_assoc_empty : injective_assoc [].
  Proof.
    unfold injective_assoc in *.
    intros a1 a2 H H0.
    simpl in *.
    contradiction.
  Qed.

  Lemma injective_assoc_cons_different :
    forall(l : list (A * B))(a : A)(b : B),
      injective_assoc l ->
      (forall(a : A), assoc l a <> Some b) ->
        injective_assoc ((a, b) :: l).
  Proof.
    unfold injective_assoc in *.
    intros l a b H H0 a1 a2.
    apply assoc_cons_split.
      apply assoc_cons_split.
        intros H1 H2 H3 H4.
        subst a1 a2.
        trivial.
      intros H1 H2 H3 H4.
      specialize (H0 a2).
      apply eq_sym in H3.
      contradiction.
    apply assoc_cons_split.
      intros H1 H2 H3 H4.
      specialize (H0 a1).
      contradiction.
    intros H1 H2.
    apply H.
  Qed.

  Lemma injective_assoc_cons_rassoc :
    forall(l : list (A * B))(a : A)(b : B),
      injective_assoc l ->
      rassoc l b = None ->
        injective_assoc ((a, b) :: l).
  Proof.
    intros l a b H H0.
    apply injective_assoc_cons_different.
      trivial.
    intros a0.
    apply rassoc_assoc_none.
    trivial.
  Qed.

  Lemma injective_assoc_append :
    forall(l1 l2 : list (A * B)),
      injective_assoc l1 ->
      injective_assoc l2 ->
      (forall(a1 a2 : A),
             is_some (assoc l1 a1) ->
               assoc l1 a1 <> assoc l2 a2) ->
        injective_assoc (l1 ++ l2).
  Proof.
    intros l1 l2 H H0 H1 a1 a2.
    apply assoc_append_split.
      apply assoc_append_split.
        intros H2 H3 H4 H5.
        apply H.
          trivial.
        trivial.
      intros H2 H3 H4 H5.
      exfalso.
      eapply H1.
        eexact H3.
      eexact H4.
    apply assoc_append_split.
      intros H2 H3 H4 H5.
      exfalso.
      eapply H1.
        eexact H2.
      apply eq_sym.
      eexact H4.
    intros H2 H3 H4 H5.
    apply H0.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Functional association lists *)
  (***************************************************************************)

  Inductive assoc_mapping : list (A * B) -> Prop :=
    | assoc_mapping_nil : assoc_mapping []
    | assoc_mapping_cons : 
        forall(a : A)(b : B)(l : list (A * B)),
          assoc_mapping l -> 
            is_none (assoc l a) -> assoc_mapping ((a, b) :: l).

  Lemma assoc_mapping_tail :
    forall(a : A)(b : B)(l : list (A * B)),
      assoc_mapping ((a, b) :: l) ->
        assoc_mapping l.
  Proof.
    clear. 
    intros a b l H.
    remember ((a, b) :: l) as abl.
    destruct H.
      discriminate.
    inversion Heqabl.
    subst l0.
    trivial.
  Qed.

  Lemma assoc_mapping_cons_first :
    forall(a : A)(b : B)(l : list (A * B)),
      assoc_mapping ((a, b) :: l) ->
        is_none (assoc l a).
  Proof.
    clear. 
    intros a b l H.
    remember ((a, b) :: l) as abl.
    destruct H.
      discriminate.
    inversion Heqabl; clear Heqabl.
    subst l0 a0.
    trivial.
  Qed.

  Lemma rassoc_assoc_some :
    forall(l : list (A * B))(a : A)(b : B),
      assoc_mapping l ->
      rassoc l b = Some a ->
        assoc l a = Some b.
  Proof.
    induction 1.
      intros H.
      discriminate.
    apply rassoc_cons_split.
      intros H1 H2.
      inversion H2; clear H2.
      subst a0 b0.
      rewrite assoc_cons_first.
      trivial.
    apply assoc_cons_split.
      intros H1 H2 H3.
      subst a0.
      apply IHassoc_mapping in H3.
      rewrite H3 in *.
      contradiction.
    intros H1 H2 H3.
    apply IHassoc_mapping.
    trivial.
  Qed.

  Lemma injective_assoc_tail :
    forall(a : A)(b : B)(l : list (A * B)),
      assoc_mapping ((a, b) :: l) ->
      injective_assoc ((a, b) :: l) ->
        injective_assoc l.
  Proof.
    intros a b l H H0 a1 a2 H1 H2.
    specialize (H0 a1 a2).
    revert H0.
    apply assoc_cons_split.
      intros H0 H3.
      clear H3.
      subst a1.
      remember ((a, b) :: l).
      destruct H.
        discriminate.
      inversion Heql0; clear Heql0.
      subst a0 b0 l0.
      exfalso.
      eapply option_contradiction.
        eexact H2.
      trivial.
    intros H0; clear H0.
    apply assoc_cons_split.
      intros H0 H3.
      clear H3.
      subst a2.
      remember ((a, b) :: l).
      destruct H.
        discriminate.
      inversion Heql0; clear Heql0.
      subst a0 b0 l0.
      rewrite H1 in *.
      exfalso.
      eapply option_contradiction.
        eexact H2.
      trivial.
    intros H0 H3.
    apply H3; trivial.
  Qed.


  Lemma assoc_rassoc_inj_some :
    forall(l : list (A * B))(a : A)(b : B),
      assoc_mapping l ->
      injective_assoc l ->
      assoc l a = Some b ->
        rassoc l b = Some a.
  Proof.
    induction l.
      discriminate.
    destruct a.
    intros a0 b0 H H0.
    apply assoc_cons_split.
      intros H1 H2.
      inversion H2; clear H2.
      subst a0 b0.
      rewrite rassoc_cons_first.
      trivial.
    intros H1 H2.
    apply rassoc_cons_split.
      intros H3.
      subst b0.
      exfalso.
      apply H1.
      apply H0.
        rewrite assoc_cons_first.
        apply assoc_cons_split.
          intros H3.
          contradiction.
        intros H3.
        auto.
      rewrite assoc_cons_first.
      simpl.
      trivial.
    intros H3.
    apply IHl.
        apply assoc_mapping_tail in H.
        trivial.
      apply injective_assoc_tail in H0.
        trivial.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Misc *)
  (***************************************************************************)

  Lemma assoc_In : forall(l : list (A * B))(a : A)(b : B),
    assoc l a = Some b -> In (a, b) l.
  Proof.
    induction l.
      intros a b H.
      discriminate.
    destruct a.
    intros a0 b0.
    apply assoc_cons_split.
      intros H H0.
      inversion H0; clear H0.
      subst a0 b0.
      left.
      trivial.
    intros H H0.
    right.
    apply IHl.
    trivial.
  Qed.

  Lemma assoc_In_rev : forall(l : list (A * B))(a : A)(b : B),
    assoc_mapping l ->
    In (a, b) l ->
      assoc l a = Some b.
  Proof.
    induction l.
      contradiction.
    destruct a.
    intros a0 b0 H H0.
    destruct H0.
      inversion H0; clear H0.
      subst a0 b0.
      rewrite assoc_cons_first.
      trivial.
    apply assoc_cons_split.
      intros H1.
      exfalso.
      subst a0.
      apply IHl in H0.
        apply assoc_mapping_cons_first in H.
        rewrite H0 in H.
        contradiction.
      apply assoc_mapping_tail in H.
      trivial.
    intros H1.
    apply IHl.
      apply assoc_mapping_tail in H.
      trivial.
    trivial.
  Qed.

  Lemma incl_assoc_some :
    forall(l1 l2 : list (A * B))(a : A),
      assoc_mapping l2 ->
      incl l1 l2 ->
      is_some (assoc l1 a) ->
        is_some (assoc l2 a).
  Proof.
    intros l1 l2 a H H0 H1.
    destruct (assoc l1 a) eqn:H2.
      apply assoc_In in H2.
      apply H0 in H2.
      apply assoc_In_rev in H2.
        rewrite H2.
        trivial.
      trivial.
    contradiction.
  Qed.


  Definition assoc_disjoint_keys(l1 l2 : list (A * B)) : Prop :=
    forall(a : A), 
      (is_some (assoc l1 a) -> is_none (assoc l2 a)) /\
      (is_some (assoc l2 a) -> is_none (assoc l1 a)).

  Lemma assoc_disjoint_keys_right_tail :
    forall(a : A)(b : B)(l1 l2 : list (A * B)),
      assoc_disjoint_keys l1 ((a, b) :: l2) ->
        assoc_disjoint_keys l1 l2.
  Proof.
    intros a b l1 l2 H a'.
    split.
      intros H0.
      apply H in H0.
      revert H0.
      apply assoc_cons_split.
        contradiction.
      trivial.
    intros H0.
    apply H.
    apply assoc_cons_split.
      simpl.
      trivial.
    trivial.
  Qed.

End Assoc.

Implicit Arguments assoc [A B].
Implicit Arguments rassoc [A B].
Implicit Arguments assoc_map_split [A B C].
Implicit Arguments injective_assoc [A B].
Implicit Arguments assoc_mapping [A B].
Implicit Arguments assoc_mapping_tail [A B].
Implicit Arguments assoc_mapping_cons_first [A B].
Implicit Arguments assoc_disjoint_keys [A B].
Implicit Arguments assoc_disjoint_keys_right_tail [A B].


Section Assoc_2.
  Variable A : Type.
  Variable aeq : eq_type A.

  Lemma assoc_swap_pairs : 
    forall(B : Type)(l : list (B * A))(a : A),
      assoc aeq (swap_pairs l) a = rassoc aeq l a.
  Proof.
    induction l.
      trivial.
    destruct a.
    intros a'.
    simpl.
    rewrite IHl.
    trivial.
  Qed.


  Lemma assoc_map_pair :
    forall(B C : Type)(f : A * B -> C)
          (l : list (A * B))(a : A),
      assoc aeq (map (fun(ab : A * B) => (fst ab, f ab)) l) a =
        match assoc aeq l a with
          | Some b => Some (f (a, b))
          | None => None
        end.
  Proof.
    induction l.
      intros a.
      simpl.
      trivial.
    destruct a as [a b].
    intros a0.
    simpl map.
    apply assoc_cons_split.
      intros H0.
      subst a0.
      rewrite assoc_cons_first.
      trivial.
    intros H0.
    apply assoc_cons_split.
      intros H1.
      contradiction.
    intros H1.
    apply IHl.
  Qed.


  (***************************************************************************)
  (** *** Apply assoc to a whole list *)
  (***************************************************************************)

  Fixpoint apply_assoc_map(m : list (A * A))(l : list A) : list A :=
    match l with
      | [] => []
      | a :: l => 
        (match assoc aeq m a with
           | Some a' => a'
           | None => a
         end
        ) :: (apply_assoc_map m l)
    end.

  Lemma In_apply_assoc_map_member :
    forall(m : list (A * A))(a1 a2 : A)(l : list A),
      In a1 l ->
      assoc aeq m a1 = Some a2 ->
        In a2 (apply_assoc_map m l).
  Proof.
    induction l.
      contradiction.
    intros H H0.
    destruct H.
      subst a.
      simpl.
      rewrite H0.
      auto.
    simpl.
    right.
    apply IHl; trivial.
  Qed.

  Lemma In_apply_assoc_map_non_member :
    forall(m : list (A * A))(a : A)(l : list A),
      In a l ->
      assoc aeq m a = None ->
        In a (apply_assoc_map m l).
  Proof.
    induction l.
      contradiction.
    intros H H0.
    destruct H.
      subst a0.
      simpl.
      rewrite H0.
      auto.
    simpl.
    right.
    apply IHl; trivial.
  Qed.

  Lemma In_apply_assoc_map_destruct :
    forall(m : list (A * A))(a : A)(l : list A),
      In a (apply_assoc_map m l) ->
        (exists(a' : A), assoc aeq m a' = Some a /\ In a' l) \/
        (assoc aeq m a = None /\ In a l).
  Proof.
    induction l.
      contradiction.
    intros H.
    simpl in *.
    destruct H.
      destruct (assoc aeq m a0) eqn:H0.
        left.
        subst a1.
        exists a0.
        auto.
      right.
      subst a0.
      auto.
    specialize (IHl H).
    decompose [ex and or] IHl; clear IHl.
      rename x into a'.
      left.
      exists a'.
      auto.
    right.
    auto.
  Qed.

  Lemma apply_assoc_map_support :
    forall(m : list (A * A))(l : list A),
      incl (apply_assoc_map m l) (apply_assoc_map m (list_support aeq l)).
  Proof.
    intros m l a H.
    apply In_apply_assoc_map_destruct in H.
    decompose [ex and or] H; clear H.
      rename x into a'.
      eapply In_apply_assoc_map_member.
        apply list_support_correct_content.
        eexact H2.
      trivial.
    apply In_apply_assoc_map_non_member.
      apply list_support_correct_content.
      trivial.
    trivial.
  Qed.

  Lemma apply_assoc_map_append :
    forall(m : list (A * A))(l1 l2 : list A),
      apply_assoc_map m (l1 ++ l2) =
        (apply_assoc_map m l1) ++ (apply_assoc_map m l2).
  Proof.
    induction l1.
      simpl.
      trivial.
    intros l2.
    simpl.
    rewrite IHl1.
    trivial.
  Qed.

  Lemma apply_assoc_map_flatten :
    forall(m : list (A * A))(ll : list (list A)),
      apply_assoc_map m (flatten ll) =
        flatten (map (apply_assoc_map m) ll).
  Proof.
    induction ll.
      trivial.
    simpl.
    rewrite apply_assoc_map_append.
    rewrite IHll.
    trivial.
  Qed.

End Assoc_2.

Implicit Arguments assoc_map_pair [A B C].
Implicit Arguments apply_assoc_map [A].

