(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: k_sound_complete.v,v 1.8 2013/04/10 11:17:15 tews Exp $
 *)


(** ** K example, Soundness, Completeness, 4.6, page 18 

      This module proves one-step soundness and one-step completeness
      for K and derives then soundness, completeness, cut-elimination
      and contraction for K by using the generic theorems.

      Note that all assumptions from the generic theorems are
      discharched here (besides [classical_logic], of course).
*)

Require Export k_semantics complete.

Section K_sound.

  Hypothesis pred_ext : forall(A : Type)(P Q : A -> Prop),
   (forall(a : A), P a <-> Q a) -> P = Q.


  (***************************************************************************)
  (** *** One-step soundness *)
  (***************************************************************************)

  Lemma k_rules_one_step_sound :
    classical_logic ->          (* maybe classical *)
      one_step_sound 0 (k_lambda pred_ext) k_rules one_step_rule_set_k_rules.
  Proof.
    unfold one_step_sound, k_rules in *.
    intros classical X r rules_r coval H.
    decompose [ex and] rules_r.
    rename x into n, x0 into sa.
    unfold KV in *.
    assert (0 < length (assumptions r)).
    rewrite H0.
      auto.
    specialize (H 0 H2).
    revert H.
    generalize (one_step_rule_propositional_assumption r
                          (one_step_rule_set_k_rules r rules_r) 0 H2).
    revert H2.
    rewrite H0.
    simpl.
    intros H2 p H.
    clear H2 H0.
    eapply mod_seq_val_valid_reorder.
      apply list_reorder_symm.
      eexact H3.
    eapply prop_seq_val_valid_reorder with (s2 := k_assumption n) in H.
      destruct n.
        apply mod_seq_valid_k_conclusion_0.
        intros P x H4.
        rewrite prop_seq_valid_k_assumption_0 in H.
        apply H.
      rewrite prop_seq_valid_k_assumption_Sn in H.
      apply mod_seq_valid_k_conclusion_Sn.
      intros P.
      intros H4; apply H4; clear H4.
      intros H0 x H4.
      specialize (H x).
      apply classical in H.
      apply H.
      intros i i_less.
      apply H0.
      trivial.
    eexact H1.
  Qed.


  (***************************************************************************)
  (** *** Derive soundness *)
  (***************************************************************************)

  Theorem sound_k_GRC : forall(s : sequent KV KL),
    classical_logic ->
    provable (GRC_set k_rules) (empty_sequent_set KV KL) s ->
      valid_all_models 0 (k_lambda pred_ext) s.
  Proof.
    intros s H H0.
    eapply sound_GRC.
        trivial.
      apply k_rules_one_step_sound; trivial.
    trivial.
  Qed.


  Theorem sound_k_GR : forall(s : sequent KV KL),
    classical_logic ->          (* maybe classical *)
    provable (GR_set k_rules) (empty_sequent_set KV KL) s ->
      valid_all_models 0 (k_lambda pred_ext) s.
  Proof.
    intros s H.
    eapply sound_GR.
    apply k_rules_one_step_sound.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** One-step completeness *)
  (***************************************************************************)

  Lemma k_rules_one_step_cut_free_complete :
    classical_logic ->
      one_step_cut_free_complete 0 (k_lambda pred_ext) k_rules 
                                 one_step_rule_set_k_rules.
  Proof.
    intros classical.
    apply simple_one_step_cut_free_complete.
      exact kv_eq.
    intros X coval gamma gamma_nonempty gamma_simple H.
    assert (H0 := gamma_simple).
    apply simple_modal_sequent_partition in H0.
    decompose [ex and] H0; clear H0.
    rename x into mods, x0 into negs.
    assert (H5 := simple_modal_sequent_list_reorder _ _ H4 gamma_simple).
    assert (H6 := list_reorder_nonempty _ _ gamma_nonempty H4).
    apply mod_seq_val_valid_reorder with (nonempty_s2 := H6)
          (propm_s2 := simple_modal_sequent_is_prop_modal_prop _ H5) 
          in H; trivial.
    apply destruct_neg_mod_sequent in H1.
      apply destruct_mod_sequent in H2.
      destruct (kv_eq (length (mods ++ negs)) 1) as [H7|H7].
        rewrite short_mod_seq_valid_char 
           with (mv := prop_var_sequent mods)(nv := prop_var_sequent negs)
           in H; trivial.
        decompose [ex and] H; clear H.
        rename x into v.
        unfold KV in *.
        rewrite H0 in *.
        rewrite H3 in *.
        simpl in *.
        subst mods negs.
        clear H3 H0 H5 H6 H7.
        rewrite app_nil_r in *.
        exists (k_rule 0), (rename_of_fun (function_update kv_eq id 0 v)).
        apply dep_conj with (a := k_rules_rule 0).
        apply dep_conj with (a := renaming_rename_of_fun _).
        split.
          exists [].
          eapply list_reorder_trans_rev.
            apply list_reorder_symm.
            eexact H4.
          apply list_reorder_refl.
        intros i i_less.
        destruct i.
          simpl nth.
          apply prop_seq_valid_renamed_k_assumption_0.
          intros x.
          rewrite function_update_eq.
          specialize (H9 (full_set _)).
          apply full_subset_full in H9.
          apply H9.
        exfalso.
        simpl in *.
        omega.
      rewrite long_mod_seq_valid_char 
           with (mv := prop_var_sequent mods)(nv := prop_var_sequent negs)
           in H; trivial.
      specialize (H (fun x => every_nth (coval x) (prop_var_sequent negs))).
      apply classical in H.
      lapply H; clear H.
        intros H.
        destruct H as [i].
        destruct H as [i_less].
        unfold KV in *.
        set (v := nth (prop_var_sequent mods) i i_less) in *.
        exists (k_rule (length (prop_var_sequent negs))), 
               (rename_of_fun (k_rename_fun (v :: prop_var_sequent negs))).
        apply dep_conj with (a := k_rules_rule _).
        apply dep_conj with (a := renaming_rename_of_fun _).
        split.
          exists (cutout_nth mods i).
          simpl conclusion.
          rewrite subst_k_rename_k_conclusion.
          unfold KV in *.
          rewrite <- H1.
          simpl.
          eapply list_reorder_trans.
            apply list_reorder_move_append.
          eapply list_reorder_trans.
            apply list_reorder_append_swap.
          eapply list_reorder_trans_rev.
            apply list_reorder_symm.
            eexact H4.
          apply list_reorder_append_both.
            unfold cutout_nth in *.
            eapply list_reorder_trans.
              apply list_reorder_move_append.
            assert (i < length mods).
              rewrite H2.
              rewrite map_length.
              trivial.
            rewrite list_split_n_equal with (n_less := H0).
              apply list_reorder_refl.
            clear - H2. 
            generalize H0.
            rewrite H2.
            intros H1.
            rewrite nth_map.
            erewrite nth_tcc_irr.
            reflexivity.
          apply list_reorder_refl.
        intros ai ai_less.
        destruct ai.
          simpl nth.
          destruct (length (prop_var_sequent negs)) eqn:H8.
            apply prop_seq_valid_renamed_k_assumption_0.
            intros x.
            rewrite k_rename_fun_less 
                  with (n_less := lt_0_Sn (length (prop_var_sequent negs))).
            simpl in *.
            apply H.
            clear - H8.
            destruct (prop_var_sequent negs).
              apply every_nth_empty.
            discriminate.
          apply prop_seq_valid_renamed_k_assumption_Sn.
          intros x.
          intros H9; apply H9; clear H9.
          intros H0.
          rewrite k_rename_fun_less with
                      (n_less := lt_0_Sn (length (prop_var_sequent negs))).
          simpl.
          apply H.
          clear - H8 H0.
          intros j j_less.
          assert (j < length (map (k_rename_fun (v :: prop_var_sequent negs)) 
                             (seq 1 (S n)))).
            rewrite map_length.
            rewrite seq_length.
            rewrite H8 in *.
            trivial.
          specialize (H0 j H).
          rewrite nth_map in H0.
          rewrite nth_seq in H0.
          rewrite k_rename_fun_less with (n_less := lt_n_S _ _ j_less) in H0.
          rewrite nth_tail in H0.
          erewrite nth_tcc_irr.
          eexact H0.
        exfalso.
        simpl in *.
        omega.
      clear. 
      intros i i_less x H.
      apply H.
    clear - H5.
    apply simple_modal_sequent_append_right in H5.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Derive completeness, cut-elimination and contraction  *)
  (***************************************************************************)

  Theorem k_complete : forall(s : sequent KV KL),
    classical_logic ->
    valid_all_models 0 (k_lambda pred_ext) s ->
      provable (GR_set k_rules) (empty_sequent_set KV KL) s.
  Proof.
    intros s H H0.
    apply cut_free_completeness with (enum_V := kv_enum)
                                     (LS := (k_lambda pred_ext))
                                     (osr := one_step_rule_set_k_rules).
            apply kv_eq.
          trivial.
        apply non_trivial_k_functor.
      apply k_rules_one_step_cut_free_complete.      
      trivial.
    trivial.
  Qed.

  Theorem k_semantic_cut :
    classical_logic ->
      admissible_rule_set (GR_set k_rules) (empty_sequent_set KV KL) 
                          is_cut_rule.
  Proof.
    intros H.
    apply semantic_admissible_cut with (enum_V := kv_enum)
                                       (LS := (k_lambda pred_ext))
                                       (osr_prop := one_step_rule_set_k_rules).
            apply kv_eq.
          trivial.
        apply non_trivial_k_functor.
      apply k_rules_one_step_sound.
      trivial.
    apply k_rules_one_step_cut_free_complete.      
    trivial.
  Qed.

  Theorem k_semantic_contraction :
    classical_logic ->
      admissible_rule_set (GR_set k_rules) (empty_sequent_set KV KL) 
                          is_contraction_rule.
  Proof.
    intros H.
    apply semantic_admissible_contraction
                          with (enum_V := kv_enum)(LS := (k_lambda pred_ext))
                                (osr_prop := one_step_rule_set_k_rules).
            apply kv_eq.
          trivial.
        apply non_trivial_k_functor.
      apply k_rules_one_step_sound.
      trivial.
    apply k_rules_one_step_cut_free_complete.      
    trivial.
  Qed.

End K_sound.
