(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: lists.v,v 1.55 2013/04/10 11:17:15 tews Exp $
 *)

(** ** Lists

      In the end, every theorem boils down to a property of lists.
      This is not always true, but often enough. I need a lot of
      definitions, that are not in the Coq library, most notably, a
      partial [nth] and [every_nth]. I also need a lot of additional
      results on the definitions in the standard library, for instance
      about [firstn], [skipn] and [In].

      Note that a lot of list material is in different files for
      modularization and better compilation speed. See the files
      [reorder], [list_set], [list_support], [assoc], [list_multiset],
      [some_nth] and [some_neg]. 
*) 

Require Export misc sets.

Section Lists0.
  Variable A : Type.


  (**************************************************************************)
  (** *** misc stuff *)
  (**************************************************************************)

  (* for a :: l1 ++ l2 = (a :: l1) ++ l2 use app_comm_cons *)
  Lemma append_single_rev : forall(a : A)(l : list A), a :: l = [a] ++ l.
  Proof.
    trivial.
  Qed.


  (**************************************************************************)
  (** *** nonempty lists *)
  (**************************************************************************)

  Definition nonempty_list(l : list A) : Prop :=
    match l with
      | [] => False
      | _ => True
    end.

  Lemma nonempty_list_tcc : 
    forall(X : Type), nonempty_list [] -> X.
  Proof.
    intros X H.
    contradiction.
  Qed.

  (**************************************************************************)
  (** ***  firstn lemmas  *)
  (**************************************************************************)

  
  Lemma length_firstn_less : forall(l : list A)(n : nat),
    n <= length l -> length (firstn n l) = n.
  Proof.
    intros l n H.
    rewrite firstn_length.
    rewrite min_l.
      trivial.
    trivial.
  Qed.
  
  Lemma firstn_whole : forall(l : list A)(n : nat),
    n >= length l -> firstn n l = l.
  Proof.
    induction l.
      intros n H.
      destruct n.
        trivial.
      trivial.
    intros n H.
    destruct n.
      exfalso.
      simpl in *.
      omega.
    simpl in *.
    rewrite IHl.
      trivial.
    omega.
  Qed.

  Lemma firstn_append_left : forall(l1 l2 : list A)(n : nat),
    n <= length l1 -> firstn n (l1 ++ l2) = firstn n l1.
  Proof.
    induction l1.
      intros l2 n H.
      simpl in *.
      assert (n = 0).
        auto with arith.
      subst n.
      trivial.
    simpl.
    intros l2 n H.
    destruct n.
      simpl.
      trivial.
    simpl.
    rewrite IHl1.
      trivial.
    omega.
  Qed.
  

  Lemma firstn_append_right : forall(l1 l2 : list A)(n : nat),
    n >= length l1 -> firstn n (l1 ++ l2) = l1 ++ (firstn (n - length l1) l2).
  Proof.
    induction l1.
      intros l2 n H.
      simpl in *.
      rewrite <- minus_n_O.
      trivial.
    intros l2 n H.
    simpl in *.
    destruct n.
      exfalso.
      omega.
    simpl in *.
    rewrite IHl1.
      trivial.
    omega.
  Qed.

  Lemma firstn_firstn_less :
    forall(l : list A)(n1 n2 : nat),
      n1 <= n2 ->
      n2 <= length l ->
        firstn n1 (firstn n2 l) = firstn n1 l.
  Proof.
    intros l n1 n2 H H0.
    rewrite <- (firstn_append_left (firstn n2 l) (skipn n2 l)).
      rewrite firstn_skipn.
      trivial.
    rewrite length_firstn_less.
      trivial.
    trivial.
  Qed.

  Lemma map_firstn : forall(B : Type)(f : A -> B)(l : list A)(n : nat),
    map f (firstn n l) = firstn n (map f l).
  Proof.
    induction l.
      intros n.
      destruct n.
        trivial.
      trivial.
    intros n.
    destruct n.
      trivial.
    simpl.
    f_equal.
    apply IHl.
  Qed.


  (**************************************************************************)
  (** ***  skipn lemmas  *)
  (**************************************************************************)

  
  Lemma length_skipn : forall(l : list A)(n : nat),
    length (skipn n l) = length l - n.
  Proof.
    induction l.
      intros n.
      destruct n.
        trivial.
      trivial.
    intros n.
    destruct n.
      trivial.
    simpl.
    apply IHl.
  Qed.

  Lemma skipn_whole : forall(l : list A)(n : nat),
    n >= length l -> skipn n l = [].
  Proof.
    induction l.
      intros n H.
      destruct n.
        trivial.
      trivial.
    intros n H.
    destruct n.
      exfalso.
      simpl in H.
      omega.
    simpl.
    rewrite IHl.
      trivial.
    simpl in H.
    omega.
  Qed.


  Lemma skipn_append_left : forall(l1 l2 : list A)(n : nat),
    n <= length l1 -> skipn n (l1 ++ l2) = (skipn n l1) ++ l2.
  Proof.
    induction l1.
      intros l2 n H.
      simpl in *.
      assert (n = 0).
        omega.
      subst n.
      simpl in *.
      trivial.
    intros l2 n H.
    destruct n.
      simpl in *.
      trivial.
    simpl in *.
    apply IHl1.
    apply le_S_n.
    trivial.
  Qed.

  Lemma skipn_append_right : forall(l1 l2 : list A)(n : nat),
    n >= length l1 -> skipn n (l1 ++ l2) = skipn (n - length l1) l2.
  Proof.
    induction l1.
      intros l2 n H.
      simpl.
      rewrite <- minus_n_O.
      trivial.
    intros l2 n H.
    destruct n.
      exfalso.
      simpl in *.
      omega.
    simpl in *.
    apply IHl1.
    omega.
  Qed.
  
  Lemma skipn_skipn : forall(l : list A)(n1 n2 : nat),
    skipn n1 (skipn n2 l) = skipn (n1 + n2) l.
  Proof.
    intros l n1 n2.
    assert (n2 <= length l \/ n2 > length l).
      omega.
    destruct H.
      assert (H0 := length_firstn_less _ _ H).
      lapply (skipn_append_right (firstn n2 l) (skipn n2 l) (n1 + n2)).
        intros H1.
        rewrite firstn_skipn in H1.
        rewrite H1.
        f_equal.
        rewrite length_firstn_less; try omega.
      omega.
    rewrite (skipn_whole l n2).
      rewrite skipn_whole.
        rewrite skipn_whole.
          trivial.
        omega.
      simpl.
      omega.
    omega.
  Qed.


  Lemma skipn_firstn : forall(l : list A)(n1 n2 : nat),
    n1 <= n2 ->
    n2 <= length l ->
      skipn n1 (firstn n2 l) = firstn (n2 - n1) (skipn n1 l).
  Proof.
    induction l.
      intros n1 n2 H H0.
      simpl in H0.
      destruct n1.
        destruct n2.
          trivial.
        exfalso.
        omega.
      exfalso.
      omega.
    intros n1 n2 H H0.
    simpl in H0.
    destruct n1.
      destruct n2.
        trivial.
      trivial.
    destruct n2.
      exfalso.
      omega.
    simpl.
    apply IHl.
      omega.
    omega.
  Qed.


  Lemma map_skipn : forall(B : Type)(f : A -> B)(l : list A)(n : nat),
    map f (skipn n l) = skipn n (map f l).
  Proof.
    induction l.
      intros n.
      destruct n.
        trivial.
      trivial.
    intros n.
    destruct n.
      trivial.
    simpl.
    apply IHl.
  Qed.

End Lists0.


Section Lists1.

  Variable A : Type.

  (**************************************************************************)
  (** ***  cutout_nth  *)
  (**************************************************************************)

   Definition cutout_nth{A : Type}(l : list A)(n : nat) : list A :=
     (firstn n l) ++ (skipn (1 + n) l).

   Lemma cutout_nth_cons_0 : 
     forall(a : A)(l : list A), cutout_nth (a :: l) 0 = l.
   Proof.
     intros a l.
     trivial.
   Qed.

   Lemma cutout_nth_cons_succ : forall(a : A)(l : list A)(n : nat), 
     cutout_nth (a :: l) (S n) = a :: cutout_nth l n.
   Proof.
     intros a l n.
     trivial.
   Qed.

   Lemma map_cutout_nth : 
     forall{A B : Type}(f : A -> B)(l : list A)(n : nat),
       map f (cutout_nth l n) = cutout_nth (map f l) n.
   Proof.
     intros A0 B f l n.
     unfold cutout_nth in *.
     rewrite map_app.
     rewrite map_firstn.
     rewrite map_skipn.
     trivial.
   Qed.


  (**************************************************************************)
  (** ***  forallb  *)
  (**************************************************************************)

  Lemma forallb_cons : forall(f : A -> bool)(a : A)(l : list A),
    forallb f (a :: l) = true <-> ((f a = true) /\ forallb f l = true).
  Proof.
    intros f a l.
    simpl.
    rewrite andb_true_iff.
    apply iff_refl.
  Qed.


  (**************************************************************************)
  (** ***  combine  *)
  (**************************************************************************)

   Lemma in_combine_same : forall (l : list A)(a b : A),
     In (a, b) (combine l l) -> a = b /\ In a l.
   Proof.
     induction l.
       intros a b H.
       contradiction.
     simpl.
     intros a0 b H.
     destruct H.
       inversion H.
       auto.
     apply IHl in H.
     destruct H.
     auto.
   Qed.


  (**************************************************************************)
  (** ***  partition  *)
  (**************************************************************************)

  Lemma partition_result : forall(f : A -> bool)(l : list A),
    forallb f (fst (partition f l)) = true /\ 
    forallb (fun a => negb (f a)) (snd (partition f l)) = true.
  Proof.
    induction l.
      simpl.
      auto.
    simpl.
    destruct IHl.
    destruct (partition f l).
    destruct (f a) eqn:?.
      simpl in *.
      rewrite H0.
      rewrite H.
      rewrite Heqb.
      rewrite andb_true_iff.
      auto.
    simpl in *.
    rewrite Heqb.
    rewrite H0.
    rewrite H.
    rewrite andb_true_iff.
    rewrite negb_true_iff.
    auto.
  Qed.


  (**************************************************************************)
  (** ***  NoDup  *)
  (**************************************************************************)

  Lemma NoDup_tail : forall(a : A)(l : list A),
    NoDup (a :: l) -> NoDup l.
  Proof.
    intros a l H.
    remember (a :: l).
    destruct H.
      discriminate.
    inversion Heql0; clear Heql0.
    subst l0.
    trivial.
  Qed.

  Lemma NoDup_head : forall(a : A)(l : list A),
    NoDup (a :: l) -> ~ In a l.
  Proof.
    intros a l H.
    remember (a :: l).
    destruct H.
      discriminate.
    inversion Heql0; clear Heql0.
    subst x l0.
    trivial.
  Qed.

  Lemma NoDup_map : forall(B : Type)(f : A -> B)(l : list A),
    (forall(a1 a2 : A), In a1 l -> In a2 l -> f a1 = f a2 -> a1 = a2) ->
    NoDup l ->
      NoDup (map f l).
  Proof.
    induction l.
      intros H H0.
      simpl.
      apply NoDup_nil.
    intros H H0.
    simpl.
    apply NoDup_cons.
      intros H1.
      apply in_map_iff in H1.
      destruct H1 as [a'].
      destruct H1.
      apply H in H1.
          subst a'.
          apply NoDup_head in H0.
          contradiction.
        right.
        trivial.
      left.
      trivial.
    apply IHl.
      intros a1 a2 H1 H2.
      apply H.
        right.
        trivial.
      right.
      trivial.
    apply NoDup_tail in H0.
    trivial.
  Qed.

  Lemma NoDup_map_injective : forall(B : Type)(f : A -> B)(l : list A),
    injective f ->
    NoDup l ->
      NoDup (map f l).
  Proof.
    intros B f l H.
    apply NoDup_map.
    intros a1 a2 H0 H1 H2.
    apply H in H2.
    trivial.
  Qed.

  (* NoDup_seq is below *)

  (**************************************************************************)
  (** ***  nth as partial function  *)
  (**************************************************************************)

  Lemma nth_0_nil_tcc : 
    forall(n : nat)(inside : n < length (@nil A))(X : Type), X.
  Proof.
    intros n inside X.
    simpl in inside.
    exfalso.
    omega.
  Qed.

  Lemma nth_succ_tcc : forall(n : nat)(a : A)(l : list A),
    S n < length (a :: l) -> n < length l.
  Proof.
    intros n a l H.
    apply lt_S_n.
    trivial.
  Qed.

  Fixpoint nth(l : list A)(n : nat)(inside : n < length l) : A :=
    match l return n < length l -> A with
      | [] => fun(H : n < length []) => nth_0_nil_tcc n H _
      | a :: r => fun(H : n < length (a :: r)) => 
        match n return n < length (a :: r) -> A with
          | 0 => fun _ => a
          | S p => fun(H0 : S p < length (a :: r)) => 
              nth r p (nth_succ_tcc p a r H0)
        end H
    end inside.

  Lemma nth_tail : 
    forall(a : A)(l : list A)(n : nat)(n_less : S n < length (a :: l)),
      nth (a :: l) (S n) n_less = nth l n (nth_succ_tcc _ _ _ n_less).
  Proof.
    intros a l n n_less.
    trivial.
  Qed.

  Lemma nth_tcc_irr : 
    forall(l : list A)(n : nat)(inside_1 inside_2 : n < length l),
      nth l n inside_1 = nth l n inside_2.
  Proof.
    induction l.
      destruct n.
        intros inside_1.
        exfalso.
        simpl in *.
        omega.
      intros inside_1.
      exfalso.
      simpl in *.
      omega.
    destruct n.
      simpl.
      trivial.
    simpl.
    intros inside_1 inside_2.
    apply IHl.
  Qed.

  Lemma list_equal_nth_char : forall(l1 l2 : list A),
    length l1 = length l2 ->
    (forall(n : nat)(n1_less : n < length l1)(n2_less : n < length l2),
               nth l1 n n1_less = nth l2 n n2_less) ->
      l1 = l2.
  Proof.
    induction l1.
      intros l2 H H0.
      destruct l2.
        trivial.
      discriminate.
    rename a into a1.
    intros l2 H H0.
    destruct l2 as [| a2].
      discriminate.
    f_equal.
      apply (H0 _ (lt_0_Sn (length l1)) (lt_0_Sn (length l2))).
    apply IHl1.
      simpl in H.
      inversion H; clear H.
      trivial.
    intros n n1_less n2_less.
    specialize (H0 _ (lt_n_S _ _ n1_less) (lt_n_S _ _ n2_less)).
    rewrite nth_tail in H0.
    rewrite nth_tail in H0.
    erewrite nth_tcc_irr.
    erewrite (nth_tcc_irr l2).
    eexact H0.
  Qed.

  Lemma nth_head_tcc : forall(l : list A), length l = 1 -> 0 < length l.
  Proof.
    intros l H.
    rewrite H.
    apply lt_0_Sn.
  Qed.

  Lemma nth_append_left :
    forall(l1 l2 : list A)(n : nat)
          (n_less_app : n < length (l1 ++ l2))
          (n_less_l1 : n < length l1),
      nth (l1 ++ l2) n n_less_app = nth l1 n n_less_l1.
  Proof.
    induction l1.
      intros l2 n n_less_app n_less_l1.
      exfalso.
      simpl in *.
      omega.
    intros l2 n n_less_app n_less_l1.
    destruct n.
      simpl.
      trivial.
    simpl in *.
    apply IHl1.
  Qed.

  Lemma nth_append_right_tcc : forall(l1 l2 : list A)(n : nat),
    n < length(l1 ++ l2) -> n >= length l1 -> n - length l1 < length l2.
  Proof.
    intros l1 l2 n H H0.
    rewrite app_length in *.
    omega.
  Qed.

  Lemma nth_append_right_ind :
    forall(l1 l2 : list A)(n : nat)
          (n_less : n < length(l1 ++ l2))(n_greater : n >= length l1)
          (n_min_less : n - length l1 < length l2),
      nth (l1 ++ l2) n n_less = nth l2 (n - length l1) n_min_less.
  Proof.
    induction l1.
      intros l2 n n_less n_greater n_min_less.
      simpl in *.
      generalize n_min_less.
      rewrite <- minus_n_O.
      apply nth_tcc_irr.
    intros l2 n n_less n_greater n_min_less.
    destruct n.
      exfalso.
      clear IHl1.
      simpl in *.
      omega.
    simpl in *.
    apply IHl1.
    omega.
  Qed.

  Lemma nth_append_right :
    forall(l1 l2 : list A)(n : nat)
          (n_less : n < length(l1 ++ l2))(n_greater : n >= length l1),
      nth (l1 ++ l2) n n_less = 
        nth l2 (n - length l1) (nth_append_right_tcc l1 l2 n n_less n_greater).
  Proof.
    intros l1 l2 n n_less n_greater.
    apply nth_append_right_ind.
    trivial.
  Qed.

  Lemma nth_firstn_tcc : forall(l : list A)(n1 n2 : nat),
    n2 < length (firstn n1 l) -> n2 < n1 -> n2 < length l.
  Proof.
    intros l n1 n2 H H0.
    rewrite firstn_length in H.
    assert (H1:= Min.min_spec n1 (length l)).
    decompose [and or] H1; clear H1.
      omega.
    omega.
  Qed.
  
  Lemma nth_firstn : 
    forall(l : list A)(n1 n2 : nat)
          (n2_less_firstn : n2 < length (firstn n1 l))(n2_less_n1 : n2 < n1),
      nth (firstn n1 l) n2 n2_less_firstn = 
        nth l n2 (nth_firstn_tcc l n1 n2 n2_less_firstn n2_less_n1).
  Proof.
    intros l n1 n2 n2_less_firstn n2_less_n1.
    generalize (nth_firstn_tcc l n1 n2 n2_less_firstn n2_less_n1).
    assert(H := firstn_skipn n1 l).
    assert (forall l0 : n2 < length ((firstn n1 l) ++ (skipn n1 l)),
              nth (firstn n1 l) n2 n2_less_firstn = 
                nth ((firstn n1 l) ++ (skipn n1 l)) n2 l0).
      intros l0.
      assert (n2 < length (firstn n1 l)).
        rewrite app_length in *.
        omega.
      rewrite nth_append_left with (n_less_l1 := H0).
      apply nth_tcc_irr.
    rewrite H in H0.
    trivial.
  Qed.

  Lemma nth_skipn_tcc :
    forall(l : list A)(n1 n2 : nat),
      n1 <= length l -> n2 < length (skipn n1 l) -> n1 + n2 < length l.
  Proof.
    intros l n1 n2 H H0.
    rewrite length_skipn in H0.
    omega.
  Qed.


  Lemma nth_skipn :
    forall(l : list A)(n1 n2 : nat)
          (n1_less : n1 <= length l)(n2_less : n2 < length (skipn n1 l)),
      nth (skipn n1 l) n2 n2_less = 
        nth l (n1 + n2) (nth_skipn_tcc l n1 n2 n1_less n2_less).
  Proof.
    induction l.
      intros n1 n2 n1_less n2_less.
      exfalso.
      destruct n1.
        simpl in *.
        omega.
      simpl in *.
      omega.
    intros n1 n2 n1_less n2_less.
    destruct n1.
      simpl skipn.
      apply nth_tcc_irr.
    simpl in *.
    rewrite (IHl _ _ (le_S_n _ _ n1_less)).
    apply nth_tcc_irr.
  Qed.


  Lemma cons_nth_skipn :
    forall(l : list A)(n : nat)(inside : n < length l),
      (nth l n inside) :: (skipn (1 + n) l) = (skipn n l).
  Proof.
    induction l.
      intros n inside.
      exfalso.
      simpl in *.
      omega.
    intros n inside.
    destruct n.
      clear IHl.
      simpl in *.
      trivial.
    simpl (skipn (S n) (a :: l)).
    assert(skipn (1 + S n) (a :: l) = skipn (1 + n) l).
      auto.
    rewrite H.
    clear H.
    simpl (nth (a :: l) (S n) inside).
    apply IHl.
  Qed.


  Lemma list_split_at_n : 
    forall(l : list A)(n : nat)(n_less : n < length l),
      l = (firstn n l) ++ (nth l n n_less) :: (skipn (1 + n) l).
  Proof.
    intros l n n_less.
    rewrite cons_nth_skipn.
    rewrite firstn_skipn.
    trivial.
  Qed.

  Lemma list_split_n_equal :
    forall(l : list A)(a : A)(n : nat)(n_less : n < length l),
      nth l n n_less = a ->
        (firstn n l) ++ a :: (skipn (1 + n) l) = l.
  Proof.
    intros l a n n_less H.
    rewrite <- H.
    rewrite <- list_split_at_n.
    trivial.
  Qed.

  Lemma list_split_at_n2 :
    forall(l : list A)(n1 n2 : nat)
          (n1_less : n1 < length l)(n2_less : n2 < length l),
      n1 <> n2 -> 
        exists(ll lm lr : list A)(al ar : A),
          l = ll ++ al :: lm ++ ar :: lr /\
          ((al = nth l n1 n1_less /\ ar = nth l n2 n2_less) \/
           (al = nth l n2 n2_less /\ ar = nth l n1 n1_less)).
  Proof.
    intros l n1 n2 n1_less n2_less H.
    assert (n1 < n2 \/ n1 > n2).
      omega.
    destruct H0.
      assert (H1 := list_split_at_n l n2 n2_less).
      assert (n1 < length (firstn n2 l)).
        rewrite length_firstn_less.
          trivial.
        apply lt_le_weak.
        trivial.
      assert (H3 := list_split_at_n _ _ H2).
      exists (firstn n1 (firstn n2 l)), (skipn (1 + n1) (firstn n2 l)),
             (skipn (1 + n2) l), (nth (firstn n2 l) n1 H2),
             (nth l n2 n2_less).
      split.
        rewrite app_comm_cons.
        rewrite app_assoc.
        rewrite <- H3.
        trivial.
      left.
      split.
        rewrite nth_firstn with (n2_less_n1 := H0).
        apply nth_tcc_irr.
      trivial.
    assert (H1 := list_split_at_n l n1 n1_less).
    assert (n2 < length (firstn n1 l)).
      rewrite length_firstn_less.
        trivial.
      apply lt_le_weak.
      trivial.
    assert (H3 := list_split_at_n _ _ H2).
    exists (firstn n2 (firstn n1 l)), (skipn (1 + n2) (firstn n1 l)),
           (skipn (1 + n1) l), (nth (firstn n1 l) n2 H2),
           (nth l n1 n1_less).
    split.
      rewrite app_comm_cons.
      rewrite app_assoc.
      rewrite <- H3.
      trivial.
    right.
    split.
      rewrite nth_firstn with (n2_less_n1 := H0).
      apply nth_tcc_irr.
    trivial.
  Qed.


End Lists1.

Implicit Arguments nonempty_list [A].
Implicit Arguments nonempty_list_tcc [A X].
Implicit Arguments length_firstn_less [A].
Implicit Arguments firstn_whole [A].
Implicit Arguments skipn_whole [A].
Implicit Arguments cutout_nth [A].
Implicit Arguments partition_result [A].
Implicit Arguments NoDup_tail [A].
Implicit Arguments nth_0_nil_tcc [A].
Implicit Arguments nth_succ_tcc [A].
Implicit Arguments nth [A].
Implicit Arguments nth_tcc_irr [A].
Implicit Arguments nth_head_tcc [A].
Implicit Arguments nth_append_left [A].
Implicit Arguments nth_append_right_tcc [A].
Implicit Arguments nth_append_right [A].
Implicit Arguments nth_skipn [A].
Implicit Arguments list_split_n_equal [A].
Implicit Arguments list_split_at_n2 [A].


Section Lists2.
  Variable A : Type.

  Lemma nth_map_tcc : 
    forall(B : Type)(f : A -> B)(l : list A)(n : nat)
          (n_less_map : n < length (map f l)),
      n < length l.
  Proof.
    intros B f l n n_less_map.
    rewrite map_length in n_less_map.
    trivial.
  Qed.

  Lemma nth_map :
    forall(B : Type)(f : A -> B)(l : list A)(n : nat)
          (n_less_map : n < length (map f l)),
      nth (map f l) n n_less_map = f (nth l n (nth_map_tcc B f l n n_less_map)).
  Proof.
    induction l.
      simpl.
      intros n n_less_map.
      exfalso.
      omega.
    simpl (length (map f (a :: l))).
    intros n n_less_map.
    destruct n.
      simpl.
      trivial.
    simpl.
    rewrite IHl.
    erewrite nth_tcc_irr.
    trivial.
  Qed.

  Lemma nth_map_inv_tcc : 
    forall(B : Type)(f : A -> B)(l : list A)(n : nat)(n_less : n < length l),
      n < length (map f l).
  Proof.
    intros B f l n n_less.
    rewrite map_length.
    trivial.
  Qed.

  Lemma nth_map_inv :
    forall(B : Type)(f : A -> B)(l : list A)(n : nat)(n_less : n < length l),
      f (nth l n n_less) = nth (map f l) n (nth_map_inv_tcc _ f _ _ n_less).
  Proof.
    intros B f l n n_less.
    rewrite nth_map.
    f_equal.
    apply nth_tcc_irr.
  Qed.


  (**************************************************************************)
  (** ***  flatten  *)
  (**************************************************************************)

  Fixpoint flatten(ll : list (list A)) : list A :=
    match ll with
      | [] => []
      | l :: ll => l ++ (flatten ll)
    end.

  Lemma flatten_append : forall(ll1 ll2 : list (list A)),
    flatten (ll1 ++ ll2) = (flatten ll1) ++ (flatten ll2).
  Proof.
    induction ll1.
      intros ll2.
      trivial.
    intros ll2.
    simpl.
    rewrite <- app_assoc.
    rewrite IHll1.
    trivial.
  Qed.

  Lemma flatten_map_singleton_id : forall(l : list A),
     flatten (map (fun(a : A) => [a]) l) = l.
  Proof.
    induction l.
      trivial.
    simpl.
    f_equal.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  member  *)
  (**************************************************************************)

  Fixpoint member(a_eq : eq_type A)(a : A)(l : list A) : bool :=
    match l with
      | [] => false
      | b :: l => if a_eq a b then true else member a_eq a l
    end.

  Lemma member_In : forall(a_eq : eq_type A)(a : A)(l : list A),
    member a_eq a l = true <-> In a l.
  Proof.
    induction l.
      simpl.
      split.
        discriminate.
      contradiction.
    simpl.
    destruct (a_eq a a0).
      subst a0.
      split.
        auto.
      auto.
    rewrite IHl.
    split.
      tauto.
    intros H.
    destruct H.
      apply eq_sym in H.
      contradiction.
    trivial.
  Qed.

  Lemma member_In_false : forall(a_eq : eq_type A)(a : A)(l : list A),
    member a_eq a l = false <-> ~ In a l.
  Proof.
    intros a_eq a l.
    split.
      intros H0 H1.
      assert (member a_eq a l = true).
        apply member_In; trivial.
      rewrite H0 in *.
      discriminate.
    intros H0.
    destruct (member a_eq a l) eqn:H1.
      apply member_In in H1; trivial.
      contradiction.
    trivial.
  Qed.

  Lemma member_append : forall(a_eq : eq_type A)(a : A)(l1 l2 : list A),
    member a_eq a (l1 ++ l2) = true ->
      member a_eq a l1 = true \/ member a_eq a l2 = true.
  Proof.
    induction l1.
      intros l2 H.
      auto.
    intros l2 H.
    simpl in *.
    destruct (a_eq a a0).
      left.
      trivial.
    apply IHl1.
    trivial.
  Qed.

  Lemma member_append_left : 
    forall(a_eq : eq_type A)(a : A)(l1 l2 : list A),
      member a_eq a l1 = true ->
        member a_eq a (l1 ++ l2) = true.
  Proof.
    induction l1.
      intros l2 H.
      discriminate.
    intros l2 H.
    simpl in *.
    destruct (a_eq a a0).
      trivial.
    apply IHl1.
    trivial.
  Qed.

  Lemma member_append_right :
    forall(a_eq : eq_type A)(a : A)(l1 l2 : list A),
      member a_eq a l2 = true ->
        member a_eq a (l1 ++ l2) = true.
  Proof.
    induction l1.
      intros l2 H.
      auto.
    intros l2 H.
    simpl.
    destruct (a_eq a a0).
      trivial.
    apply IHl1.
    trivial.
  Qed.

  (* 
   * Lemma nth_member : 
   *   forall(a_eq : A -> A -> bool)(a : A)(l : list A)
   *         (n : nat)(n_less : n < length l),
   *     a_eq a (nth l n n_less) = true ->
   *       member a_eq a l = true.
   * Proof.
   *   induction l.
   *     intros n n_less H.
   *     simpl in *.
   *     omega.
   *   intros n n_less H.
   *   destruct n.
   *     simpl in *.
   *     rewrite H.
   *     trivial.
   *   simpl in *.
   *   erewrite IHl.
   *     apply orb_true_r.
   *   eexact H.
   * Qed.
   *)

  (* 
   * Lemma member_nth :
   *   forall(a_eq : A -> A -> bool)(l : list A)(a : A),
   *     member a_eq a l = true ->
   *       exists(n : nat), n_less # n < length l /#\
   *         a_eq a (nth l n n_less) = true.
   * Proof.
   *   induction l.
   *     intros a H.
   *     discriminate.
   *   intros a0 H.
   *   simpl in H.
   *   destruct (a_eq a0 a) eqn:?.
   *     exists 0.
   *     constructor 1 with (a := lt_0_Sn (length l)).
   *     trivial.
   *   simpl in H.
   *   specialize (IHl _ H).
   *   decompose [ex and or dep_and] IHl; clear IHl.
   *   exists (S x).
   *   constructor 1 with (a := lt_n_S _ _ a1).
   *   simpl.
   *   erewrite nth_tcc_irr.
   *   eexact b.
   *)

  Lemma In_flatten : 
    forall(ll : list (list A))(a : A),
      In a (flatten ll) ->
        exists(n : nat), n_less # n < length ll /#\
          In a (nth ll n n_less).
  Proof.
    induction ll.
      intros a H.
      contradiction.
    rename a into l1.
    intros a H.
    simpl in H.
    apply in_app_or in H.
    destruct H.
      exists 0.
      apply dep_conj with (a := lt_0_Sn (length ll)).
      trivial.
    apply IHll in H; clear IHll.
    decompose [ex and or dep_and] H; clear H.
    rename x into n, a0 into n_less, b into H.
    exists (S n).
    apply dep_conj with (a := lt_n_S _ _ n_less).
    simpl.
    erewrite nth_tcc_irr.
    eexact H.
  Qed.


  (**************************************************************************)
  (** ***  In  *)
  (**************************************************************************)

  Lemma in_map_reverse :
    forall{B : Type}(f : A -> B)(l : list A)(a2 : A),
      (forall(a1 : A), In a1 l -> f a1 = f a2 -> a1 = a2) ->
      In (f a2) (map f l) ->
        In a2 l.
  Proof.
    induction l.
      intros a H H0.
      contradiction.
    intros a2 H H0.
    simpl in H0.
    destruct H0.
      apply H in H0.
        subst a.
        left.
        trivial.
      left.
      trivial.
    right.
    apply IHl.
      clear - H.
      intros a1 H0 H1.
      apply H.
        right.
        trivial.
      trivial.
    trivial.
  Qed.

  Lemma In_nth : 
    forall(a : A)(l : list A),
      In a l -> 
        exists(n : nat), n_less # n < length l /#\ nth l n n_less = a.
  Proof.
    intros a l H.
    apply in_split in H.
    destruct H as [ll].
    destruct H as [lr].
    subst l.
    exists (length ll).
    assert (length ll < length (ll ++ a :: lr)).
      rewrite app_length.
      simpl.
      omega.
    apply dep_conj with (a := H).
    rewrite nth_append_right with (n_greater := ge_refl _).
    generalize (nth_append_right_tcc ll (a :: lr) (length ll) H 
                                        (ge_refl (length ll))).
    rewrite minus_diag.
    intros l.
    trivial.
  Qed.

  Lemma In_nth_rev : 
    forall(a : A)(l : list A)(n : nat)(n_less : n < length l),
      a = nth l n n_less ->
        In a l.
  Proof.
    intros a l n n_less H.
    erewrite list_split_at_n.
    apply in_or_app.
    right.
    left.
    apply eq_sym.
    eexact H.
  Qed.

End Lists2.

Implicit Arguments In_flatten [A].

Section Lists3.

  Variable A : Type.

  (**************************************************************************)
  (** ***  every via nth  *)
  (**************************************************************************)

  Definition every_nth(P : A -> Prop)(l : list A) : Prop :=
    forall(n : nat)(n_less : n < length l), P (nth l n n_less).

  Lemma every_nth_empty : forall(P : A -> Prop),
    every_nth P [].
  Proof.
    intros P n n_less.
    simpl in n_less.
    exfalso.
    omega.
  Qed.

  Lemma every_nth_mono : forall(P Q : set A)(l : list A),
    subset P Q -> every_nth P l -> every_nth Q l.
  Proof.
    unfold every_nth in *.
    intros P Q l H H0 n n_less.
    apply H.
    apply H0.
  Qed.

  Lemma every_nth_head : forall(P : A -> Prop)(a : A)(l : list A),
    every_nth P (a :: l) -> P a.
  Proof.
    unfold every_nth in *.
    intros P a l H.
    specialize (H 0 (lt_0_Sn (length l))).
    trivial.
  Qed.


  Lemma every_nth_tail : forall(P : A -> Prop)(a : A)(l : list A),
    every_nth P (a :: l) -> every_nth P l.
  Proof.
    unfold every_nth in *.
    intros P a l H n n_less.
    specialize (H (S n) (lt_n_S _ _ n_less)).
    simpl in H.
    erewrite nth_tcc_irr.
    eexact H.
  Qed.

  Lemma every_nth_cons : forall(P : A -> Prop)(a : A)(l : list A),
    P a -> every_nth P l -> every_nth P (a :: l).
  Proof.
    unfold every_nth in *.
    intros P a l H H0 n n_less.
    destruct n.
      trivial.
    apply H0.
  Qed.

  Lemma every_nth_append : forall(P : A -> Prop)(l1 l2 : list A),
    every_nth P l1 -> every_nth P l2 -> every_nth P (l1 ++ l2).
  Proof.
    unfold every_nth in *.
    intros P l1 l2 H H0 n n_less.
    assert(n < length l1 \/ n >= length l1).
      omega.
    destruct H1.
      rewrite (nth_append_left _ _ _ _ H1).
      auto.
    rewrite (nth_append_right _ _ _ _ H1).
    auto.
  Qed.

  Lemma every_nth_append_left : forall(P : A -> Prop)(l1 l2 : list A),
    every_nth P (l1 ++ l2) -> every_nth P l1.
  Proof.
    unfold every_nth in *.
    intros P l1 l2 H n n_less.
    assert (n < length (l1 ++ l2)).
      rewrite app_length.
      omega.
    specialize (H n H0).
    rewrite (nth_append_left _ _ _ _ n_less) in H.
    trivial.
  Qed.

  Lemma every_nth_append_right : forall(P : A -> Prop)(l1 l2 : list A),
    every_nth P (l1 ++ l2) -> every_nth P l2.
  Proof.
    unfold every_nth in *.
    intros P l1 l2 H n n_less.
    assert (length l1 + n < length (l1 ++ l2)).
      rewrite app_length.
      omega.
    specialize (H (length l1 + n) H0).
    assert (length l1 + n >= length l1).
      omega.
    rewrite (nth_append_right _ _ _ _ H1) in H.
    assert (nth l2 (length l1 + n - length l1)
                (nth_append_right_tcc l1 l2 (length l1 + n) H0 H1) 
            = nth l2 n n_less).
      generalize (nth_append_right_tcc l1 l2 (length l1 + n) H0 H1).
      rewrite minus_plus.
      intros l.
      apply nth_tcc_irr.
    rewrite H2 in H.
    trivial.
  Qed.

  Lemma every_nth_In : forall(P : A -> Prop)(l : list A)(a : A),
    every_nth P l -> In a l -> P a.
  Proof.
    induction l.
      intros a H H0.
      contradiction.
    simpl.
    intros a0 H H0.
    destruct H0.
      subst a0.
      eapply every_nth_head.
      eexact H.
    apply IHl.
      eapply every_nth_tail.
      eexact H.
    trivial.
  Qed.

  Lemma every_nth_In_rev : forall(P : A -> Prop)(l : list A),
    (forall(a : A), In a l -> P a) ->
       every_nth P l.
  Proof.
    intros P l H n n_less.
    apply H.
    eapply In_nth_rev.
    trivial.
  Qed.

  Lemma forallb_every_nth : forall(f : A -> bool)(l : list A),
    forallb f l = true  <->  every_nth (fun a => f a = true) l.
  Proof.
    induction l.
      split.
        intros H.
        apply every_nth_empty.
      intros H.
      trivial.
    simpl.
    rewrite andb_true_iff.
    split.
      intros H.
      apply every_nth_cons.
        apply H.
      apply IHl.
      apply H.
    intros H.
    split.
      eapply every_nth_head with (P := fun a => f a = true).
      eexact H.
    apply IHl.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma every_nth_cutout_nth : forall(P : A -> Prop)(l : list A)(n : nat),
    every_nth P l -> every_nth P (cutout_nth l n).
  Proof.
    intros P l n H.
    apply every_nth_append.
      eapply every_nth_append_left.
      rewrite firstn_skipn.
      trivial.
    eapply every_nth_append_right.
    rewrite firstn_skipn.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  map  *)
  (**************************************************************************)

  Lemma map_id : forall(l : list A), map id l = l.
  Proof.
    induction l.
      trivial.
    simpl.
    f_equal.
    trivial.
  Qed.

  Lemma restricted_map_ext :
    forall(B : Type)(f g : A -> B)(l : list A),
      (every_nth (fun(a : A) => f a = g a) l) -> 
        map f l = map g l.
  Proof.
    induction l.
      trivial.
    intros H.
    simpl.
    f_equal.
      apply every_nth_head in H.
      trivial.
    apply IHl.
    apply every_nth_tail in H.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  remove  *)
  (**************************************************************************)

  Lemma length_remove :
    forall(a_eq : eq_type A)(a : A)(l : list A),
      length (remove a_eq a l) <= length l.
  Proof.
    induction l.
      trivial.
    simpl.
    destruct (a_eq a a0).
      omega.
    simpl.
    omega.
  Qed.

  Lemma length_remove_In :
    forall(a_eq : eq_type A)(a : A)(l : list A),
      In a l ->
        length (remove a_eq a l) < length l.
  Proof.
    induction l.
      contradiction.
    intros H.
    destruct H.
      subst a0.
      simpl.
      rewrite eq_equal.
      unfold "<" in *.
      apply le_n_S.
      apply length_remove.
    simpl.
    destruct (a_eq a a0).
      apply IHl in H.
      omega.
    simpl.
    apply lt_n_S.
    apply IHl.
    trivial.
  Qed.

  Lemma In_remove_other :
    forall(a_eq : eq_type A)(a1 a2 : A)(l : list A),
      In a1 l ->
      a1 <> a2 ->
        In a1 (remove a_eq a2 l).
  Proof.
    induction l.
      contradiction.
    intros H H0.
    destruct H.
      subst a.
      simpl.
      rewrite eq_unequal.
        left.
        trivial.
      auto.
    simpl.
    destruct (a_eq a2 a).
      apply IHl; trivial.
    right.
    apply IHl; trivial.
  Qed.


  (**************************************************************************)
  (** ***  seq  *)
  (**************************************************************************)

  Lemma nth_seq : 
    forall(n l i : nat)(i_less : i < length (seq n l)),
      nth (seq n l) i i_less = n + i.
  Proof.
    intros n l.
    revert n.
    induction l.
      intros n i i_less.
      simpl in *.
      omega.
    intros n i i_less.
    destruct i.
      simpl.
      auto.
    simpl.
    rewrite IHl.
    omega.
  Qed.

  (* currently not needed *)
  Lemma NoDup_seq : forall(n len : nat), NoDup (seq n len).
  Proof.
    intros n len.
    revert len n.
    induction len.
      intros n.
      apply NoDup_nil.
    intros n.
    simpl.
    apply NoDup_cons.
      intros H.
      apply In_nth in H.
      decompose [ex and or dep_and] H; clear H.
      rename x into i, a into i_less, b into H.
      rewrite nth_seq in H.
      omega.
    apply IHlen.
  Qed.


  (**************************************************************************)
  (** ***  nat list sum  *)
  (**************************************************************************)
  
  Fixpoint nat_list_sum(l : list nat) : nat :=
    match l with
      | [] => 0
      | n :: l => n + (nat_list_sum l)
    end.

  Lemma nat_list_sum_append : forall(l1 l2 : list nat),
    nat_list_sum(l1 ++ l2) = (nat_list_sum l1) + (nat_list_sum l2).
  Proof.
    induction l1.
      intros l2.
      simpl.
      trivial.
    intros l2.
    simpl.
    rewrite IHl1.
    rewrite plus_assoc.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  nat maximum *)
  (**************************************************************************)
  
  Fixpoint nat_list_max(l : list nat) : nat :=
    match l with
      | [] => 0
      | n :: l => max n (nat_list_max l)
    end.

  (* 
   * Lemma nat_list_max_append : forall(l1 l2 : list nat),
   *   nat_list_max(l1 ++ l2) = max (nat_list_max l1) (nat_list_max l2).
   * Proof.
   *   induction l1.
   *     intros l2.
   *     simpl.
   *     trivial.
   *   intros l2.
   *   simpl.
   *   rewrite <- Max.max_assoc.
   *   rewrite <- IHl1.
   *   trivial.
   *)

  Lemma nat_list_max_pairwise_le :
    forall(l1 l2 : list nat)(len_eq : length l1 = length l2),
      (forall(n : nat)(n_less : n < length l1),
         nth l1 n n_less <= nth l2 n (eq_rect (length l1) (fun m => n < m) 
                                              n_less (length l2) len_eq))
      -> nat_list_max l1 <= nat_list_max l2.
  Proof.
    induction l1.
      intros l2 len_eq H.
      simpl.
      omega.
    intros l2 len_eq H.
    destruct l2.
      discriminate.
    simpl.
    apply max_mono_both.
      apply (H 0 (lt_0_Sn (length l1))).
    simpl in len_eq.
    assert (H0 := eq_add_S _ _ len_eq).
    apply (IHl1 _ H0).
    clear - H.
    intros n0 n_less.
    specialize (H (S n0) (lt_n_S _ _ n_less)).
    simpl in H.
    erewrite nth_tcc_irr.
    erewrite (nth_tcc_irr l2).
    eexact H.
  Qed.


  (**************************************************************************)
  (** ***  list search  *)
  (**************************************************************************)

  Fixpoint list_search(l : list A)(count : nat)(pred : A -> bool) 
                                                               : option nat :=
    match l with 
      | [] => None
      | a :: r =>
        if pred a then Some count
        else list_search r (1 + count) pred
    end.

  Lemma list_search_none : 
    forall(l : list A)(count : nat)(pred : A -> bool)
          (n : nat)(n_less : n < length l),
      list_search l count pred = None ->
        pred (nth l n n_less) = false.
  Proof.
    induction l.
      intros count pred n n_less H.
      exfalso.
      simpl in *.
      omega.
    intros count pred n n_less H.
    simpl in n_less, H.
    destruct n.
      simpl.
      destruct (pred a).
        discriminate.
      trivial.
    destruct (pred a).
      discriminate.
    simpl.
    eapply IHl.
    eexact H.
  Qed.

  Lemma list_search_some_ind : 
    forall(l : list A)(count : nat)(test : A -> bool)(n : nat),
      list_search l count test = Some n ->
        n >= count /\
        n_less # (n - count < length l) /#\
            test (nth l (n - count) n_less) = true.
  Proof.
    induction l.
      intros count test n H.
      exfalso.
      simpl in *.
      discriminate.
    intros count test n H.
    simpl in H.
    destruct (test a) eqn:?.
      inversion H; clear H.
      split.
        omega.
      rewrite minus_diag.
      assert(0 < length (a :: l)).
        simpl.
        apply lt_0_Sn.
      constructor 1 with (a := H).
      trivial.
    specialize(IHl (S count) test n H).
    decompose [and dep_and] IHl; clear IHl.
    split.
      omega.
    destruct (n - count) eqn:?.
      exfalso.
      omega.
    assert (S n0 < length (a :: l)).
      simpl.
      omega.
    constructor 1 with (a := H1).
    simpl.
    assert (n - S count = n0).
      omega.
    generalize (nth_succ_tcc n0 a l H1).
    rewrite <- H2.
    intros l0.
    rewrite (nth_tcc_irr l _ _ a0).
    trivial.
  Qed.

  Lemma list_search_some : 
    forall(l : list A)(test : A -> bool)(n : nat),
      list_search l 0 test = Some n ->
        n_less # n < length l /#\
            test (nth l n n_less) = true.
  Proof.
    intros l test n H.
    rewrite (minus_n_O n).
    assert(H0:=list_search_some_ind l 0 test n H).
    destruct H0.
    trivial.
  Qed.

  Lemma list_search_some_less :
    forall(l : list A)(test : A -> bool)(n : nat),
      list_search l 0 test = Some n ->
        n < length l.
  Proof.
    intros l test n H.
    assert(H0:=list_search_some l test n H).
    destruct H0.
    trivial.
  Qed.

  Lemma list_search_some_test :
    forall(l : list A)(test : A -> bool)(n : nat)
          (H : list_search l 0 test = Some n),
      test (nth l n (list_search_some_less l test n H)) = true.
  Proof.
    intros l test n H.
    assert(H0:=list_search_some l test n H).
    destruct H0.
    erewrite nth_tcc_irr.
    eexact e.
  Qed.


  (**************************************************************************)
  (** ***  swap list of pairs  *)
  (**************************************************************************)

  Definition swap_pairs(B : Type)(l : list (A * B)) : list (B * A) :=
    map swap_pair l.

End Lists3.

Implicit Arguments nth_map_tcc [A B].
Implicit Arguments nth_map [A B].
Implicit Arguments flatten [A].
Implicit Arguments member [A].
Implicit Arguments member_In [A].
Implicit Arguments member_append [A].
Implicit Arguments in_map_reverse [A B].
Implicit Arguments every_nth [A].
Implicit Arguments every_nth_empty [A].
Implicit Arguments every_nth_mono [A].
Implicit Arguments every_nth_cons [A].
Implicit Arguments every_nth_head [A].
Implicit Arguments every_nth_tail [A].
Implicit Arguments every_nth_In [A].
Implicit Arguments list_split_at_n [A].
Implicit Arguments list_search [A].
Implicit Arguments list_search_some [A].
Implicit Arguments list_search_none [A].
Implicit Arguments list_search_some_less [A].
Implicit Arguments list_search_some_test [A].
Implicit Arguments swap_pairs [A B].

Lemma every_nth_map : 
  forall(A B : Type)(P : B -> Prop)(f : A -> B)(l : list A),
    every_nth P (map f l)  <->  every_nth (fun(a : A) => P (f a)) l.
Proof.
  intros A B P f l.
  split.
    intros H.
    intros n n_less.
    assert (n < length (map f l)).
      rewrite map_length.
      trivial.
    specialize (H n H0).
    rewrite nth_map in H.
    erewrite nth_tcc_irr.
    eexact H.
  intros H.
  intros n n_less.
  rewrite nth_map.
  apply H.
Qed.


Lemma nat_list_max_le : forall(l : list nat)(n : nat),
  every_nth (fun(m : nat) => m <= n) l -> nat_list_max l <= n.
Proof.
  induction l.
    intros n H.
    simpl.
    apply le_0_n.
  intros n H.
  assert (H0 := every_nth_head _ _ _ H).
  assert (H1 := every_nth_tail _ _ _ H).
  simpl in *.
  apply Max.max_lub.
    trivial.
  apply IHl.
  trivial.
Qed.

Lemma nat_list_max_le_inv : forall(l : list nat)(n : nat),
  nat_list_max l <= n -> every_nth (fun(m : nat) => m <= n) l. 
Proof.
  induction l.
    intros n H.
    apply every_nth_empty.
  intros n H.
  simpl in H.
  apply every_nth_cons.
    eapply Max.max_lub_l.
    eexact H.
  apply IHl.
  eapply Max.max_lub_r.
  eexact H.
Qed.

Lemma nat_list_max_lub : forall(l : list nat),
  every_nth (fun(n : nat) => n <= nat_list_max l) l.
Proof.
  induction l.
    apply every_nth_empty.
  apply every_nth_cons.
    simpl.
    apply Max.le_max_l.
  intros n n_less.
  simpl.
  eapply le_trans.
    apply IHl.
  apply Max.le_max_r.
Qed.

Lemma fold_left_map :
  forall{X Y Z : Type}(f : X -> Y)(g : Z -> Y -> Z)(l : list X)(z : Z),
    fold_left g (map f l) z = fold_left (fun z x => g z (f x)) l z.
Proof.
  induction l.
    intros z.
    trivial.
  intros z.
  simpl.
  apply IHl.
Qed.



