(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: propositional_completeness.v,v 1.54 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Completeness for propositional logic

      This module proves Proposition 3.2.1, which is also needed as
      base case for the completeness of coalgebric logics.

      I prove prove completeness here by using proof construction. If
      the proof construction fails, [build_proof] returns a
      non-tautological sequent for which I can build a counter model
      in order to derive a contradiction.

      The use of the generic [build_proof] function has the problem
      that the returned proof trees have the wrong type. They are
      proof trees over rule set G of arbitrary rank, instead of over
      the propositional subset of G. One way to fix this problem is to
      use a dependent proof search, which moves a well-formedness
      predicate (here being a propositional sequent) up during the
      search. However, this seemed to complicated.

      I therefore change the rule set of the proof afterwards.
 *)

Require Export classic propositional_sound build_prop_proof.

Section Propositional_Completeness.

  Variable V : Type.
  Variable L : modal_operators.

  (** need a decidable equality on propositional constants for 
      the proof construction 
   *)
  Variable v_eq : eq_type V.

  (**************************************************************************)
  (** *** Simple tautologies *)
  (**************************************************************************)

  Lemma simple_tautology_cons_append :
    forall(v : V)(f g : lambda_formula V L)(s1 s2 : sequent V L),
      (f = lf_prop v /\ g = lf_neg (lf_prop v)) \/
      (f = lf_neg (lf_prop v) /\ g = lf_prop v) ->
        simple_tautology (f :: s1 ++ g :: s2).
  Proof.
    intros v f g s1 s2 H.
    apply simple_tautology_reorder with (s1 := [f; g] ++ s1 ++ s2).
      apply simple_tautology_append_right.
      unfold simple_tautology, simple_tautology_witness in *.
      decompose [and or] H; clear H.
        subst f g.
        exists 0, 1, v.
        apply dep_conj with (a := lt_0_Sn _).
        apply dep_conj with (a := lt_n_S _ _ (lt_0_Sn _)).
        auto.
      subst f g.
      exists 1, 0, v.
      apply dep_conj with (a := lt_n_S _ _ (lt_0_Sn _)).
      apply dep_conj with (a := lt_0_Sn _).
      auto.
    simpl.
    apply list_reorder_cons_head.
    apply list_reorder_cons_parts.
    apply list_reorder_refl.
  Qed.


  (**************************************************************************)
  (** *** [build_proof] for propositional sequents  *)
  (**************************************************************************)

  Definition prop_hyp_oracle : 
                         hypotheses_oracle_type V L (empty_sequent_set V L) :=
    fun _ => None.

  Definition prop_build_proof(s : sequent V L) : 
         (proof (G_set V L) (empty_sequent_set V L) s) + (sequent V L) :=
    build_proof (S (sequent_measure s)) 
      prop_hyp_oracle (prop_G_oracle v_eq) s.

  Definition prop_build_proof_prop_type(s : sequent V L) : Type :=
    propositional_sequent s -> prop_proof_type s.

  Fixpoint apply_propositional_subproofs(sl : list (sequent V L))
        (cond_subproofs : dep_list (sequent V L) prop_build_proof_prop_type sl)
        (prop_subproofs : every_nth propositional_sequent sl)
                                : dep_list (sequent V L) prop_proof_type sl :=
    match cond_subproofs 
      in dep_list _ _ sl
      return every_nth propositional_sequent sl -> 
                 dep_list (sequent V L) prop_proof_type sl
    with
      | dep_nil => fun _ => dep_nil
      | dep_cons s rsl prop_fun_s r_prop_funs =>
        fun(prop_s_rsl : every_nth propositional_sequent (s :: rsl)) =>
          dep_cons s rsl (prop_fun_s (every_nth_head _ _ _ prop_s_rsl))
            (apply_propositional_subproofs rsl r_prop_funs
               (every_nth_tail _ _ _ prop_s_rsl))
    end prop_subproofs.

  Definition prop_build_proof_prop(s : sequent V L)
                      (p : proof (G_set V L) (empty_sequent_set V L) s) :
                                                prop_build_proof_prop_type s :=
    proof_rect V L (G_set V L) (empty_sequent_set V L)
      prop_build_proof_prop_type
      (fun(gamma : sequent V L)(in_hyp : empty_sequent_set V L gamma) _ =>
         assume (prop_G_set V L) (empty_sequent_set V L) gamma in_hyp)
      (fun(r : sequent_rule V L)(in_G : G_set V L r)
          (subproofs : dep_list (sequent V L)
                                prop_build_proof_prop_type
                                (assumptions r))
          (prop_concl : propositional_sequent (conclusion r))
          =>
        rule (prop_G_set V L) (empty_sequent_set V L) r 
             (G_set_prop_set r in_G prop_concl)
             (apply_propositional_subproofs (assumptions r) subproofs 
                (const_rank_G_set 1 r in_G prop_concl)))
      s p.


  (**************************************************************************)
  (** *** Counter models  *)
  (**************************************************************************)

  Definition all_true_model : prop_model V := fun _ => True.

  Definition formula_counter_model(f : lambda_formula V L)
                                  (m : prop_model V) : prop_model V :=
    match f with
      | lf_prop v => function_update v_eq m v False
      | lf_neg (lf_prop v) => function_update v_eq m v True
      | _ => m
    end.

  Fixpoint build_counter_model(l : sequent V L)(m : prop_model V) 
                                                               : prop_model V :=
    match l with
      | [] => m
      | f :: r => build_counter_model r (formula_counter_model f m)
    end.

  Lemma build_counter_model_const_v_ind :
    forall(s1 s2 : sequent V L)(f : lambda_formula V L)
          (m : prop_model V)(v : V),
      (f = lf_prop v \/ f = lf_neg (lf_prop v)) ->
      not (simple_tautology (f :: (s1 ++ s2))) ->
      (formula_counter_model f m v = m v) ->
        ((build_counter_model s2 m v) = (m v)).
  Proof.
    intros s1 s2.
    revert s1.
    induction s2.
      intros s1 f m v H H0 H1.
      simpl.
      trivial.
    intros s1 f m v H H0 H1.
    simpl.
    specialize (IHs2 (s1 ++ [a]) f (formula_counter_model a m) v H).
    rewrite <- app_assoc in IHs2.
    simpl in IHs2.
    specialize (IHs2 H0).
    destruct a.
          destruct H.
            subst f.
            simpl in *.
            rewrite function_update_eq in H1; trivial.
            rewrite IHs2; clear IHs2.
              apply function_update_split_result; trivial.
            clear - H1.
            rewrite function_update_eq; trivial.
            erewrite function_update_split_result; auto.
          subst f.
          simpl in *.
          rewrite function_update_eq in H1; trivial.
          destruct (v_eq v v0).
            subst v0.
            exfalso.
            apply H0; clear.
            eapply simple_tautology_cons_append.
            right.
            eauto.
          rewrite IHs2; clear IHs2.
            apply function_update_unequal.
            auto.
          rewrite function_update_eq; trivial.
          rewrite function_update_unequal; trivial.
          auto.
        destruct a.
              destruct H.
                subst f.
                destruct (v_eq v v0).
                  exfalso.
                  subst v0.
                  apply H0; clear.
                  eapply simple_tautology_cons_append.
                  left.
                  eauto.
                rewrite IHs2; clear IHs2.
                  simpl in *.
                  apply function_update_unequal.
                  auto.
                simpl in *.
                rewrite function_update_eq; trivial.
                rewrite function_update_eq in H1; trivial.
                rewrite function_update_unequal; trivial.
                auto.
              subst f.
              simpl in *.
              rewrite function_update_eq in H1; trivial.
              rewrite IHs2; clear IHs2.
                apply function_update_split_result; trivial.
              rewrite function_update_eq; trivial.
              erewrite function_update_split_result; auto.
            simpl in *.
            apply IHs2; trivial.
          simpl in *.
          apply IHs2; trivial.
        simpl in *.
        apply IHs2; trivial.
      simpl in *.
      apply IHs2; trivial.
    simpl in *.
    apply IHs2; trivial.
  Qed.

  Lemma build_counter_model_const_v :
    forall(s : sequent V L)(f : lambda_formula V L)(m : prop_model V)(v : V),
      (f = lf_prop v \/ f = lf_neg (lf_prop v)) ->
      not (simple_tautology (f :: s)) ->
      (formula_counter_model f m v = m v) ->
        ((build_counter_model s m v) = (m v)).
  Proof.
    intros s f m v H H0 H1.
    eapply (build_counter_model_const_v_ind [] s).
        eexact H.
      simpl.
      eexact H0.
    trivial.
  Qed.

  Lemma formula_counter_model_correct :
    forall(m : prop_model V)(f : lambda_formula V L)
          (prop_f : propositional f)(s : sequent V L),
      neg_form_maybe prop_form f ->
      ~ simple_tautology (f :: s) ->
        ~ is_prop_model (build_counter_model s (formula_counter_model f m)) 
                        f prop_f.
  Proof.
    intros m f prop_f s H H0 H1.
    unfold is_prop_model in *.
    destruct f; try contradiction.
      simpl in *.
      erewrite build_counter_model_const_v in H1.
            rewrite function_update_eq in H1.
            trivial.
          left.
          reflexivity.
        trivial.
      simpl.
      rewrite function_update_twice.
      trivial.
    destruct f; try contradiction.
    simpl in *.
    erewrite build_counter_model_const_v in H1.
          rewrite function_update_eq in H1.
          auto.
        right.
        reflexivity.
      trivial.
    simpl.
    rewrite function_update_twice.
    trivial.
  Qed.

  Lemma build_counter_model_correct :
    forall(s : sequent V L)(prop_s : propositional_sequent s)(m : prop_model V),
      prop_sequent s ->
      (not (simple_tautology s)) ->
        not (prop_sequent_interpretation (build_counter_model s m) s prop_s).
  Proof.
    induction s.
      intros prop_s m H H0 H1.
      apply prop_sequent_interpretation_empty in H1.
      trivial.
    rename a into f.
    intros prop_s m H H0 H1.
    simpl in *.
    apply prop_sequent_interpretation_cons_case_elim in H1.
    decompose [and or] H1; clear H1.
      apply formula_counter_model_correct in H4; trivial.
      apply prop_sequent_head in H.
      trivial.
    apply H4; clear H4; split; intro H4.
      clear IHs.
      apply formula_counter_model_correct in H4; trivial.
      apply prop_sequent_head in H.
      trivial.
    apply (IHs (propositional_sequent_tail _ _ prop_s) 
                (formula_counter_model f m)).
        apply prop_sequent_tail in H.
        trivial.
      apply (contrapositive (simple_tautology_cons _ _) H0).
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  Upward correctness  *)
  (**************************************************************************)
  
  Definition prop_correct(m : prop_model V)(s : sequent V L) : Prop :=
    prop_s # propositional_sequent s /#\ prop_sequent_interpretation m s prop_s.

  Lemma prop_correct_propositional :
    forall(m : prop_model V)(s : sequent V L),
      prop_correct m s -> propositional_sequent s.
  Proof.
    clear. 
    intros m s H.
    unfold prop_correct in *.
    decompose [ex and or dep_and] H; clear H.
    trivial.
  Qed.

  Definition upward_correct_rule(r : sequent_rule V L) : Prop :=
    forall(m : prop_model V),
      prop_correct m (conclusion r) ->
        every_nth (prop_correct m) (assumptions r).

  Lemma upward_correct_context :
    forall(r : sequent_rule V L)(sl sr : sequent V L),
      every_nth propositional_sequent (assumptions r) ->
      every_nth (fun a => a <> []) (assumptions r) ->
      upward_correct_rule r -> 
        upward_correct_rule (rule_add_context sl sr r).
  Proof.
    unfold upward_correct_rule in *.
    intros r sl sr H H0 H1 m H2.
    unfold prop_correct in H2.
    decompose [ex and or dep_and] H2; clear H2.
    rename a into prop_r_context, b into H2.
    simpl in *.
    assert (prop_sl := propositional_add_context_left _ _ _ prop_r_context).
    assert (prop_sr := propositional_add_context_right _ _ _ prop_r_context).
    apply every_nth_map.
    intros n n_less.
    specialize (H n n_less).
    apply dep_conj with 
              (a := (propositional_add_context _ _ _ prop_sl prop_sr H)).
    unfold add_context in *.
    apply prop_sequent_interpretation_length_case_intro.
      intros H3.
      generalize (propositional_add_context sl sr 
                   (nth (assumptions r) n n_less) prop_sl prop_sr H).
      unfold add_context in *.
      specialize (H0 n n_less).
      simpl in H0.
      fold (sequent V L) in H0.
      destruct (nth (assumptions r) n n_less) eqn:H4.
        exfalso.
        auto.
      repeat rewrite app_length in H3.
      destruct sl.
        destruct sr.
          destruct s.
            simpl in *.
            rewrite <- H4.
            lapply (H1 m); clear H1.
              intros H1 prop_ass_n.
              specialize (H1 n n_less).
              unfold prop_correct in H1.
              decompose [ex and or dep_and] H1; clear H1.
              eapply prop_sequent_interpretation_tcc_irr.
              eexact b.
            apply dep_conj with (a := propositional_add_context_propositional
                                          [] _ _ prop_r_context).
            generalize (propositional_add_context_propositional [] []
                             (conclusion r) prop_r_context).
            revert prop_r_context H2.
            rewrite app_nil_r.
            intros prop_r_context H2 p.
            eapply prop_sequent_interpretation_tcc_irr.
            eexact H2.
          simpl in H3.
          exfalso.
          omega.
        simpl in H3.
        exfalso.
        omega.
      simpl in H3.
      exfalso.
      omega.
    clear H0.
    intros H3 H4.
    apply prop_sequent_interpretation_add_context_split in H2.
    apply H2; clear H2; repeat split; intro H2; apply H4; clear H4.
        apply prop_sequent_interpretation_add_context.
        auto.
      apply prop_sequent_interpretation_add_context.
      right.
      left.
      lapply (H1 m).
        intros H0.
        specialize (H0 n n_less).
        unfold prop_correct in H0.
        decompose [ex and or dep_and] H0; clear H0.
        eapply prop_sequent_interpretation_tcc_irr.
        eexact b.
      unfold prop_correct.
      apply dep_conj with (a := propositional_add_context_propositional
                                    _ _ _ prop_r_context).
      apply H2.
    apply prop_sequent_interpretation_add_context.
    auto.
  Qed.


  (***************************************************************************)
  (** **** Ax *)
  (***************************************************************************)

  Lemma upward_correct_ax : forall(r : sequent_rule V L),
      is_ax_rule r -> upward_correct_rule r.
  Proof.
    unfold upward_correct_rule in *.
    intros r H m H0.
    unfold is_ax_rule in *.
    destruct H.
    rewrite H.
    apply every_nth_empty.
  Qed.


  (***************************************************************************)
  (** **** And *)
  (***************************************************************************)

  Lemma upward_propositional_bare_and_left :
    forall(f1 f2 : lambda_formula V L),
      propositional_sequent [lf_and f1 f2] -> propositional_sequent [f1]. 
  Proof.
    clear.
    intros f1 f2 H.
    apply propositional_sequent_cons.
      eapply propositional_and_left.
      eapply propositional_sequent_head.
      eexact H.
    apply propositional_sequent_empty.
  Qed.

  Lemma upward_propositional_bare_and_right :
    forall(f1 f2 : lambda_formula V L),
      propositional_sequent [lf_and f1 f2] -> propositional_sequent [f2]. 
  Proof.
    clear.
    intros f1 f2 H.
    apply propositional_sequent_cons.
      eapply propositional_and_right.
      eapply propositional_sequent_head.
      eexact H.
    apply propositional_sequent_empty.
  Qed.

  Lemma upward_propositional_bare_and :
    forall(f1 f2 : lambda_formula V L),
      propositional_sequent (conclusion (bare_and_rule f1 f2)) ->
        every_nth propositional_sequent
                  (assumptions (bare_and_rule f1 f2)).
  Proof.
    clear. 
    intros f1 f2 H.
    simpl in *.
    apply every_nth_cons.
      eapply upward_propositional_bare_and_left; eauto.
    apply every_nth_cons.
      eapply upward_propositional_bare_and_right; eauto.
    apply every_nth_empty.
  Qed.

  Lemma upward_correct_and : forall(r : sequent_rule V L),
    is_and_rule r -> upward_correct_rule r.
  Proof.
    unfold upward_correct_rule in *.
    intros r H m H0.
    apply and_rule_context in H.
    decompose [ex] H; clear H.
    subst r.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    apply upward_correct_context; auto.
        apply upward_propositional_bare_and.
        eapply propositional_add_context_propositional.
        apply prop_correct_propositional with (m := m).
        simpl in H0.
        eexact H0.
      simpl.
      apply every_nth_cons.
        discriminate.
      apply every_nth_cons.
        discriminate.
      apply every_nth_empty.
    clear.
    unfold upward_correct_rule in *.
    intros m H.
    simpl in *.
    unfold prop_correct in H.
    decompose [ex and or dep_and] H; clear H.
    apply every_nth_cons.
      unfold prop_correct in *.
      constructor 1 with (a := upward_propositional_bare_and_left f1 f2 a).
      rewrite prop_sequent_interpretation_singleton in *.
      simpl in *.
      destruct b.
      eapply is_prop_model_tcc_irr.
      eexact H.
    apply every_nth_cons.
      unfold prop_correct in *.
      constructor 1 with (a := upward_propositional_bare_and_right f1 f2 a).
      rewrite prop_sequent_interpretation_singleton in *.
      simpl in *.
      destruct b.
      eapply is_prop_model_tcc_irr.
      eexact H0.
    apply every_nth_empty.
  Qed.


  (***************************************************************************)
  (** **** Neg And *)
  (***************************************************************************)

  Lemma upward_propositional_bare_neg_and_sequent :
    forall(f1 f2 : lambda_formula V L),
      propositional_sequent [lf_neg (lf_and f1 f2)] -> 
        propositional_sequent [lf_neg f1; lf_neg f2]. 
  Proof.
    clear.
    intros f1 f2 H.
    assert(H0 := propositional_sequent_head [] _ H).
    apply propositional_sequent_cons.
      apply propositional_neg_inv.
      eapply propositional_and_left.
      apply propositional_neg.
      eexact H0.
    apply propositional_sequent_cons.
      apply propositional_neg_inv.
      eapply propositional_and_right.
      apply propositional_neg.
      eexact H0.
    apply propositional_sequent_empty.
  Qed.

  Lemma upward_propositional_bare_neg_and :
    forall(f1 f2 : lambda_formula V L),
      propositional_sequent (conclusion (bare_neg_and_rule f1 f2)) ->
        every_nth propositional_sequent 
                  (assumptions (bare_neg_and_rule f1 f2)).
  Proof.
    clear. 
    intros f1 f2 H.
    simpl in *.
    apply every_nth_cons.
      eapply upward_propositional_bare_neg_and_sequent; eauto.
    apply every_nth_empty.
  Qed.

  Lemma upward_correct_neg_and : forall(r : sequent_rule V L),
    is_neg_and_rule r -> 
      upward_correct_rule r.
  Proof.
    clear v_eq. 
    unfold upward_correct_rule in *.
    intros r H m H0.
    apply neg_and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    subst r.
    apply upward_correct_context; trivial.
        apply upward_propositional_bare_neg_and.
        eapply propositional_add_context_propositional.
        apply prop_correct_propositional with (m := m).
        simpl in H0.
        eexact H0.
      simpl.
      apply every_nth_cons.
        discriminate.
      apply every_nth_empty.
    clear.
    unfold upward_correct_rule in *.
    intros m H.
    simpl in *.
    unfold prop_correct in H.
    decompose [ex and or dep_and] H; clear H.
    apply every_nth_cons.
      unfold prop_correct in *.
      apply dep_conj
           with (a := upward_propositional_bare_neg_and_sequent f1 f2 a).
      apply prop_sequent_interpretation_cons_cons_intro.
      intros H.
      apply prop_sequent_interpretation_singleton in b.
      simpl in *.
      assert (H1 := contrapositive 
               (prop_sequent_interpretation_nth_intro m [_ ; _] 0 
                      (lt_0_Sn 1) _) H).
      apply (contrapositive 
               (prop_sequent_interpretation_nth_intro m [_ ; _] 1
                      (lt_n_S _ _ (lt_0_Sn 0)) _)) in H.
      simpl in *.
      apply H1; clear H1; intros H1.
      apply H; clear H; intros H.
      apply b; clear b; split.
        eapply is_prop_model_tcc_irr.
        eexact H1.
      eapply is_prop_model_tcc_irr.
      eexact H.
    apply every_nth_empty.
  Qed.


  (***************************************************************************)
  (** **** Neg Neg *)
  (***************************************************************************)

  Lemma upward_propositional_bare_neg_neg_sequent :
    forall(f : lambda_formula V L),
      propositional_sequent [lf_neg (lf_neg f)] -> 
        propositional_sequent [f]. 
  Proof.
    clear.
    intros f H.
    assert(H0 := propositional_sequent_head [] _ H).
    apply propositional_sequent_cons.
      simpl in *.
      tauto.
    apply every_nth_empty.
  Qed.

  Lemma upward_propositional_bare_neg_neg :
    forall(f : lambda_formula V L),
      propositional_sequent (conclusion (bare_neg_neg_rule f)) ->
        every_nth propositional_sequent
                  (assumptions (bare_neg_neg_rule f)).
  Proof.
    clear. 
    intros f H.
    simpl in *.
    apply every_nth_cons.
      eapply upward_propositional_bare_neg_neg_sequent; eauto.
    apply every_nth_empty.
  Qed.

  Lemma upward_correct_neg_neg : forall(r : sequent_rule V L),
    classical_logic ->
    is_neg_neg_rule r -> 
      upward_correct_rule r.
  Proof.
    clear v_eq.
    unfold upward_correct_rule in *.
    intros r classic H m H0.
    apply neg_neg_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f.
    subst r.
    apply upward_correct_context; trivial.
        apply upward_propositional_bare_neg_neg.
        eapply propositional_add_context_propositional.
        apply prop_correct_propositional with (m := m).
        simpl in H0.
        eexact H0.
      simpl.
      apply every_nth_cons.
        discriminate.
      apply every_nth_empty.
    clear - classic.
    unfold upward_correct_rule in *.
    intros m H.
    simpl in *.
    unfold prop_correct in H.
    decompose [ex and or dep_and] H; clear H.
    apply every_nth_cons.
      unfold prop_correct in *.
      apply dep_conj
           with (a := upward_propositional_bare_neg_neg_sequent f a).
      rewrite prop_sequent_interpretation_singleton in *.
      simpl in *.
      apply classic in b.
      eapply is_prop_model_tcc_irr.
      eexact b.
    apply every_nth_empty.
  Qed.


  (***************************************************************************)
  (** **** Whole G *)
  (***************************************************************************)

  Lemma correct_rule_inductive_G : forall(m : prop_model V),
    classical_logic ->
      rule_inductive (G_set V L) (prop_correct m).
  Proof.
    clear. 
    intros m.
    unfold rule_inductive in *.
    intros classic r H H0.
    unfold G_set, union in *.
    decompose [or] H; clear H.
          apply upward_correct_ax; auto.
        apply upward_correct_and; auto.
      apply upward_correct_neg_and; auto.
    apply upward_correct_neg_neg; auto.
  Qed.


  (**************************************************************************)
  (** ***  Completeness  *)
  (**************************************************************************)
  
  Lemma propositional_complete_G : 
    forall(nonempty_v : V)(s : sequent V L)
          (prop_s : propositional_sequent s),
      classical_logic ->
      prop_valid_sequent nonempty_v s prop_s -> 
        provable (prop_G_set V L) (empty_sequent_set V L) s.
  Proof.
    intros nonempty_v s prop_s classic H.
    destruct (prop_build_proof s) eqn:H0.
      exists (prop_build_proof_prop s p prop_s).
      trivial.
    exfalso.
    unfold prop_build_proof in *.
    assert (H1 := well_founded_G_oracle V L v_eq).
    assert (H2 := H0).
    apply build_proof_right_result with (measure := sequent_measure) in H2;
          auto.
    destruct H2.
    clear H3.
    apply build_proof_right_property with (measure := sequent_measure)
            (P := prop_correct (build_counter_model s0 all_true_model))
            in H0; auto.
        unfold prop_correct in *.
        decompose [ex and or dep_and] H0; clear H0.
        apply build_counter_model_correct in b.
            trivial.
          eapply prop_G_oracle_None_simple; eauto.
        eapply prop_G_oracle_None_tautology; eauto.
      apply correct_rule_inductive_G.
      trivial.
    unfold prop_correct in *.
    apply dep_conj with (a := prop_s).
    rewrite prop_model_sequent_interpretation.
    apply H.
  Qed.


  (* Proposition 3.2.1, page 10 *)
  Theorem prop_3_2_1 :
    forall(nonempty_v : V)(s : sequent V L)(prop_s : propositional_sequent s),
      classical_logic ->
        (prop_valid_sequent nonempty_v s prop_s
           <-> 
         provable (prop_G_set V L) (empty_sequent_set V L) s).
  Proof.
    clear v_eq.
    intros nonempty_v s prop_s classic.
    split.
      apply propositional_complete_G; trivial.
    apply propositional_sound_G; trivial.
  Qed.


End Propositional_Completeness.

Implicit Arguments build_counter_model_correct [V L].
