(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: reorder.v,v 1.24 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Reordering of lists

      This module defines [list_reorder] as equivalence relation on
      lists containing the same elements with same multiplicity in
      different order. This is the intended equality on sequents.

*)


Require Export lists.

Section Reorder.
  Variable A : Type.

  (**************************************************************************)
  (** *** Definition  *)
  (**************************************************************************)

  Inductive list_reorder : list A -> list A -> Prop :=
    | list_reorder_nil : list_reorder [] []
    | list_reorder_cons : forall(a : A)(l1 l2 : list A)(n : nat),
        list_reorder l1 l2 -> 
          list_reorder (a :: l1) ((firstn n l2) ++ a :: (skipn n l2)).

  Lemma list_reorder_cons_parts : forall(l1 l2l l2r : list A)(a : A),
    list_reorder l1 (l2l ++ l2r) ->
      list_reorder (a :: l1) (l2l ++ a :: l2r).
  Proof.
    intros l1 l2l l2r a H.
    assert (l2l = firstn (length l2l) (l2l ++ l2r)).
      rewrite firstn_append_left.
        rewrite firstn_whole.
          trivial.
        omega.
      trivial.
    assert (l2r = skipn (length l2l) (l2l ++ l2r)).
      rewrite skipn_append_right.
        rewrite minus_diag.
        trivial.
      omega.
    rewrite H0.
    rewrite H1 at 2.
    apply list_reorder_cons.
    trivial.
  Qed.

  Lemma list_reorder_cons_head : forall(l1 l2 : list A)(a : A),
    list_reorder l1 l2 ->
      list_reorder (a :: l1) (a :: l2).
  Proof.
    intros l1 l2 a H.
    apply list_reorder_cons_parts with (l2l := []).
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Equivalence *)
  (***************************************************************************)

  Lemma list_reorder_refl : forall(l : list A), list_reorder l l.
  Proof.
    induction l.
      apply list_reorder_nil.
    assert (a :: l = (firstn 0 l) ++ a :: (skipn 0 l)).
      trivial.
    rewrite H at 2.
    eapply list_reorder_cons.
    trivial.
  Qed.

  Lemma list_reorder_length : forall(l1 l2 : list A),
    list_reorder l1 l2 -> length l1 = length l2.
  Proof.
    intros l1 l2 H.
    induction H.
      trivial.
    rewrite app_length.
    simpl.
    rewrite <- plus_n_Sm.
    rewrite <- app_length.
    rewrite firstn_skipn.
    omega.
  Qed.

  Lemma list_reorder_symm : 
    forall(l1 l2 : list A), list_reorder l1 l2 -> list_reorder l2 l1.
  Proof.
    induction 1.
      apply list_reorder_nil.
    clear H.
    revert l1 l2 IHlist_reorder.
    induction n.
      intros l1 l2 IHlist_reorder.
      simpl.
      assert (a :: l1 = firstn 0 l1 ++ a :: skipn 0 l1).
        trivial.
      rewrite H.
      apply list_reorder_cons.
      trivial.
    intros l1 l2 IHlist_reorder.
    destruct l2.
      simpl.
      assert (H := list_reorder_length _ _ IHlist_reorder).
      destruct l1.
        assert ([a] = firstn 0 [] ++ a :: skipn 0 []).
          trivial.
        rewrite H0 at 2.
        apply list_reorder_cons.
        apply list_reorder_nil.
      simpl in H.
      discriminate.
    simpl.
    remember (a0 :: l2).
    destruct IHlist_reorder.
      discriminate.
    inversion Heql; clear Heql.
    subst a0 l1.
    rename l0 into l1.
    assert (a :: firstn n0 l1 ++ a1 :: skipn n0 l1
             = firstn (1 + n0) (a :: l1) ++ a1 :: skipn (1 + n0) (a :: l1)).
      simpl.
      trivial.
    rewrite H.
    apply list_reorder_cons.
    apply IHn.
    trivial.
  Qed.


  Lemma list_reorder_occurence_full : 
    forall(l1 l2 : list A)(n1 : nat)(n1_less : n1 < length l1),
      list_reorder l1 l2 ->
      exists(n2 : nat),
        n2_less # n2 < length l2 /#\ 
          nth l1 n1 n1_less = nth l2 n2 n2_less /\
          list_reorder (cutout_nth l1 n1) (cutout_nth l2 n2).
  Proof.
    intros l1 l2 n1.
    revert l1 l2.
    induction n1.
      intros l1 l2 n1_less H.
      destruct H.
        exfalso.
        simpl in n1_less.
        omega.
      exists (length (firstn n l2)).
      assert (length (firstn n l2) < length (firstn n l2 ++ a :: skipn n l2)).
        rewrite app_length.
        simpl.
        omega.
      constructor 1 with (a := H0).
      split.
        rewrite (nth_append_right _ _ _ H0 (ge_refl (length (firstn n l2)))).
        generalize (nth_append_right_tcc (firstn n l2) (a :: skipn n l2)
                    (length (firstn n l2)) H0 (ge_refl (length (firstn n l2)))).
        rewrite minus_diag.
        intros l.
        trivial.
      unfold cutout_nth in *.
      rewrite firstn_append_left; auto.
      rewrite (firstn_whole (firstn n l2)); auto.
      rewrite skipn_append_right; auto.
      rewrite (plus_comm 1 (length (firstn n l2))).
      rewrite minus_plus.
      simpl.
      rewrite firstn_skipn.
      trivial.
    intros l1 l2 n1_less H.
    unfold cutout_nth in *.
    destruct H.
      exfalso.
      simpl in n1_less.
      omega.
    simpl nth.
    specialize (IHn1 l1 l2 (nth_succ_tcc n1 a l1 n1_less) H).
    decompose [ex and or dep_and] IHn1; clear IHn1.
    rewrite H0.
    clear - H1.
    assert (n <= length l2 \/ n > length l2).
      omega.
    destruct H.
      assert (H2 := length_firstn_less _ _ H).
      assert (x < n \/ x >= n).
        omega.
      destruct H0.
        exists x.
        assert (x < length (firstn n l2 ++ a :: skipn n l2)).
          rewrite app_length.
          omega.
        constructor 1 with (a := H3).
        assert (x < length (firstn n l2)).
          omega.
        split.
          rewrite (nth_append_left _ _ _ H3 H4).
          rewrite nth_firstn with (n2_less_n1 := H0).
          apply nth_tcc_irr.
        assert (firstn (S n1) (a :: l1) ++ skipn (1 + S n1) (a :: l1)
                = a :: (firstn n1 l1 ++ skipn (1 + n1) l1)).
          trivial.
        rewrite H5; clear H5.
        assert ((firstn x (firstn n l2 ++ a :: skipn n l2) ++ 
                   skipn (1 + x) (firstn n l2 ++ a :: skipn n l2))
                = (firstn (pred n) (firstn x l2 ++ skipn (1 + x) l2) ++
                     a :: skipn (pred n) (firstn x l2 ++ skipn (1 + x) l2))).
          rewrite firstn_append_left; try omega.
          rewrite firstn_firstn_less; try omega.
          assert (length (firstn x l2) = x).
            rewrite length_firstn_less.
              trivial.
            omega.
          rewrite firstn_append_right; try omega.
          rewrite <- app_assoc.
          f_equal.
          rewrite H5.
          rewrite skipn_append_left; try omega.
          rewrite skipn_firstn; try omega.
          f_equal.
            f_equal.
            omega.
          f_equal.
          rewrite skipn_append_right; try omega.
          rewrite H5.
          rewrite skipn_skipn.
          f_equal.
          omega.
        rewrite H5; clear H5.
        apply list_reorder_cons.
        trivial.
      exists (1 + x).
      assert (1 + x < length (firstn n l2 ++ a :: skipn n l2)).
        rewrite app_length.
        simpl.
        rewrite <- plus_n_Sm.
        rewrite <- app_length.
        rewrite firstn_skipn.
        omega.
      constructor 1 with (a := H3).
      assert (1 + x >= length (firstn n l2)).
        omega.
      split.
        rewrite (nth_append_right _ _ _ H3 H4).
        generalize (nth_append_right_tcc (firstn n l2) 
                      (a :: skipn n l2) (1 + x) H3 H4).
        rewrite H2.
        assert (1 + x - n = 1 + (x - n)).
          omega.
        rewrite H5; clear H5.
        simpl.
        intros l.
        rewrite (nth_skipn _ _ _ H).
        generalize (nth_skipn_tcc A l2 n (x - n) H 
                     (nth_succ_tcc (x - n) a (skipn n l2) l)).
        rewrite le_plus_minus_r.
          apply nth_tcc_irr.
        trivial.
      assert (firstn (S n1) (a :: l1) ++ skipn (1 + S n1) (a :: l1)
               = a:: (firstn n1 l1 ++ skipn (1 + n1) l1)).
        trivial.
      rewrite H5; clear H5.
      assert ((firstn (1 + x) (firstn n l2 ++ a :: skipn n l2) ++
                 skipn (1 + (1 + x)) (firstn n l2 ++ a :: skipn n l2))
              = (firstn n (firstn x l2 ++ skipn (1 + x) l2) ++
                 a :: skipn n (firstn x l2 ++ skipn (1 + x) l2))).
        assert (length (firstn x l2) = x).
          rewrite length_firstn_less.
            trivial.
          omega.
        rewrite firstn_append_right; try omega.
        rewrite H2.
        rewrite firstn_append_left; try omega.
        rewrite firstn_firstn_less; try omega.
        rewrite <- app_assoc.
        f_equal.
        assert (1 + x - n = S (x - n)).
          omega.
        rewrite H6; clear H6.
        simpl (firstn (S (x - n)) (a :: skipn n l2)).
        rewrite <- app_comm_cons.
        f_equal.
        rewrite skipn_append_right; try omega.
        rewrite skipn_append_left; try omega.
        rewrite skipn_firstn; try omega.
        f_equal.
        rewrite H2.
        assert (1 + (1 + x) - n = S (1 + x - n)).
          omega.
        rewrite H6; clear H6.
        assert (skipn (S (1 + x - n)) (a :: skipn n l2)
          = skipn (1 + x - n) (skipn n l2)).
          trivial.
        rewrite H6; clear H6.
        rewrite skipn_skipn.
        f_equal.
        omega.
      rewrite H5; clear H5.
      apply list_reorder_cons.
      trivial.
    rewrite firstn_whole.
      rewrite skipn_whole.
        exists x.
        assert (x < length (l2 ++ [a])).
          rewrite app_length.
          omega.
        constructor 1 with (a := H0).
        split.
          rewrite nth_append_left with (n_less_l1 := a0).
          trivial.
        assert (firstn (S n1) (a :: l1) ++ skipn (1 + S n1) (a :: l1)
                = a:: firstn n1 l1 ++ skipn (1 + n1) l1).
          trivial.
        rewrite H2; clear H2.
        assert ((firstn x (l2 ++ [a]) ++ skipn (1 + x) (l2 ++ [a]))
                = (firstn (pred (length l2)) (firstn x l2 ++ skipn (1 + x) l2) 
                   ++ a :: skipn (pred (length l2)) 
                                (firstn x l2 ++ skipn (1 + x) l2))).
          rewrite firstn_append_left; try omega.
          assert (pred (length l2) >= length (firstn x l2 ++ skipn (1 + x) l2)).
            rewrite app_length.
            rewrite length_skipn.
            rewrite length_firstn_less.
              omega.
            omega.
          rewrite (firstn_whole _ (pred (length l2))); trivial.
          rewrite <- app_assoc.
          f_equal.
          rewrite skipn_append_left; try omega.
          f_equal.
          f_equal.
          rewrite skipn_whole; trivial.
        rewrite H2; clear H2.
        apply list_reorder_cons.
        trivial.
      omega.
    omega.
  Qed.

  Lemma list_reorder_occurence : 
    forall(l1 l2 : list A)(n1 : nat)(n1_less : n1 < length l1),
      list_reorder l1 l2 ->
      exists(n2 : nat),
        n2_less # n2 < length l2 /#\ 
          nth l1 n1 n1_less = nth l2 n2 n2_less.
  Proof.
    intros l1 l2 n1 n1_less H.
    assert (H0 := list_reorder_occurence_full l1 l2 n1 n1_less H).
    decompose [ex and or dep_and] H0; clear H0.
    exists x.
    constructor 1 with (a := a).
    trivial.
  Qed.


  Lemma list_reorder_trans :
    forall(l1 l2 l3 : list A), 
      list_reorder l1 l2 -> list_reorder l2 l3 -> list_reorder l1 l3.
  Proof.
    induction l1.
      intros l2 l3 H H0.
      assert (H1 := list_reorder_length _ _ H).
      assert (H2 := list_reorder_length _ _ H0).
      destruct l3.
        apply list_reorder_nil.
      destruct l2.
        discriminate.
      discriminate.
    intros l2 l3 H H0.
    remember (a :: l1) as al1.
    destruct H.
      discriminate.
    inversion Heqal1; clear Heqal1.
    subst a0 l0.
    assert (length (firstn n l2) < length (firstn n l2 ++ a :: skipn n l2)).
      rewrite app_length.
      simpl.
      omega.
    assert (H2 := list_reorder_occurence_full _ _ _ H1 H0).
    decompose [ex and or dep_and] H2; clear H2.
    unfold cutout_nth in *.
    rewrite firstn_append_left in H4; auto.
    rewrite firstn_whole in H4; auto.
    rewrite skipn_append_right in H4; auto.
    rewrite plus_comm in H4.
    rewrite minus_plus in H4.
    simpl (skipn 1 (a :: skipn n l2)) in H4.
    rewrite firstn_skipn in H4.
    assert (l3 = firstn x (firstn x l3 ++ skipn (1 + x) l3) 
               ++ (nth l3 x a0) :: skipn x (firstn x l3 ++ skipn (1 + x) l3)).
      assert (length (firstn x l3) = x).
        rewrite length_firstn_less.
          trivial.
        omega.
      rewrite firstn_append_left.
        rewrite firstn_whole.
          rewrite skipn_append_right.
            rewrite length_firstn_less.
              rewrite minus_diag.
              simpl (skipn 0).
              apply list_split_at_n.
            omega.
          rewrite H2.
          omega.
        rewrite H2.
        omega.
      rewrite H2.
      omega.
    rewrite H2; clear H2.
    assert (nth l3 x a0 = a).
      rewrite <- H3.
      clear. 
      rewrite nth_append_right with (n_greater := ge_refl _).
      generalize (nth_append_right_tcc (firstn n l2) (a :: skipn n l2)
                   (length (firstn n l2)) H1 (ge_refl (length (firstn n l2)))).
      rewrite minus_diag.
      trivial.
    rewrite H2.
    apply list_reorder_cons.
    eapply IHl1; eauto.
  Qed.

  Lemma list_reorder_trans_rev :
    forall(l1 l2 l3 : list A), 
      list_reorder l2 l3 -> list_reorder l1 l2 -> list_reorder l1 l3.
  Proof.
    intros l1 l2 l3 H H0.
    eapply list_reorder_trans; eauto.
  Qed.

  (***************************************************************************)
  (** *** Generic properties *)
  (***************************************************************************)

  Lemma every_nth_list_reorder :
    forall(P : A -> Prop)(l1 l2 : list A),
      list_reorder l1 l2 ->
        (every_nth P l1 <-> every_nth P l2).
  Proof.
    intros P l1 l2 H.
    split.
      unfold every_nth in *.
      intros H0 n n_less.
      lapply (list_reorder_occurence l2 l1 n n_less).
        intros H1.
        decompose [ex and or dep_and] H1; clear H1.
        rewrite b.
        apply H0.
      apply list_reorder_symm.
      trivial.
    unfold every_nth in *.
    intros H0 n n_less.
    lapply (list_reorder_occurence l1 l2 n n_less).
      intros H1.
      decompose [ex and or dep_and] H1; clear H1.
      rewrite b.
      apply H0.
    trivial.
  Qed.

  Lemma list_reorder_insert : forall(l1l l1r l2l l2r : list A)(a1 a2 : A),
    list_reorder (l1l ++ l1r) (l2l ++ l2r) ->
    a1 = a2 ->
      list_reorder (l1l ++ a1 :: l1r) (l2l ++ a2 :: l2r).
  Proof.
    intros l1l l1r l2l l2r a1 a2 H H0.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      apply list_reorder_cons_parts.
      apply list_reorder_refl.
    rewrite H0.
    apply list_reorder_cons_parts.
    trivial.
    (* dependent evars bug ? *)
  Qed.

  Lemma list_reorder_insert_list : forall(l1l l1r l2l l2r lm1 lm2 : list A),
    list_reorder (l1l ++ l1r) (l2l ++ l2r) ->
    list_reorder lm1 lm2 ->
      list_reorder (l1l ++ lm1 ++ l1r) (l2l ++ lm2 ++ l2r).
  Proof.
    induction 2.
      simpl.
      trivial.
    simpl.
    rewrite <- app_assoc.
    simpl.
    rewrite app_assoc.
    apply list_reorder_insert.
      rewrite <- app_assoc.
      rewrite app_assoc with (l := firstn n l2).
      rewrite firstn_skipn.
      trivial.
    trivial.
  Qed.

  Lemma list_reorder_tail : forall(l1 l2l l2r : list A)(a : A),
    list_reorder (a :: l1) (l2l ++ a :: l2r) ->
      list_reorder l1 (l2l ++ l2r).
  Proof.
    intros l1 l2l l2r a H.
    assert (H0 := list_reorder_symm _ _ H).
    clear H.
    assert (length l2l < length (l2l ++ a :: l2r)).
      rewrite app_length.
      simpl.
      omega.
    assert (H1 := list_reorder_occurence_full _ _ _ H H0).
    decompose [ex and or dep_and] H1; clear H1.
    assert (nth (a :: l1) x a0 = a).
      clear H3.
      rewrite <- H2.
      rewrite nth_append_right with (n_greater := ge_refl _).
      generalize (nth_append_right_tcc l2l (a :: l2r) (length l2l) H 
                     (ge_refl (length l2l))).
      rewrite minus_diag.
      trivial.
    clear H2.
    unfold cutout_nth in *.
    rewrite firstn_append_left in H3.
      rewrite firstn_whole in H3.
        rewrite skipn_append_right in H3.
          rewrite add_minus_diag in H3.
          simpl in H3.
          destruct x.
            simpl in H3.
            apply list_reorder_symm.
            trivial.
          change (firstn (S x) (a :: l1) ++ skipn (S x) l1) 
             with (a :: (firstn x l1 ++ skipn (1 + x) l1)) in H3.
          simpl in H1.
          assert (H4 := list_reorder_symm _ _ H3).
          clear H3.
          assert (H5 := list_reorder_occurence_full 
                          (a :: (firstn x l1 ++ skipn (1 + x) l1)) _ _
                          (lt_0_Sn (length (firstn x l1 ++ skipn (1 + x) l1)))
                          H4).
          decompose [ex and or dep_and] H5; clear H5.
          simpl in H2.
          change (cutout_nth (a :: firstn x l1 ++ skipn (1 + x) l1) 0)
            with (firstn x l1 ++ skipn (1 + x) l1) in H3.
          lapply (list_reorder_insert _ _ _ _ 
                         (nth l1 x (nth_succ_tcc x a l1 a0))
                         (nth (l2l ++ l2r) x0 a1)
                         H3).
            intros H5.
            rewrite <- list_split_at_n in H5.
            rewrite <- list_split_at_n in H5.
            trivial.
          rewrite H1.
          trivial.
        omega.
      apply ge_refl.
    apply le_refl.
  Qed.

  Lemma list_reorder_tail_head : forall(l1 l2 : list A)(a : A),
    list_reorder (a :: l1) (a :: l2) ->
      list_reorder l1 l2.
  Proof.
    intros l1 l2 a H.
    eapply list_reorder_tail with (l2l := []); eauto.
  Qed.


  Lemma list_reorder_inserted_2 :
    forall(l1l l1m l1r l2 : list A)(a1l a1r : A),
      list_reorder (l1l ++ a1l :: l1m ++ a1r :: l1r) l2 ->
        exists(l2l l2m l2r : list A)(a2l a2r : A),
          l2 = l2l ++ a2l :: l2m ++ a2r :: l2r /\
          list_reorder (l1l ++ l1m ++ l1r) (l2l ++ l2m ++ l2r) /\
          ((a1l = a2l /\ a1r = a2r) \/
           (a1l = a2r /\ a1r = a2l)).
  Proof.
    intros l1l l1m l1r l2 a1l a1r H.
    assert (length l1l < length (l1l ++ a1l :: l1m ++ a1r :: l1r)).
      rewrite app_length.
      simpl.
      omega.
    assert (H1 := list_reorder_occurence_full _ _ _ H0 H).
    decompose [ex and or dep_and] H1; clear H1.
    unfold cutout_nth in *.
    assert (nth l2 x a = a1l).
      rewrite <- H2.
      clear. 
      rewrite nth_append_right with (n_greater := ge_refl (length l1l)).
      generalize (nth_append_right_tcc l1l (a1l :: l1m ++ a1r :: l1r) 
                       (length l1l) H0 (ge_refl (length l1l))).
      rewrite minus_diag.
      simpl.
      trivial.
    clear H2.
    rewrite firstn_append_left with (1 := le_refl _) in H3.
    rewrite firstn_whole with (1 := ge_refl _) in H3.
    rewrite skipn_append_right in H3.
      rewrite add_minus_diag in H3.
      simpl (skipn 1 (a1l :: l1m ++ a1r :: l1r)) in H3.
      assert (length (l1l ++ l1m) < length (l1l ++ l1m ++ a1r :: l1r)).
        repeat rewrite app_length.
        simpl.
        omega.
      assert (H4 := list_reorder_occurence_full _ _ _ H2 H3).
      unfold cutout_nth in *.
      decompose [ex and or dep_and] H4; clear H4.
      assert (nth (firstn x l2 ++ skipn (1 + x) l2) x0 a0 = a1r).
        rewrite <- H5.
        clear. 
        revert H2.
        rewrite app_assoc.
        intros H1.
        rewrite nth_append_right 
                    with (n_greater := ge_refl (length (l1l ++ l1m))).
        generalize (nth_append_right_tcc (l1l ++ l1m) (a1r :: l1r) 
                      (length (l1l ++ l1m)) H1 (ge_refl (length (l1l ++ l1m)))).
        rewrite minus_diag.
        trivial.
      clear H5.
      rewrite app_assoc in H6.
      rewrite firstn_append_left with (1 := le_refl _) in H6.
      rewrite firstn_whole with (1 := ge_refl _) in H6.
      rewrite skipn_append_right in H6.
        rewrite add_minus_diag in H6.
        simpl (skipn 1 (a1r :: l1r)) in H6.
        assert (H7 := lt_le_weak _ _ a).
        assert (H8 := length_firstn_less l2 x H7).
        assert (x <= x0 \/ x > x0).
          omega.
        destruct H5.
          assert (x0 >= length (firstn x l2)).
            rewrite H8.
            trivial.
          exists (firstn x l2), (firstn (x0 - x) (skipn (1 + x) l2)),
                 (skipn (1 + x0 - x) (skipn (1 + x) l2)), a1l, a1r.
          repeat split; trivial.
              rewrite (plus_minus_assoc 1 x0 x); auto with arith.
              assert (x0 - x < length (skipn (1 + x) l2)).
                clear - a a0 H5 H8.
                rewrite app_length in a0.
                rewrite H8 in *.
                omega.
              rewrite list_split_n_equal with (n_less := H10).
                erewrite list_split_n_equal.
                  trivial.
                eexact H1.
              rewrite <- H4.
              rewrite nth_append_right with (n_greater := H9).
              generalize (nth_append_right_tcc (firstn x l2) 
                              (skipn (1 + x) l2) x0 a0 H9).
              rewrite H8.
              apply nth_tcc_irr.
            rewrite firstn_append_right in H6; trivial.
            rewrite skipn_append_right in H6.
              rewrite length_firstn_less in H6; trivial.
              repeat rewrite <- app_assoc in H6.
              trivial.
            rewrite H8.
            auto with arith.
          left.
          auto.
        assert (x0 < length (firstn x l2)).
          rewrite H8.
          trivial.
        exists (firstn x0 l2), (skipn (1 + x0) (firstn x l2)), 
               (skipn (1 + x) l2), a1r, a1l.
        repeat split; trivial.
            rewrite skipn_firstn; auto with arith.
            assert (skipn (1 + x) l2 = 
                      skipn (1 + (x - (1 + x0))) (skipn (1 + x0) l2)).
              rewrite skipn_skipn.
              f_equal.
              omega.
            rewrite H10.
            assert ((x - (1 + x0)) < length (skipn (1 + x0) l2)).
              rewrite length_skipn.
              omega.
            rewrite list_split_n_equal with (n_less := H11).
              erewrite list_split_n_equal.
                trivial.
              rewrite <- H4.
              rewrite nth_append_left with (n_less_l1 := H9).
              rewrite nth_firstn with (n2_less_n1 := H5).
              trivial.
            assert (1 + x0 <= length l2).
              omega.
            rewrite nth_skipn with (n1_less := H12).
            generalize (nth_skipn_tcc A l2 (1 + x0) (x - (1 + x0)) H12 H11).
            rewrite <- plus_minus_assoc.
              rewrite minus_plus.
              intros l.
              erewrite nth_tcc_irr.
              eexact H1.
            auto with arith.
          rewrite firstn_append_left in H6; auto with arith.
          rewrite firstn_firstn_less in H6; auto with arith.
          rewrite skipn_append_left in H6; auto with arith.
          rewrite <- app_assoc in H6.
          trivial.
        auto.
      omega.
    omega.
  Qed.


  Lemma list_reorder_partition : forall(f : A -> bool)(l : list A),
    list_reorder l ((fst (partition f l)) ++ (snd (partition f l))).
  Proof.
    induction l.
      apply list_reorder_nil.
    simpl.
    destruct (partition f l).
    destruct (f a).
      simpl in *.
      apply list_reorder_cons_head.
      trivial.
    simpl in *.
    apply list_reorder_cons_parts.
    trivial.
  Qed.

  Lemma list_reorder_single_append :
    forall(a : A)(l rl rr : list A),
      list_reorder (a :: l) (rl ++ rr) ->
        (exists(rlt : list A), list_reorder (a :: rlt) rl) \/
        (exists(rrt : list A), list_reorder (a :: rrt) rr).
  Proof.
    intros a l rl rr H.
    apply (list_reorder_occurence (a :: l) _ _ (lt_0_Sn (length l))) in H.
    decompose [ex and or dep_and] H; clear H.
    rename x into n.
    simpl in *.
    apply (split_nat_case_lt n (length rl)).
      intros H.
      left.
      rewrite nth_append_left with (n_less_l1 := H) in b.
      assert (H0 := list_split_at_n _ _ H).
      rewrite <- b in *.
      rewrite H0.
      exists ((firstn n rl) ++ (skipn (S n) rl)).
      apply list_reorder_cons_parts.
      apply list_reorder_refl.
    intros H.
    right.
    rewrite nth_append_right with (n_greater := H) in b.
    assert (H0 := list_split_at_n _ _ (nth_append_right_tcc rl rr n a0 H)).
    rewrite <- b in *.
    rewrite H0.
    exists ((firstn (n - length rl) rr) ++ (skipn (S (n - length rl)) rr)).
    apply list_reorder_cons_parts.
    apply list_reorder_refl.
  Qed.


  Lemma list_reorder_In : forall(a : A)(l1 l2 : list A),
    list_reorder l1 l2 ->
    In a l1 ->
      In a l2.
  Proof.
    intros a l1 l2 H H0.
    apply in_split in H0.
    destruct H0 as [l2l].
    destruct H0 as [l2r].
    subst l1.
    assert (length l2l < length (l2l ++ a :: l2r)).
      rewrite app_length.
      simpl.
      omega.
    assert (H1 := list_reorder_occurence _ _ (length l2l) H0 H).
    decompose [ex dep_and] H1; clear H1.
    rename x into n2.
    revert b.
    rewrite nth_append_right with (n_greater := ge_refl _).
    generalize (nth_append_right_tcc l2l (a :: l2r) (length l2l) H0
                    (ge_refl (length l2l))).
    rewrite minus_diag.
    intros l b.
    simpl in *.
    clear l.
    assert (H1 := list_split_at_n _ _ a0).
    rewrite H1.
    rewrite <- b.
    apply in_or_app.
    right.
    left.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Special Properties *)
  (***************************************************************************)

  Lemma list_reorder_nil_is_nil : forall(l : list A),
    list_reorder [] l -> l = [].
  Proof.
    intros l H.
    apply list_reorder_length in H.
    destruct l.
      trivial.
    discriminate.
  Qed.

  Lemma list_reorder_singleton : forall(a : A)(l : list A),
    list_reorder [a] l -> l = [a].
  Proof.
    intros a l H.
    destruct l as [| a' ].
      apply list_reorder_length in H.
      discriminate.
    destruct l.
      apply list_reorder_occurence with (n1 := 0) (n1_less := lt_0_Sn 0) in H.
      decompose [ex and or dep_and] H; clear H.
      rename x into n, a0 into n_less, b into H.
      destruct n.
        simpl in *.
        subst a'.
        trivial.
      simpl in n_less.
      omega.
    apply list_reorder_length in H.
    discriminate.
  Qed.

  Lemma list_reorder_nonempty : forall(l1 l2 : list A),
    l1 <> [] ->
    list_reorder l1 l2 ->
      l2 <> [].
  Proof.
    intros l1 l2 H H0.
    apply list_reorder_length in H0.
    destruct l1.
      exfalso.
      auto.
    destruct l2.
      discriminate.
    discriminate.
  Qed.

  Lemma list_reorder_first_occurence :
    forall(a : A)(l1 l2 : list A),
      list_reorder (a :: l1) l2 ->
        exists(n : nat), 
          n_less # n < length l2 /#\ 
            nth l2 n n_less = a /\ list_reorder l1 (cutout_nth l2 n).
  Proof.
    intros a l1 l2 H.
    apply (list_reorder_occurence_full (a :: l1) _ 0 (lt_0_Sn (length l1)))
          in H.
    decompose [ex and or dep_and] H; clear H.
    exists x.
    apply dep_conj with (a := a0).
    auto.
  Qed.

  Lemma list_reorder_swap_head : forall(a b : A)(l : list A),
    list_reorder (a :: b :: l) (b :: a :: l).
  Proof.
    intros a b l.
    apply (list_reorder_insert [] (b :: l) [b] l).
      apply list_reorder_refl.
    trivial.
  Qed.

  Lemma list_reorder_rot_3 : forall(a b c : A)(l : list A),
     list_reorder (a :: b :: c :: l) (b :: c :: a :: l).
  Proof.
    intros a b c l.
    apply list_reorder_cons_parts with (l2l := [b; c]).
    apply list_reorder_refl.
  Qed.

  Lemma list_reorder_move_append : forall(a : A)(ll lr : list A),
    list_reorder (a :: ll ++ lr) (ll ++ a :: lr).
  Proof.
    intros a ll lr.
    apply list_reorder_cons_parts.
    apply list_reorder_refl.
  Qed.

  Lemma list_reorder_append_3_middle : forall(l1 l2 l3 : list A),
    list_reorder (l2 ++ l1 ++ l3) (l1 ++ l2 ++ l3).
  Proof.
    intros l1 l2 l3.
    apply (list_reorder_insert_list [] (l1 ++ l3) l1 l3 l2 l2).
      apply list_reorder_refl.
    apply list_reorder_refl.
  Qed.

  Lemma list_reorder_append_swap : forall(l1 l2 : list A),
    list_reorder (l1 ++ l2) (l2 ++ l1).
  Proof.
    intros l1 l2.
    lapply (list_reorder_insert_list l1 [] [] l1 l2 l2).
      intros H; lapply H; clear H.
        intros H.
        rewrite app_nil_r in H.
        trivial.
      apply list_reorder_refl.
    rewrite app_nil_r.
    apply list_reorder_refl.
  Qed.

  Lemma list_reorder_append_both : forall(l1 l2 l3 l4 : list A), 
    list_reorder l1 l3 -> 
    list_reorder l2 l4 -> 
      list_reorder (l1 ++ l2) (l3 ++ l4). 
  Proof.
    intros l1 l2 l3 l4 H H0.
    apply list_reorder_insert_list with (l1l := [])(l2l := []).
      trivial.
    trivial.
  Qed.

  Lemma list_reorder_append_right : forall(l1 l2 l3 : list A), 
    list_reorder l1 l2 -> list_reorder (l1 ++ l3) (l2 ++ l3). 
  Proof.
    intros l1 l2 l3 H.
    apply list_reorder_append_both.
      trivial.
    apply list_reorder_refl.
  Qed.

  Lemma list_reorder_append_left : forall(l1 l2 l3 : list A), 
    list_reorder l1 l2 -> list_reorder (l3 ++ l1) (l3 ++ l2). 
  Proof.
    intros l1 l2 l3 H.
    apply list_reorder_append_both.
      apply list_reorder_refl.
    trivial.
  Qed.

  Lemma list_reorder_In_split : forall(l : list A)(a : A),
    In a l ->
      exists(ll lr : list A),
        list_reorder l (a :: ll ++ lr).
  Proof.
    intros l a H.
    apply in_split in H.
    decompose [ex] H; clear H.
    exists x, x0.
    subst l.
    apply list_reorder_symm.
    apply list_reorder_cons_parts.
    apply list_reorder_refl.
  Qed.

  Lemma list_reorder_2_char : forall(a1 a2 : A)(l : list A),
    list_reorder [a1; a2] l ->
      l = [a1; a2] \/ l = [a2; a1].
  Proof.
    intros a1 a2 l H.
    assert (H0 := list_reorder_length _ _ H).
    destruct l as [| b1].
      discriminate.
    destruct l as [| b2].
      discriminate.
    destruct l.
      clear H0.
      apply list_reorder_first_occurence in H.
      decompose [ex and dep_and] H; clear H.
      rename x into n1.
      apply list_reorder_singleton in H1.
      unfold cutout_nth in *.
      destruct n1.
        simpl in *.
        inversion H1; clear H1.
        subst b1 b2.
        auto.
      destruct n1.
        simpl in *.
        inversion H1; clear H1.
        subst b1 b2.
        auto.
      simpl in *.
      omega.
    discriminate.
  Qed.

  Lemma list_reorder_double_append :
    forall(a : A)(l rl rr : list A),
      list_reorder (a :: a :: l) (rl ++ rr) ->
        (exists(rlt : list A), list_reorder (a :: a :: rlt) rl) \/
        (exists(rlt rrt : list A), 
           list_reorder (a :: rlt) rl /\ list_reorder (a :: rrt) rr) \/
        (exists(rrt : list A), list_reorder (a :: a :: rrt) rr).
  Proof.
    intros a l rl rr H.
    assert (H0 := list_reorder_single_append _ _ _ _ H).
    destruct H0.
      destruct H0 as [rlt].
      assert (list_reorder (a :: l) (rlt ++ rr)).
        eapply list_reorder_tail_head.
        eapply list_reorder_trans.
          eexact H.
        rewrite app_comm_cons.
        apply list_reorder_append_right.
        apply list_reorder_symm.
        trivial.
      apply list_reorder_single_append in H1.
      destruct H1.
        destruct H1 as [rltt].
        left.
        exists rltt.
        eapply list_reorder_trans_rev.
          eexact H0.
        apply list_reorder_cons_head.
        trivial.
      destruct H1 as [rrt].
      right.
      left.
      exists rlt, rrt.
      auto.
    destruct H0 as [rrt].
    assert (list_reorder (a :: l) (rl ++ rrt)).
      eapply list_reorder_tail.
      eapply list_reorder_trans.
        eexact H.
      apply list_reorder_append_left.
      apply list_reorder_symm.
      trivial.
    apply list_reorder_single_append in H1.
    destruct H1.
      destruct H1 as [rlt].
      right.
      left.
      exists rlt, rrt.
      auto.
    destruct H1 as [rrtt].
    right.
    right.
    exists rrtt.
    eapply list_reorder_trans_rev.
          eexact H0.
    apply list_reorder_cons_head.
    trivial.
  Qed.

End Reorder.


Implicit Arguments list_reorder [A].
Implicit Arguments list_reorder_length [A].
Implicit Arguments list_reorder_occurence_full [A].
Implicit Arguments list_reorder_occurence [A].
Implicit Arguments every_nth_list_reorder [A P].
Implicit Arguments list_reorder_insert [A].
Implicit Arguments list_reorder_insert_list [A].
Implicit Arguments list_reorder_tail [A].
Implicit Arguments list_reorder_inserted_2 [A].
Implicit Arguments list_reorder_single_append [A].
Implicit Arguments list_reorder_nonempty [A].
Implicit Arguments list_reorder_first_occurence [A].
Implicit Arguments list_reorder_move_append [A].
Implicit Arguments list_reorder_append_3_middle [A].
Implicit Arguments list_reorder_double_append [A].

(* 
 * Lemma nat_list_max_reorder : forall(l1 l2 : list nat),
 *   list_reorder l1 l2 ->
 *     nat_list_max l1 = nat_list_max l2.
 * Proof.
 *   induction 1.
 *     trivial.
 *   rewrite nat_list_max_append.
 *   simpl.
 *   rewrite Max.max_assoc.
 *   rewrite (Max.max_comm _ a).
 *   rewrite <- Max.max_assoc.
 *   rewrite <- nat_list_max_append.
 *   rewrite firstn_skipn.
 *   auto.
 *)


Lemma list_reorder_map : 
  forall{A B : Type}{f : A -> B}(l1 l2 : list A),
    list_reorder l1 l2 ->
      list_reorder (map f l1) (map f l2).
Proof.
  induction 1.
    simpl.
    apply list_reorder_nil.
  rewrite map_app.
  simpl.
  apply list_reorder_cons_parts.
  rewrite <- map_app.
  rewrite firstn_skipn.
  trivial.
Qed.

Lemma list_reorder_left_map :
  forall{A B : Type}(f : A -> B)(l1 : list A)(l2 : list B),
    list_reorder (map f l1) l2 ->
      exists(l2_pre : list A),
        l2 = map f l2_pre /\
        list_reorder l1 l2_pre.
Proof.
  intros A B f l1 l2 H.
  remember (map f l1) as l1f.
  revert l1 Heql1f.
  induction H.
    intros l1 H.
    destruct l1.
      exists [].
      split.
        trivial.
      apply list_reorder_nil.
    discriminate.
  rename l1 into l1f.
  intros l1 H1.
  destruct l1 as [| a_pre l1].
    discriminate.
  simpl in H1.
  inversion H1; clear H1.
  specialize (IHlist_reorder l1 H3).
  decompose [ex and] IHlist_reorder; clear IHlist_reorder.
  rename x into l2_pre.
  exists (firstn n l2_pre ++ a_pre :: skipn n l2_pre).
  split.
    rewrite map_app.
    simpl.
    f_equal.
      rewrite map_firstn.
      rewrite <- H1.
      trivial.
    f_equal.
    rewrite map_skipn.
    rewrite <- H1.
    trivial.
  apply list_reorder_cons_parts.
  rewrite firstn_skipn.
  trivial.
Qed.

Lemma list_reorder_right_map :
  forall{A B : Type}(f : A -> B)(l1 : list B)(l2 : list A),
    list_reorder l1 (map f l2) ->
      exists(l1_pre : list A),
        l1 = map f l1_pre /\
        list_reorder l1_pre l2.
Proof.
  intros A B f l1 l2 H.
  apply list_reorder_symm in H.
  apply list_reorder_left_map in H.
  decompose [ex and] H; clear H.
  exists x.
  split.
    trivial.
  apply list_reorder_symm.
  trivial.
Qed.

