(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: rules.v,v 1.53 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Various rules

      The type of rules and proofs is defined in modula [formulas].
      This module defines all the propositional rules as predicate or
      subset of the rule type.

*)
Require Export substitution.

Section Rules.

  Variable V : Type.
  Variable L : modal_operators.


  (**************************************************************************)
  (** ***  adding context  *)
  (**************************************************************************)

  Definition add_context(sl sr s : sequent V L) : sequent V L := sl ++ s ++ sr.

  Definition rule_add_context(sl sr : sequent V L)
                             (r : sequent_rule V L) : sequent_rule V L :=
    {| assumptions := map (add_context sl sr) r.(assumptions);
       conclusion := add_context sl sr r.(conclusion)
    |}.

  Lemma rank_rule_add_context_rev :
    forall(sl sr : sequent V L)(r : sequent_rule V L)(n : nat),
      rule_has_rank n (rule_add_context sl sr r) ->
        rule_has_rank n r.
  Proof.
    unfold rule_has_rank in *.
    intros sl sr r n H.
    destruct H.
    unfold rule_add_context, add_context in *.
    simpl in *.
    split.
      intros i i_less.
      rewrite every_nth_map in H.
      eapply every_nth_append_left.
      eapply every_nth_append_right.
      apply H.
    eapply every_nth_append_left.
    eapply every_nth_append_right.
    eexact H0.
  Qed.

  Lemma reorder_rule_add_context :
    forall(r : sequent_rule V L)(s : sequent V L)(sl0 sr0 : sequent V L),
      list_reorder (add_context sl0 sr0 (conclusion r)) s ->
      length (conclusion r) = 1 ->
      exists(sl1 sr1 : sequent V L),
        reordered_rule (rule_add_context sl0 sr0 r) s
                       (rule_add_context sl1 sr1 r).
  Proof.
    clear. 
    intros r s sl0 sr0 H H0.
    destruct r.
    simpl in *.
    destruct conclusion.
      exfalso.
      discriminate.
    destruct conclusion.
      clear H0.
      unfold add_context in H.
      simpl in *.
        assert (length sl0 < length (sl0 ++ l :: sr0)).
        rewrite app_length.
        simpl.
        omega.
      assert (H1 := list_reorder_occurence_full _ _ (length sl0) H0 H).
      unfold cutout_nth in *.
      decompose [ex and or dep_and] H1; clear H1.
      assert (nth (sl0 ++ l :: sr0) (length sl0) H0 = l).
        rewrite nth_append_right with (n_greater := ge_refl _).
        generalize (nth_append_right_tcc sl0 (l :: sr0) (length sl0) H0 
                       (ge_refl (length sl0))).
        rewrite minus_diag.
        intros l0.
        simpl.
        trivial.
      rewrite H1 in *.
      remember (skipn (1 + x) s).
      exists (firstn x s), l0.
      repeat split.
          simpl.
          unfold add_context.
          simpl.
          subst l l0.
          apply list_split_at_n.
        simpl.
        rewrite map_length.
        rewrite map_length.
        trivial.
      simpl.
      intros n n_less_rs n_less_rr.
      rewrite nth_map.
      rewrite nth_map.
      unfold add_context in *.
      simpl.
      apply list_reorder_insert_list.
        clear - H3.
        rewrite firstn_append_left in H3.
          rewrite firstn_whole in H3.
            rewrite skipn_append_right in H3.
              assert (1 + length sl0 - length sl0 = 1).
                omega.
              rewrite H in H3.
              trivial.
            omega.
          omega.
        trivial.
      erewrite nth_tcc_irr.
      apply list_reorder_refl.
    exfalso.
    simpl in *.
    omega.
  Qed.

  Lemma const_rank_add_context :
    forall(n : nat)(sl sr : sequent V L)(r rc : sequent_rule V L),
      rc = rule_add_context sl sr r ->
      (rank_sequent n (conclusion r) -> 
              every_nth (rank_sequent n) (assumptions r)) ->
      rank_sequent n (conclusion rc) ->
        every_nth (rank_sequent n) (assumptions rc).
  Proof.
    intros n sl sr r rc H H0 H1.
    rewrite H in *.
    simpl in *.
    intros i i_less.
    rewrite nth_map.
    unfold add_context in *.
    apply rank_sequent_append.
      eapply rank_sequent_append_left.
      eexact H1.
    apply rank_sequent_append.
      apply H0.
      eapply rank_sequent_append_left.
      eapply rank_sequent_append_right.
      eexact H1.
    eapply rank_sequent_append_right.
    eapply rank_sequent_append_right.
    eexact H1.
  Qed.


  (**************************************************************************)
  (** ***  Ax rule  *)
  (**************************************************************************)

  Definition simple_tautology_witness(l : sequent V L)
                                               (n1 n2 : nat)(v : V) : Prop :=
    n1_less # n1 < length l /#\
      n2_less # n2 < length l /#\
        nth l n1 n1_less = lf_prop v /\
        nth l n2 n2_less = lf_neg (lf_prop v).

  Definition simple_tautology(l : sequent V L) : Prop :=
    exists n1 : nat, exists n2 : nat, exists v : V,
      simple_tautology_witness l n1 n2 v.

  Lemma simple_tautology_cons : 
    forall(s : sequent V L)(f : lambda_formula V L),
      simple_tautology s -> simple_tautology (f :: s).
  Proof.
    unfold simple_tautology in *.
    intros s f H.
    decompose [ex] H; clear H.
    exists (1 + x).
    exists (1 + x0).
    exists x1.
    unfold simple_tautology_witness in *.
    decompose [and dep_and] H0; clear H0.
    simpl.
    constructor 1 with (a := lt_n_S _ _ a).
    constructor 1 with (a := lt_n_S _ _ a0).
    split.
      erewrite nth_tcc_irr.
      eexact H.
    erewrite nth_tcc_irr.
    eexact H1.
  Qed.

  Lemma simple_tautology_append_left : 
    forall(s1 s2 : sequent V L),
      simple_tautology s2 -> simple_tautology (s1 ++ s2).
  Proof.
    unfold simple_tautology in *.
    intros s1 s2 H.
    decompose [ex] H; clear H.
    exists (length s1 + x), (length s1 + x0), x1.
    unfold simple_tautology_witness in *.
    decompose [and dep_and] H0; clear H0.
    assert (length s1 + x < length (s1 ++ s2)).
      rewrite app_length.
      omega.
    constructor 1 with (a := H0).
    assert (length s1 + x0 < length (s1 ++ s2)).
      rewrite app_length.
      omega.
    constructor 1 with (a := H2).
    split.
      assert (length s1 + x >= length s1).
        omega.
      rewrite nth_append_right with (n_greater := H3).
      generalize (nth_append_right_tcc s1 s2 (length s1 + x) H0 H3).
      rewrite minus_plus.
      intros l.
      erewrite nth_tcc_irr.
      eexact H.
    assert (length s1 + x0 >= length s1).
      omega.
    rewrite nth_append_right with (n_greater := H3).
    generalize (nth_append_right_tcc s1 s2 (length s1 + x0) H2 H3).
    rewrite minus_plus.
    intros l.
    erewrite nth_tcc_irr.
    eexact H1.
  Qed.

  Lemma simple_tautology_tail :
    forall(s : sequent V L)(f : lambda_formula V L),
      simple_tautology (f :: s) -> 
      (not (neg_form_maybe prop_form f)) ->
        simple_tautology s.
  Proof.
    clear. 
    intros s f H H0.
    unfold simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H; clear H.
    destruct x.
      exfalso.
      simpl in H1.
      subst f.
      apply H0.
      simpl.
      trivial.
    destruct x0.
      exfalso.
      simpl in H2.
      subst f.
      apply H0.
      simpl.
      trivial.
    simpl in *.
    exists x, x0, x1.
    constructor 1 with (a := lt_S_n _ _ a).
    constructor 1 with (a := lt_S_n _ _ a0).
    split.
      erewrite nth_tcc_irr.
      eexact H1.
    erewrite nth_tcc_irr.
    eexact H2.
  Qed.


  Lemma simple_tautology_reorder : forall(s1 s2 : sequent V L),
    simple_tautology s1 -> list_reorder s1 s2 -> simple_tautology s2.
  Proof.
    intros s1 s2 H H0.
    unfold simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H; clear H.
    assert (H3 := list_reorder_occurence _ _ x a H0).
    decompose [ex and or dep_and] H3; clear H3.
    assert (H3 := list_reorder_occurence _ _ x0 a0 H0).
    decompose [ex and or dep_and] H3; clear H3.
    exists x2, x3, x1.
    constructor 1 with (a := a1).
    constructor 1 with (a := a2).
    rewrite <- b.
    rewrite <- b0.
    auto.
  Qed.

  Lemma simple_tautology_append_right : 
    forall(s1 s2 : sequent V L),
      simple_tautology s1 -> simple_tautology (s1 ++ s2).
  Proof.
    intros s1 s2 H.
    eapply simple_tautology_reorder.
      apply simple_tautology_append_left.
      eexact H.
    apply list_reorder_append_swap.
  Qed.

  Lemma simple_tautology_cons_destruct :
    forall(f : lambda_formula V L)(s : sequent V L),
      simple_tautology (f :: s) ->
        simple_tautology s \/
        (exists(sr : sequent V L), 
           list_reorder s ((lf_neg f) :: sr) /\ prop_form f) \/
        (exists(g : lambda_formula V L)(sr : sequent V L),
           f = lf_neg g /\
           list_reorder s (g :: sr) /\
           prop_form g).
  Proof.
    intros f s H.
    unfold simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H; clear H.
    rename x into i1, x0 into i2, x1 into v, a into i1_less, a0 into i2_less.
    destruct i1.
      right.
      left.
      simpl in H0.
      subst f.
      destruct i2.
        exfalso.
        simpl in *.
        discriminate.
      simpl in H1.
      assert (H2 := list_split_at_n _ _ 
                         (nth_succ_tcc i2 (lf_prop v) s i2_less)).
      rewrite H1 in H2.
      rewrite H2.
      clear. 
      exists (firstn i2 s ++ skipn (S i2) s).
      split.
        apply list_reorder_symm.
        apply list_reorder_move_append.
      simpl.
      trivial.
    destruct i2.
      right.
      right.
      simpl in *.
      assert (H2 := list_split_at_n _ _ (nth_succ_tcc i1 f s i1_less)).
      rewrite H0 in H2.
      rewrite H2.
      clear - H1.
      exists (lf_prop v), (firstn i1 s ++ skipn (S i1) s).
      repeat split.
        trivial.
      apply list_reorder_symm.
      apply list_reorder_move_append.
    left.
    simpl in *.
    exists i1, i2, v.
    apply dep_conj with (a := nth_succ_tcc i1 f s i1_less).
    apply dep_conj with (a := nth_succ_tcc i2 f s i2_less).
    auto.
  Qed.

  Lemma simple_tautology_contract_head :
    forall(f : lambda_formula V L)(s : sequent V L),
      simple_tautology (f :: f :: s) -> simple_tautology (f :: s).
  Proof.
    intros f s H.
    unfold simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H; clear H.
    rename x into n1, x0 into n2, x1 into v.
    destruct n1.
      destruct n2.
        exfalso.
        simpl in *.
        rewrite H0 in *.
        discriminate.
      remember (f :: s).
      simpl in * |-.
      subst l.
      exists 0, n2, v.
      apply dep_conj with (a := lt_0_Sn (length s)).
      apply dep_conj with (a := lt_S_n _ _ a0).
      split.
        simpl.
        trivial.
      erewrite nth_tcc_irr.
      eexact H1.
    destruct n2.
      remember (f :: s).
      simpl in * |-.
      subst l.
      exists n1, 0, v.
      apply dep_conj with (a := lt_S_n _ _ a).
      apply dep_conj with (a := lt_0_Sn (length s)).
      split.
        erewrite nth_tcc_irr.
        eexact H0.
      simpl.
      trivial.
    remember (f :: s).
    simpl in * |-.
    exists n1, n2, v.
    apply dep_conj with (a := lt_S_n _ _ a).
    apply dep_conj with (a := lt_S_n _ _ a0).
    split.
      erewrite nth_tcc_irr.
      eexact H0.
    erewrite nth_tcc_irr.
    eexact H1.
  Qed.

  Definition is_ax_rule(r : sequent_rule V L) : Prop :=
    assumptions r = [] /\ simple_tautology (conclusion r).

  Lemma ax_rule_no_empty_conclusion : forall(r : sequent_rule V L),
    is_ax_rule r -> conclusion r <> [].
  Proof.
    intros r H H0.
    unfold is_ax_rule in *.
    destruct H.
    rewrite H0 in *.
    clear H H0.
    unfold simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H1; clear H1.
    simpl in *.
    omega.
  Qed.


  (**************************************************************************)
  (** ***  and rule  *)
  (**************************************************************************)

  Definition is_and_rule(r : sequent_rule V L) : Prop :=
    exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      assumptions r = [sl ++ f1 :: sr; sl ++ f2 :: sr] /\
      conclusion r = sl ++ (lf_and f1 f2) :: sr.        

  Definition bare_and_rule(f1 f2 : lambda_formula V L) : sequent_rule V L :=
    {| assumptions := [[f1]; [f2]];
       conclusion := [lf_and f1 f2]
    |}.

  Lemma and_rule_context :
    forall(r : sequent_rule V L),
      is_and_rule r ->
        exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
          r = rule_add_context sl sr (bare_and_rule f1 f2).
  Proof.
    intros r H.
    unfold is_and_rule, rule_add_context, add_context in *.
    decompose [ex and or dep_and] H; clear H.
    exists x. 
    exists x0. 
    exists x1. 
    exists x2.
    destruct r.
    simpl in *.
    rewrite H0.
    rewrite H2.
    trivial.
  Qed.

  Lemma context_and_rule : 
    forall(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      is_and_rule (rule_add_context sl sr (bare_and_rule f1 f2)).
  Proof.
    clear. 
    intros sl sr f1 f2.
    unfold is_and_rule, rule_add_context in *.
    exists sl, sr, f1, f2.
    auto.
  Qed.

  Lemma const_rank_and_rule :
    forall(n : nat)(f1 f2 : lambda_formula V L),
      rank_sequent n [lf_and f1 f2] ->
        every_nth (rank_sequent n) [[f1]; [f2]].
  Proof.
    clear. 
    intros n f1 f2 H.
    apply every_nth_cons.
      unfold rank_sequent in *.
      apply every_nth_cons.
        assert (H0 := every_nth_head _ _ _ H).
        clear H.
        unfold rank_formula in *.
        rewrite modal_rank_char in H0.
        eapply Max.max_lub_l.
        eexact H0.
      apply every_nth_empty.
    apply every_nth_cons.
      unfold rank_sequent in *.
      apply every_nth_cons.
        assert (H0 := every_nth_head _ _ _ H).
        clear H.
        unfold rank_formula in *.
        rewrite modal_rank_char in H0.
        eapply Max.max_lub_r.
        eexact H0.
      apply every_nth_empty.
    apply every_nth_empty.
  Qed.
    
  Lemma const_rank_and_rule_left_context :
    forall(n : nat)(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      rank_sequent n (sl ++ (lf_and f1 f2) :: sr) ->
        rank_sequent n (sl ++ f1 :: sr).
  Proof.
    intros n sl sr f1 f2 H.
    lapply (const_rank_add_context n sl sr (bare_and_rule f1 f2) _ eq_refl).
      simpl in *.
      intros H0; lapply H0; clear H0.
        intros H0.
        apply every_nth_head in H0.
        trivial.
      trivial.
    apply const_rank_and_rule.
  Qed.

  Lemma const_rank_and_rule_right_context :
    forall(n : nat)(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      rank_sequent n (sl ++ (lf_and f1 f2) :: sr) ->
        rank_sequent n (sl ++ f2 :: sr).
  Proof.
    intros n sl sr f1 f2 H.
    lapply (const_rank_add_context n sl sr (bare_and_rule f1 f2) _ eq_refl).
      simpl in *.
      intros H0; lapply H0; clear H0.
        intros H0.
        apply every_nth_tail in H0.
        apply every_nth_head in H0.
        trivial.
      trivial.
    apply const_rank_and_rule.
  Qed.

  Lemma subst_closed_and : subst_closed_rule_set is_and_rule.
  Proof.
    unfold subst_closed_rule_set in *.
    intros sigma r H.
    unfold is_and_rule, subst_sequent_rule in *.
    simpl in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H0.
    rewrite H2.
    clear H0 H2.
    simpl.
    eexists _, _, _, _.
    unfold subst_sequent.
    split.
      f_equal.
        rewrite map_app.
        f_equal.
        simpl.
        trivial.
      f_equal.
      rewrite map_app.
      simpl.
      f_equal.
    rewrite map_app.
    f_equal.
  Qed.

  Lemma and_rule_no_empty_conclusion : forall(r : sequent_rule V L),
    is_and_rule r -> conclusion r <> [].
  Proof.
    intros r H H0.
    assert (length (conclusion r) = 0).
      rewrite H0.
      trivial.
    clear H0.
    unfold is_and_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H3 in H1.
    rewrite app_length in H1.
    simpl in *.
    omega.
  Qed.


  (**************************************************************************)
  (** ***  negated and rule  *)
  (**************************************************************************)

  Definition is_neg_and_rule(r : sequent_rule V L) : Prop :=
    exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      assumptions r = [sl ++ (lf_neg f1) :: (lf_neg f2) :: sr] /\
      conclusion r = sl ++ (lf_neg (lf_and f1 f2)) :: sr.

  Definition bare_neg_and_rule(f1 f2 : lambda_formula V L) : sequent_rule V L :=
    {| assumptions := [[lf_neg f1; lf_neg f2]];
       conclusion := [lf_neg (lf_and f1 f2)]
    |}.

  Lemma neg_and_rule_context :
    forall(r : sequent_rule V L),
      is_neg_and_rule r ->
        exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
          r = rule_add_context sl sr (bare_neg_and_rule f1 f2).
  Proof.
    intros r H.
    unfold is_neg_and_rule, rule_add_context, add_context in *.
    decompose [ex and or dep_and] H; clear H.
    exists x. 
    exists x0. 
    exists x1. 
    exists x2.
    destruct r.
    simpl in *.
    rewrite H0.
    rewrite H2.
    trivial.
  Qed.

  Lemma context_neg_and_rule : 
    forall(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      is_neg_and_rule (rule_add_context sl sr (bare_neg_and_rule f1 f2)).
  Proof.
    clear. 
    intros sl sr f1 f2.
    unfold is_neg_and_rule, rule_add_context in *.
    exists sl, sr, f1, f2.
    auto.
  Qed.

  Lemma const_rank_neg_and_rule :
    forall(n : nat)(f1 f2 : lambda_formula V L),
      rank_sequent n [lf_neg (lf_and f1 f2)] ->
        every_nth (rank_sequent n) [[lf_neg f1; lf_neg f2]].
  Proof.
    clear. 
    intros n f1 f2 H.
    apply every_nth_cons.
      unfold rank_sequent in *.
      assert (H0 := every_nth_head _ _ _ H).
      clear H.
      unfold rank_formula in *.
      rewrite modal_rank_char in H0.
      rewrite modal_rank_char in H0.
      apply every_nth_cons.
        rewrite modal_rank_char.
        eapply Max.max_lub_l.
        eexact H0.
      apply every_nth_cons.
        rewrite modal_rank_char.
        eapply Max.max_lub_r.
        eexact H0.
      apply every_nth_empty.
    apply every_nth_empty.
  Qed.

  Lemma const_rank_neg_and_rule_context :
    forall(n : nat)(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      rank_sequent n (sl ++ (lf_neg (lf_and f1 f2)) :: sr) ->
        rank_sequent n (sl ++ (lf_neg f1) :: (lf_neg f2) :: sr).
  Proof.
    intros n sl sr f1 f2 H.
    lapply (const_rank_add_context n sl sr (bare_neg_and_rule f1 f2) _ eq_refl).
      simpl in *.
      intros H0; lapply H0; clear H0.
        intros H0.
        apply every_nth_head in H0.
        trivial.
      trivial.
    apply const_rank_neg_and_rule.
  Qed.

  Lemma subst_closed_neg_and : subst_closed_rule_set is_neg_and_rule.
  Proof.
    unfold subst_closed_rule_set in *.
    intros sigma r H.
    unfold is_neg_and_rule, subst_sequent_rule in *.
    simpl in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H0.
    rewrite H2.
    clear H0 H2.
    simpl.
    eexists _, _, _, _.
    unfold subst_sequent.
    split.
      f_equal.
      rewrite map_app.
      f_equal.
      simpl.
      rewrite subst_form_char.
      rewrite subst_form_char with (f := lf_neg x2).
      trivial.
    rewrite map_app.
    f_equal.
  Qed.

  Lemma neg_and_rule_no_empty_conclusion : forall(r : sequent_rule V L),
    is_neg_and_rule r -> conclusion r <> [].
  Proof.
    intros r H H0.
    assert (length (conclusion r) = 0).
      rewrite H0.
      trivial.
    clear H0.
    unfold is_neg_and_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H3 in H1.
    rewrite app_length in H1.
    simpl in *.
    omega.
  Qed.


  (**************************************************************************)
  (** ***  double negation rule  *)
  (**************************************************************************)

  Definition is_neg_neg_rule(r : sequent_rule V L) : Prop :=
    exists(sl sr : sequent V L)(f : lambda_formula V L),
      assumptions r = [sl ++ f :: sr] /\
      conclusion r = sl ++ (lf_neg (lf_neg f)) :: sr.

  Definition bare_neg_neg_rule(f : lambda_formula V L) : sequent_rule V L :=
    {| assumptions := [[f]];
       conclusion := [lf_neg (lf_neg f)]
    |}.

  Lemma neg_neg_rule_context :
    forall(r : sequent_rule V L),
      is_neg_neg_rule r ->
        exists(sl sr : sequent V L)(f : lambda_formula V L),
          r = rule_add_context sl sr (bare_neg_neg_rule f).
  Proof.
    intros r H.
    unfold is_neg_neg_rule, rule_add_context, add_context in *.
    decompose [ex and or dep_and] H; clear H.
    exists x. 
    exists x0. 
    exists x1. 
    destruct r.
    simpl in *.
    rewrite H1.
    rewrite H2.
    trivial.
  Qed.

  Lemma context_neg_neg_rule : 
    forall(sl sr : sequent V L)(f : lambda_formula V L),
      is_neg_neg_rule (rule_add_context sl sr (bare_neg_neg_rule f)).
  Proof.
    clear. 
    intros sl sr f.
    unfold is_neg_neg_rule, rule_add_context in *.
    exists sl, sr, f.
    auto.
  Qed.

  Lemma const_rank_neg_neg_rule :
    forall(n : nat)(f : lambda_formula V L),
      rank_sequent n [lf_neg (lf_neg f)] ->
        every_nth (rank_sequent n) [[f]].
  Proof.
    clear. 
    intros n f H.
    apply every_nth_cons.
      unfold rank_sequent in *.
      apply every_nth_cons.
        assert (H0 := every_nth_head _ _ _ H).
        clear H.
        unfold rank_formula in *.
        rewrite modal_rank_char in H0.
        rewrite modal_rank_char in H0.
        trivial.
      apply every_nth_empty.
    apply every_nth_empty.
  Qed.
    
  Lemma const_rank_neg_neg_rule_context :
    forall(n : nat)(sl sr : sequent V L)(f : lambda_formula V L),
      rank_sequent n (sl ++ (lf_neg (lf_neg f)) :: sr) ->
        rank_sequent n (sl ++ f :: sr).
  Proof.
    intros n sl sr f H.
    lapply (const_rank_add_context n sl sr (bare_neg_neg_rule f) _ eq_refl).
      simpl in *.
      intros H0; lapply H0; clear H0.
        intros H0.
        apply every_nth_head in H0.
        trivial.
      trivial.
    apply const_rank_neg_neg_rule.
  Qed.

  Lemma subst_closed_neg_neg : subst_closed_rule_set is_neg_neg_rule.
  Proof.
    unfold subst_closed_rule_set in *.
    intros sigma r H.
    unfold is_neg_neg_rule, subst_sequent_rule in *.
    simpl in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H1.
    rewrite H2.
    clear H1 H2.
    simpl.
    eexists _, _, _.
    unfold subst_sequent.
    split.
      f_equal.
      rewrite map_app.
      simpl.
      trivial.
    rewrite map_app.
    f_equal.
  Qed.

  Lemma neg_neg_rule_no_empty_conclusion : forall(r : sequent_rule V L),
    is_neg_neg_rule r -> conclusion r <> [].
  Proof.
    intros r H H0.
    assert (length (conclusion r) = 0).
      rewrite H0.
      trivial.
    clear H0.
    unfold is_neg_neg_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H3 in H1.
    rewrite app_length in H1.
    simpl in *.
    omega.
  Qed.


  (***************************************************************************)
  (** ***  non-atomic axioms, page 15  *)
  (***************************************************************************)

  Definition tautology_witness(s : sequent V L)(n1 n2 : nat) : Prop :=
    n1_less # n1 < length s /#\
      n2_less # n2 < length s /#\
        nth s n1 n1_less = lf_neg (nth s n2 n2_less).

  Definition subst_Ax_set(sigma : lambda_subst V L)(s : sequent V L) : Prop :=
    exists(v : V), s = [sigma v; lf_neg (sigma v)].

  Lemma rank_sequent_subst_Ax_set : 
    forall(k : nat)(sigma : lambda_subst V L)(s : sequent V L),
      rank_subst k sigma ->
      subst_Ax_set sigma s -> 
        rank_sequent k s.
  Proof.
    intros k sigma s H H0.
    unfold subst_Ax_set in *.
    decompose [ex] H0; clear H0.
    subst s.
    apply rank_sequent_cons.
      apply H.
    apply rank_sequent_cons.
      apply rank_formula_lf_neg.
      apply H.
    apply rank_sequent_empty.
  Qed.


  Definition subst_Ax_n_set(sigma : lambda_subst V L)(n : nat)
                           (s : sequent V L) : Prop :=
    exists(ax delta : sequent V L),
      subst_Ax_set sigma ax /\
      list_reorder (ax ++ delta) s /\
      rank_sequent n delta.

  Lemma ax_rule_subst : 
    forall(r : sequent_rule V L)(sigma : lambda_subst V L)(n k : nat),
      rank_subst (S k) sigma ->
      rank_rules n is_ax_rule r ->
        subst_Ax_n_set sigma (n + k) (subst_sequent sigma (conclusion r)).
  Proof.
    intros r sigma n k H H0.
    unfold rank_rules, rule_has_rank in *.
    decompose [and] H0; clear H0.
    clear H3.
    unfold is_ax_rule in *.
    destruct H1.
    clear H0.
    unfold simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H1; clear H1.
    lapply (list_split_at_n2 (conclusion r) x x0 a a0).
      intros H1.
      decompose [ex and dep_and] H1; clear H1.
      unfold subst_Ax_n_set in *.
      eexists.
      exists (subst_sequent sigma (x2 ++ x3 ++ x4)).
      repeat split.
          exists x1.
          reflexivity.
        simpl.
        change (sigma x1) with (subst_form sigma (lf_prop x1)).
        change (lf_neg (subst_form sigma (lf_prop x1))) 
          with (subst_form sigma (lf_neg (lf_prop x1))).
        change (_ :: _ :: _) 
          with (subst_sequent sigma 
                  ((lf_prop x1) :: (lf_neg (lf_prop x1)) :: x2 ++ x3 ++ x4)).
        apply list_reorder_subst_sequent.
        rewrite H5.
        decompose [and or] H6; clear H6.
          subst x5 x6.
          rewrite H0.
          rewrite H2.
          apply list_reorder_cons_parts.
          rewrite app_assoc.
          rewrite app_assoc.
          apply list_reorder_cons_parts.
          apply list_reorder_refl.
        subst x5 x6.
        rewrite H0.
        rewrite H2.
        rewrite (app_comm_cons x3).
        rewrite (app_assoc _ (_ :: x3)).
        apply list_reorder_cons_parts.
        rewrite <- app_assoc.
        rewrite <- app_comm_cons.
        apply list_reorder_cons_parts.
        apply list_reorder_refl.
      eapply rank_sequent_subst_add.
            rewrite H5 in H4.
            apply rank_sequent_append.
              eapply rank_sequent_append_left.
              eexact H4.
            apply rank_sequent_append.
              eapply rank_sequent_append_left.
              eapply rank_sequent_tail.
              eapply rank_sequent_append_right.
              eexact H4.
            eapply rank_sequent_tail.
            eapply rank_sequent_append_right.
            eapply rank_sequent_tail.
            eapply rank_sequent_append_right.
            eexact H4.
          eexact H.
        apply le_n_S.
        apply le_0_n.
      omega.
    intros H3.
    subst x0.
    rewrite nth_tcc_irr with (inside_2 := a) in H2.
    rewrite H0 in H2.
    discriminate.
  Qed.


  (***************************************************************************)
  (** ***  cut, page 10  *)
  (***************************************************************************)

  Definition is_cut_rule(r : sequent_rule V L) : Prop :=
    exists(gamma_l gamma_r delta_l delta_r : sequent V L)
          (f : lambda_formula V L),
      assumptions r = [gamma_l ++ f :: gamma_r; 
                       delta_l ++ (lf_neg f) :: delta_r] /\
      list_reorder (conclusion r) (gamma_l ++ gamma_r ++ delta_l ++ delta_r).

  Lemma cut_rule_multiset : rule_multiset is_cut_rule.
  Proof.
    clear. 
    unfold rule_multiset in *.
    intros or s H H0.
    exists {| assumptions := assumptions or; conclusion := s |}.
    repeat split.
      simpl.
      intros n n_less_rs n_less_rr.
      erewrite nth_tcc_irr.
      apply list_reorder_refl.
    unfold is_cut_rule in *.
    decompose [ex and or dep_and] H; clear H.
    exists x, x0, x1, x2, x3.
    simpl.
    split.
      trivial.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      eexact H0.
    trivial.
  Qed.

  Lemma const_rank_cut_rule_left :
    forall(n : nat)(gl gr dl dr : sequent V L)(f : lambda_formula V L),
      rank_formula n f ->
      rank_sequent n (gl ++ gr ++ dl ++ dr) ->
        rank_sequent n (gl ++ f :: gr).
  Proof.
    intros n gl gr dl dr f H H0.
    apply rank_sequent_append.
      apply rank_sequent_append_left in H0.
      trivial.
    apply rank_sequent_cons.
      trivial.
    apply rank_sequent_append_right in H0.
    apply rank_sequent_append_left in H0.
    trivial.
  Qed.

  Lemma const_rank_cut_rule_right :
    forall(n : nat)(gl gr dl dr : sequent V L)(f : lambda_formula V L),
      rank_formula n f ->
      rank_sequent n (gl ++ gr ++ dl ++ dr) ->
        rank_sequent n (dl ++ (lf_neg f) :: dr).
  Proof.
    intros n gl gr dl dr f H H0.
    apply rank_sequent_append_right in H0.
    apply rank_sequent_append_right in H0.
    apply rank_sequent_append.
      apply rank_sequent_append_left in H0.
      trivial.
    apply rank_sequent_cons.
      trivial.
    apply rank_sequent_append_right in H0.
    trivial.
  Qed.

  Lemma subst_cut_rule : subst_closed_rule_set is_cut_rule.
  Proof.
    unfold subst_closed_rule_set in *.
    intros sigma r H.
    unfold is_cut_rule in *.
    decompose [ex and or dep_and] H; clear H.
    eexists _, _, _, _, _.
    simpl.
    rewrite H1.
    split.
      simpl.
      f_equal.
        rewrite subst_sequent_append.
        simpl.
        f_equal.
      f_equal.
      rewrite subst_sequent_append.
      simpl.
      f_equal.
    repeat rewrite <- subst_sequent_append.
    apply list_reorder_map.
    trivial.
  Qed.


  Definition bounded_cut_rules(n : nat) : set (sequent_rule V L) :=
    rank_rules n is_cut_rule.


  (***************************************************************************)
  (** ***  bounded weakening, Lemma 3.11, page 14  *)
  (***************************************************************************)

  Definition bounded_weakening_rules(n : nat)(r : sequent_rule V L) : Prop :=
    exists(s : sequent V L)(f : lambda_formula V L),
      rank_formula n f /\
      rank_sequent n s /\
      assumptions r = [s] /\
      list_reorder (conclusion r) (f :: s).

  Lemma rank_conclusion_bounded_weakening_rule :
    forall(n : nat)(r : sequent_rule V L),
      bounded_weakening_rules n r ->
        rank_sequent n (conclusion r).
  Proof.
    intros n r H.
    unfold bounded_weakening_rules in *.
    decompose [ex and or dep_and] H; clear H.
    eapply rank_sequent_list_reorder.
      eexact H4.
    apply rank_sequent_cons.
      trivial.
    trivial.
  Qed.

  Lemma rank_assumptions_bounded_weakening_rule :
    forall(n : nat)(r : sequent_rule V L),
      bounded_weakening_rules n r ->
        every_nth (rank_sequent n) (assumptions r).
  Proof.
    clear. 
    intros n r H.
    unfold bounded_weakening_rules in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H2.
    apply every_nth_cons.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma bounded_weakening_rules_multiset : forall(n : nat),
    rule_multiset (bounded_weakening_rules n).
  Proof.
    unfold rule_multiset in *.
    intros n or s H H0.
    unfold bounded_weakening_rules in *.
    decompose [ex and or dep_and] H; clear H.
    exists {| assumptions := [x]; conclusion := s |}.
    repeat split.
        rewrite H3.
        trivial.
      rewrite H3.
      simpl assumptions.
      intros i i_less_rs i_less_rr.
      assert (i = 0).
        simpl in *.
        omega.
      subst i.
      simpl.
      apply list_reorder_refl.
    simpl.
    exists x, x0.
    repeat split; auto.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      eexact H0.
    trivial.
  Qed.

  Definition bounded_weakening_closed(n : nat)(ss : set (sequent V L)) 
                                                                     : Prop :=
    forall(s r : sequent V L)(f : lambda_formula V L),
      ss s -> rank_formula n f -> list_reorder r (f :: s) -> ss r.

  Lemma bounded_weakening_closed_empty : forall(n : nat),
    bounded_weakening_closed n (empty_sequent_set V L).
  Proof.
    unfold bounded_weakening_closed in *.
    intros n s r f H H0 H1.
    contradiction.
  Qed.


  (***************************************************************************)
  (** ***  full weakening, page 15?  *)
  (***************************************************************************)

  Definition full_weakening_rules(r : sequent_rule V L) : Prop :=
    exists(s : sequent V L)(f : lambda_formula V L),
      assumptions r = [s] /\
      list_reorder (conclusion r) (f :: s).

  Lemma full_weakening_rules_multiset : rule_multiset full_weakening_rules.
  Proof.
    unfold rule_multiset in *.
    intros or s H H0.
    unfold full_weakening_rules in *.
    decompose [ex and or dep_and] H; clear H.
    exists {| assumptions := [x]; conclusion := s |}.
    repeat split.
        rewrite H1.
        trivial.
      rewrite H1.
      simpl assumptions.
      intros n n_less_rs n_less_rr.
      assert (n = 0).
        simpl in *.
        omega.
      subst n.
      simpl.
      apply list_reorder_refl.
    simpl.
    exists x, x0.
    split.
      trivial.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      eexact H0.
    trivial.
  Qed.

  Definition full_weakening_closed(ss : set (sequent V L)) : Prop :=
    forall(s r : sequent V L)(f : lambda_formula V L),
      ss s -> list_reorder r (f :: s) -> ss r.

  Lemma bounded_full_weakening :
    forall(r : sequent_rule V L),
      full_weakening_rules r ->
        bounded_weakening_rules (minimal_rule_rank r) r.
  Proof.
    intros r H.
    unfold full_weakening_rules, bounded_weakening_rules in *.
    decompose [ex and or dep_and] H; clear H.
    exists x, x0.
    repeat split.
          eapply rank_sequent_head.
          rewrite <- rank_sequent_list_reorder.
            apply minimal_rule_rank_conclusion.
          eexact H2.
        eapply rank_sequent_tail.
        rewrite <- rank_sequent_list_reorder.
          apply minimal_rule_rank_conclusion.
        eexact H2.
      trivial.
    trivial.
  Qed.

  Lemma full_weakening_rules_nonempty_conclusion :
    forall(r : sequent_rule V L),
      full_weakening_rules r ->
        conclusion r <> [].
  Proof.
    intros r H.
    unfold full_weakening_rules in *.
    decompose [ex and or dep_and] H; clear H.
    destruct (conclusion r).
      exfalso.
      assert (H3 := list_reorder_length _ _ H2).
      simpl in *.
      discriminate.
    discriminate.
  Qed.

  (***************************************************************************)
  (** ***  inverted rules, page 15  *)
  (***************************************************************************)

  (***************************************************************************)
  (** ****  left inverted and rule  *)
  (***************************************************************************)

  Definition inverted_and_left_rule(r : sequent_rule V L) : Prop :=
    exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      assumptions r = [sl ++ (lf_and f1 f2) :: sr] /\
      conclusion r = sl ++ f1 :: sr.

  Definition bare_inverted_and_left_rule(f1 f2 : lambda_formula V L) 
                                                         : sequent_rule V L :=
    {| assumptions := [[lf_and f1 f2]];
       conclusion := [f1]
    |}.

  Lemma inverted_and_left_rule_context :
    forall(r : sequent_rule V L),
      inverted_and_left_rule r ->
        exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
          r = rule_add_context sl sr (bare_inverted_and_left_rule f1 f2).
  Proof.
    clear. 
    intros r H.
    unfold inverted_and_left_rule, rule_add_context, add_context in *.
    decompose [ex and or dep_and] H; clear H.
    exists x, x0, x1, x2.
    destruct r.
    simpl in *.
    rewrite H0.
    rewrite H2.
    trivial.
  Qed.

  Lemma context_inverted_and_left_rule :
    forall(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      inverted_and_left_rule 
        (rule_add_context sl sr (bare_inverted_and_left_rule f1 f2)).
  Proof.
    clear. 
    intros sl sr f1 f2.
    unfold inverted_and_left_rule, rule_add_context in *.
    exists sl, sr, f1, f2.
    auto.
  Qed.


  (***************************************************************************)
  (** ****  right inverted and rule  *)
  (***************************************************************************)

  Definition inverted_and_right_rule(r : sequent_rule V L) : Prop :=
    exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      assumptions r = [sl ++ (lf_and f1 f2) :: sr] /\
      conclusion r = sl ++ f2 :: sr.

  Definition bare_inverted_and_right_rule(f1 f2 : lambda_formula V L) 
                                                         : sequent_rule V L :=
    {| assumptions := [[lf_and f1 f2]];
       conclusion := [f2]
    |}.

  Lemma inverted_and_right_rule_context :
    forall(r : sequent_rule V L),
      inverted_and_right_rule r ->
        exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
          r = rule_add_context sl sr (bare_inverted_and_right_rule f1 f2).
  Proof.
    clear. 
    intros r H.
    unfold inverted_and_right_rule, rule_add_context, add_context in *.
    decompose [ex and or dep_and] H; clear H.
    exists x, x0, x1, x2.
    destruct r.
    simpl in *.
    rewrite H0.
    rewrite H2.
    trivial.
  Qed.

  Lemma context_inverted_and_right_rule :
    forall(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
      inverted_and_right_rule 
        (rule_add_context sl sr (bare_inverted_and_right_rule f1 f2)).
  Proof.
    clear. 
    intros sl sr f1 f2.
    unfold inverted_and_right_rule, rule_add_context in *.
    exists sl, sr, f1, f2.
    auto.
  Qed.

  (***************************************************************************)
  (** ****  inverted or rule  *)
  (***************************************************************************)

  Definition inverted_or_rule(r : sequent_rule V L) : Prop :=
    exists(sl sm sr : sequent V L)(f1 f2 : lambda_formula V L),
      assumptions r = [sl ++ (lf_neg (lf_and f1 f2)) :: sm ++ sr] /\
      (conclusion r = sl ++ (lf_neg f1) :: sm ++ (lf_neg f2) :: sr \/
       conclusion r = sl ++ (lf_neg f2) :: sm ++ (lf_neg f1) :: sr).

  Lemma inverted_or_left_reordered_rule :
    forall(or : sequent_rule V L)(rc sl sm sr : sequent V L)
          (f1 f2 : lambda_formula V L),
      list_reorder (conclusion or) rc ->
      assumptions or = [sl ++ (lf_neg (lf_and f1 f2)) :: sm ++ sr] ->
      conclusion or = sl ++ (lf_neg f1) :: sm ++ (lf_neg f2) :: sr ->
        exists(rr : sequent_rule V L),
          reordered_rule or rc rr /\ inverted_or_rule rr.
  Proof.
    intros or rc sl sm sr f1 f2 H H0 H1.
    rewrite H1 in *.
    assert (H2 := list_reorder_inserted_2 _ _ _ _ _ _ H).
    decompose [ex and] H2; clear H2.
    exists {| assumptions := [x ++ (lf_neg (lf_and f1 f2)) :: x0 ++ x1];
              conclusion := rc
           |}.
    repeat split; trivial.
        rewrite H0.
        trivial.
      simpl assumptions.
      rewrite H0.
      simpl length.
      intros n n_less_rs n_less_rr.
      assert (n = 0).
        omega.
      subst n.
      simpl.
      apply list_reorder_insert.
        trivial.
      trivial.
    exists x, x0, x1, f1, f2.
    simpl.
    split.
      trivial.
    decompose [and or] H6; clear H6.
      left.
      subst rc x2 x3.
      trivial.
    right.
    subst rc x2 x3.
    trivial.
  Qed.

  Definition inverted_or_rule_multiset : rule_multiset inverted_or_rule.
  Proof.
    clear. 
    unfold rule_multiset in *.
    intros or s H H0.
    unfold inverted_or_rule in H.
    decompose [ex and or] H; clear H.
      eapply inverted_or_left_reordered_rule; eauto.
    set (oor := {| assumptions := assumptions or;
                   conclusion := x ++ lf_neg x2 :: x0 ++ lf_neg x3 :: x1 |}).
    eapply inverted_or_left_reordered_rule 
             with (or := oor); eauto.
    apply list_reorder_symm.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      eexact H0.
    rewrite H1.
    simpl.
    clear. 
    rewrite app_comm_cons.
    rewrite app_assoc.
    apply list_reorder_insert.
      rewrite <- app_assoc.
      rewrite <- app_comm_cons.
      rewrite app_assoc.
      apply list_reorder_insert.
        rewrite <- app_assoc.
        apply list_reorder_refl.
      trivial.
    trivial.
  Qed.

  (***************************************************************************)
  (** ****  inverted neg rule  *)
  (***************************************************************************)

  Definition inverted_neg_rule(r : sequent_rule V L) : Prop :=
    exists(sl sr : sequent V L)(f : lambda_formula V L),
      assumptions r = [sl ++ (lf_neg (lf_neg f)) :: sr] /\
      conclusion r = sl ++ f :: sr.

  Definition bare_inverted_neg_rule(f : lambda_formula V L) 
                                                         : sequent_rule V L :=
    {| assumptions := [[lf_neg (lf_neg f)]];
       conclusion := [f]
    |}.

  Lemma inverted_neg_rule_context :
    forall(r : sequent_rule V L),
      inverted_neg_rule r ->
        exists(sl sr : sequent V L)(f : lambda_formula V L),
          r = rule_add_context sl sr (bare_inverted_neg_rule f).
  Proof.
    clear. 
    intros r H.
    unfold inverted_neg_rule, rule_add_context, add_context in *.
    decompose [ex and or dep_and] H; clear H.
    exists x, x0, x1.
    destruct r.
    simpl in *.
    rewrite H1.
    rewrite H2.
    trivial.
  Qed.

  Lemma context_inverted_neg_rule :
    forall(sl sr : sequent V L)(f : lambda_formula V L),
      inverted_neg_rule 
        (rule_add_context sl sr (bare_inverted_neg_rule f)).
  Proof.
    clear. 
    intros sl sr f.
    unfold inverted_and_right_rule, rule_add_context in *.
    exists sl, sr, f.
    auto.
  Qed.


  (***************************************************************************)
  (** ****  collected inversion rules  *)
  (***************************************************************************)

  Definition inversion_rules(r : sequent_rule V L) : Prop :=
    inverted_and_left_rule r \/ inverted_and_right_rule r \/
    inverted_or_rule r \/ inverted_neg_rule r.

  Lemma inversion_rules_multiset : rule_multiset inversion_rules.
  Proof.
    clear. 
    unfold rule_multiset, inversion_rules in *.
    intros or s H H0.
    decompose [or] H; clear H.
          assert (H2 := inverted_and_left_rule_context _ H1).
          clear H1.
          decompose [ex] H2; clear H2.
          subst or.
          lapply (reorder_rule_add_context _ s x x0 H0).
            intros H.
            decompose [ex] H; clear H.
            eexists.
            split.
              eexact H2.
            left.
            apply context_inverted_and_left_rule.
          trivial.
        assert (H3 := inverted_and_right_rule_context _ H2).
        clear H2.
        decompose [ex] H3; clear H3.
        subst or.
        lapply (reorder_rule_add_context _ s x x0 H0).
          intros H.
          decompose [ex] H; clear H.
          eexists.
          split.
            eexact H2.
          right.
          left.
          apply context_inverted_and_right_rule.
        trivial.
      assert (H2 := inverted_or_rule_multiset _ _ H1 H0).
      decompose [ex and or dep_and] H2; clear H2.
      eexists.
      split.
        eexact H3.
      auto.
    assert (H2 := inverted_neg_rule_context _ H1).
    clear H1.
    decompose [ex] H2; clear H2.
    subst or.
    lapply (reorder_rule_add_context _ s x x0 H0).
      intros H.
      decompose [ex] H; clear H.
      eexists.
      split.
        eexact H2.
      right.
      right.
      right.
      apply context_inverted_neg_rule.
    trivial.
  Qed.

  Lemma inversion_rules_nonempty_conclusion : forall(r : sequent_rule V L),
    inversion_rules r ->
      conclusion r <> [].
  Proof.
    intros r H.
    unfold inversion_rules in *.
    decompose [or] H; clear H.
          unfold inverted_and_left_rule in *.
          decompose [ex and or dep_and] H0; clear H0.
          rewrite H2.
          destruct x.
            discriminate.
          discriminate.
        unfold inverted_and_right_rule in *.
        decompose [ex and or dep_and] H1; clear H1.
        rewrite H2.
        destruct x.
          discriminate.
        discriminate.
      unfold inverted_or_rule in *.
      decompose [ex and or dep_and] H0; clear H0.
        rewrite H.
        destruct x.
          discriminate.
        discriminate.
      rewrite H.
      destruct x.
        discriminate.
      discriminate.
    unfold inverted_neg_rule in *.
    decompose [ex and or dep_and] H0; clear H0.
    rewrite H2.
    destruct x.
      discriminate.
    discriminate.
  Qed.


  (***************************************************************************)
  (** ***  contraction rule  *)
  (***************************************************************************)

  Definition is_contraction_rule(r : sequent_rule V L) : Prop :=
    exists(n : nat),
      n_less # n < length (conclusion r) /#\
      assumptions r = [(nth (conclusion r) n n_less) :: (conclusion r)].

  Lemma contraction_rule_multiset : rule_multiset is_contraction_rule.
  Proof.
    unfold rule_multiset, is_contraction_rule, reordered_rule in *.
    intros or s H H0.
    decompose [ex and or dep_and] H; clear H.
    exists {| assumptions := [nth (conclusion or) x a :: s];
              conclusion := s |}.
    repeat split.
        rewrite b.
        trivial.
      rewrite b.
      simpl length.
      intros n n_less_rs n_less_rr.
      assert (n = 0).
        omega.
      subst n.
      simpl.
      apply list_reorder_cons_head.
      trivial.
    simpl.
    assert (H1 := list_reorder_occurence _ _ x a H0).
    decompose [ex and or dep_and] H1; clear H1.
    exists x0.
    constructor 1 with (a := a0).
    rewrite b0.
    trivial.
  Qed.

End Rules.


Implicit Arguments simple_tautology_witness [V L].
Implicit Arguments simple_tautology [[V] [L]].
Implicit Arguments simple_tautology_cons [V L].
Implicit Arguments add_context [V L].
Implicit Arguments rule_add_context [V L].
Implicit Arguments rank_rule_add_context_rev [V L].
Implicit Arguments reorder_rule_add_context [V L].
Implicit Arguments rank_sequent_subst_Ax_set [V L].
Implicit Arguments is_ax_rule [[V] [L]].
Implicit Arguments is_and_rule [[V] [L]].
Implicit Arguments bare_and_rule [V L].
Implicit Arguments and_rule_context [V L].
Implicit Arguments is_neg_and_rule [[V] [L]].
Implicit Arguments bare_neg_and_rule [V L].
Implicit Arguments neg_and_rule_context [V L].
Implicit Arguments is_neg_neg_rule [[V] [L]].
Implicit Arguments bare_neg_neg_rule [V L].
Implicit Arguments neg_neg_rule_context [V L].
Implicit Arguments tautology_witness [V L].
Implicit Arguments subst_Ax_set [V L].
Implicit Arguments subst_Ax_n_set [V L].
Implicit Arguments is_cut_rule [[V] [L]].
Implicit Arguments bounded_weakening_closed [V L].
Implicit Arguments full_weakening_closed [V L].
Implicit Arguments inverted_and_left_rule [V L].
Implicit Arguments inverted_and_right_rule [V L].
Implicit Arguments inverted_or_rule [V L].
Implicit Arguments inverted_neg_rule [V L].
Implicit Arguments is_contraction_rule [[V] [L]].
