(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: syntactic_cut.v,v 1.20 2013/04/10 11:17:17 tews Exp $
 *)


(** ** Syntactic proof of contraction and cut elimination, 5.6.2-3, 5.7

      This module puts the proof for contraction in GR_n (5.6.2) and
      cut elimination in GR_n (5.6.3) together. The different cases of
      this proof are spread over several files. This module contains
      the double induction for cut elimination, the final Lemma 5.6
      and Theorem 5.7. 
 *)

Require Export osr_cut mixed_cut prop_cut.


Section Syntactic_cut.

  Variable V : Type.
  Variable L : modal_operators.

  Variable op_eq : eq_type (operator L).
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Towards proposition 5.6 (2-3), page 28 *)
  (***************************************************************************)

  (***************************************************************************)
  (** ****  Nested induction for cut elimination  *)
  (***************************************************************************)

  (** This lemma does the nested induction in G_n + H and distinguishes 
      the major cases.
   *)
  Lemma syntactic_GR_n_cut_head_elimination_ind :
   forall(rules : set (sequent_rule V L))(n : nat)(ssn_pos : 0 < 2 + n),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      absorbs_contraction op_eq v_eq rules -> 
      absorbs_cut op_eq v_eq rules ->
      (forall(s : sequent V L)(f : lambda_formula V L),
         provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) 
                  (f :: f :: s) ->
           provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) (f :: s))
      ->
      (forall(s : sequent V L),
           provable (GRC_n_set rules (S n)) (empty_sequent_set V L) s ->
             provable (GR_n_set rules (S n)) (empty_sequent_set V L) s)
      ->
         forall(m sd : nat)
               (f : lambda_formula V L)(r q : sequent V L)
               (p_fq : proof (G_n_set V L (2 + n))
                          (provable_subst_n_conclusions rules (2 + n) ssn_pos)
                          (f :: q))
               (p_nfr : proof (G_n_set V L (2 + n))
                          (provable_subst_n_conclusions rules (2 + n) ssn_pos)
                          ((lf_neg f) :: r)),
           proof_depth p_fq + proof_depth p_nfr <= sd ->
           formula_measure f < m ->
             provable (G_n_set V L (2 + n))
                      (provable_subst_n_conclusions rules (2 + n) ssn_pos)
               (q ++ r).
  Proof.
    induction m.
      intros sd f r q p_fq p_nfr H6 H7.
      omega.
    induction sd.
      intros f r q p_fq p_nfr H6 H7.
      clear - H6.
      destruct (proof_depth p_fq).
        simpl in H6.
        eapply proof_depth_0.
        eexact H6.
      omega.
    intros f r q p_fq p_nfr H6 H7.
    remember (f :: q) as ls eqn:H8.
    remember ((lf_neg f) :: r) as rs eqn:H9.
    destruct p_fq as [gamma H10 | fq_rule H10 fq_sub].
      subst gamma.
      destruct p_nfr as [gamma H11 | nfr_rule H11 nfr_sub].
        subst gamma.
        eapply syntactic_GR_n_cut_eli_two_osr; eauto.
      apply mixed_cut_left_osr 
            with (H := H11)(m := m)(f := f)(negf_sub := nfr_sub)(sd := sd); 
            auto.
      clear - IHm.
      intros f r q H H0 H1.
      destruct H as [pfq].
      destruct H0 as [pnfr].
      clear H H0.
      eapply (IHm (S _) _ _ _ pfq pnfr); eauto.
    destruct p_nfr as [gamma H11 | nfr_rule H11 nfr_sub].
      subst gamma.
      eapply mixed_cut_right_osr; eauto.
    apply syntactic_GR_n_cut_two_prop with (3 := H2)(8 := H7)(9 := H6); auto.
    clear - IHm.
    intros f r q H H0 H1.
    destruct H as [pfq].
    destruct H0 as [pnfr].
    clear H H0.
    eapply (IHm (S _) _ _ _ pfq pnfr); eauto.
  Qed.


  (***************************************************************************)
  (** **** Modal rank induction step for cut elimination *)
  (***************************************************************************)

  (** This lemma reduces general cut-elimination to cut-elimination at 
      head position in the calculus G_n + H
      The lemma also adds rank assumptions to the induction hypothesis.
   *)
  Lemma syntactic_GR_n_cut_elimination :
    forall(rules : set (sequent_rule V L))(n : nat),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      absorbs_contraction op_eq v_eq rules -> 
      absorbs_cut op_eq v_eq rules ->
      (forall(s : sequent V L)(f : lambda_formula V L),
         provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) 
                  (f :: f :: s) ->
           provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) (f :: s)) 
      ->
      (forall(s : sequent V L),
         provable (GRC_n_set rules (S n)) (empty_sequent_set V L) s ->
           provable (GR_n_set rules (S n)) (empty_sequent_set V L) s)
      ->
        forall(s : sequent V L),
           provable (GRC_n_set rules (2 + n)) (empty_sequent_set V L) s ->
             provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) s.
  Proof.
    intros rules n H H0 H1 H2 H3 H4 H5 s H6.
    apply countably_infinite_non_empty with (X := V); trivial.
    intros nonempty_v.
    apply provable_GRC_n_GR_n_from_head_elim; trivial.
    clear s H6.
    intros f q r H6 H7.
    assert (H8 := lt_0_Sn (S n)).
    rewrite GR_n_provable_with_premises with (npos := H8); trivial.
    rewrite GR_n_provable_with_premises with (npos := H8) in H6; trivial.
    rewrite GR_n_provable_with_premises with (npos := H8) in H7; trivial.
    destruct H6 as [p_fq].
    clear H6.
    destruct H7 as [p_nfr].
    clear H6.
    eapply syntactic_GR_n_cut_head_elimination_ind 
               with (p_fq := p_fq)(p_nfr := p_nfr); eauto.
  Qed.


  (***************************************************************************)
  (** **** Proposition 5.6 (2-3) mutual induction lemma 

          Finally, this Lemma formalizes case 2 and 3 of Proposition
          5.6. It proves contraction and cut elimination by mutual
          induction over the modal rank. 
   *)
  (***************************************************************************)

  Lemma syntactic_GR_n_cc : 
    forall(rules : set (sequent_rule V L))(n : nat),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      absorbs_contraction op_eq v_eq rules -> 
      absorbs_cut op_eq v_eq rules -> 
        (forall(s : sequent V L)(f : lambda_formula V L),
           provable (GR_n_set rules n) (empty_sequent_set V L) 
                    (f :: f :: s) ->
             provable (GR_n_set rules n) (empty_sequent_set V L) 
                      (f :: s))
        /\
        (forall(s : sequent V L),
           provable (GRC_n_set rules n) (empty_sequent_set V L) s ->
             provable (GR_n_set rules n) (empty_sequent_set V L) s).
  Proof.
    induction n.
      intros H H0 H1 H2 H3.
      split.
        intros s f H4.
        eapply rank_formula_zero_TCC; eauto.
        eapply rank_sequent_head.
        apply provable_rank_rules_has_rank_n in H4.
        eexact H4.
      intros s H4.
      exfalso.
      clear - H0 H4.
      destruct H4.
      clear H.
      destruct x.
        contradiction.
      apply GRC_n_set_empty in g.
        contradiction.
      trivial.
    destruct n.
      clear IHn.
      intros H H0 H1 H2 H3.
      split.
        intros s f H4.
        eapply proof_mono_rules.
          apply G_n_subset_GR_n.
        apply prop_contraction_head; trivial.
        eapply proof_set_equal_rules.
          apply GR_1_is_G_prop.
          eexact H0.
        trivial.
      intros s H4.
      apply provable_GRC_1_GR_1; trivial.
    intros H H0 H1 H2 H3.
    specialize (IHn H H0 H1 H2 H3).
    destruct IHn.
    assert (forall(s : sequent V L)(f : lambda_formula V L),
              provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) 
                       (f :: f :: s) ->
                provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) 
                         (f :: s)).
      intros s f H6.
      eapply syntactic_GR_n_contraction_ind; eauto.
    clear H4.
    split; trivial.
    apply syntactic_GR_n_cut_elimination; trivial.
  Qed.


  (***************************************************************************)
  (** ***  Proposition 5.7, page 32  *)
  (***************************************************************************)

  (** 5.7, cut part *)
  Theorem syntactic_admissible_cut : 
    forall(rules : set (sequent_rule V L)),
      countably_infinite V ->
      one_step_rule_set rules ->
      absorbs_congruence rules ->
      absorbs_contraction op_eq v_eq rules ->
      absorbs_cut op_eq v_eq rules -> 
        admissible_rule_set (GR_set rules) (empty_sequent_set V L) is_cut_rule.
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros osr H H0 H1 H2 H3 r H4 H5.
    assert (provable (GRC_set osr) (empty_sequent_set V L) (conclusion r)).
      apply provable_with_rule with (assum := assumptions r).
        right.
        auto.
      intros i i_less.
      eapply proof_mono_rules.
        apply GR_subset_GRC.
      apply H5.
    clear H5.
    rewrite rank_proof_GRC in H6.
    destruct H6 as [d].
    apply syntactic_GR_n_cc in H5; trivial.
    rewrite rank_proof_GR.
    eexists.
    eexact H5.
  Qed.

  (** 5.7, contraction part *)
  Theorem syntactic_admissible_contraction : 
    forall(osr : set (sequent_rule V L)),
      countably_infinite V ->
      one_step_rule_set osr ->
      absorbs_congruence osr ->
      absorbs_contraction op_eq v_eq osr ->
      absorbs_cut op_eq v_eq osr -> 
        admissible_rule_set (GR_set osr) (empty_sequent_set V L)
          is_contraction_rule.
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros osr H H0 H1 H2 H3 r H4 H5.
    unfold is_contraction_rule in *.
    decompose [ex and or dep_and] H4; clear H4.
    rename x into i, a into i_less.
    rewrite b in *.
    clear b.
    apply every_nth_head in H5.
    rewrite rank_proof_GR in H5.
    destruct H5 as [n].
    rewrite rank_proof_GR.
    exists n.
    apply syntactic_support_contraction_ind 
           with (s1 := [nth (conclusion r) i i_less]).
        apply syntactic_GR_n_cc; trivial.
      trivial.
    clear.
    intros f H.
    destruct H.
      subst f.
      rewrite (list_split_at_n _ i i_less).
      apply in_or_app.
      right.
      left.
      trivial.
    contradiction.
  Qed.


End Syntactic_cut.

