(*
 * Memcheck -- ocaml runtime type checking
 *
 * Copyright (C) 2006, Hendrik Tews, all right reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this directory
 * for more details.
 *
 * $Id: generate_type_descr.ml,v 1.6 2016/10/14 19:24:08 tews Exp $
 *
 * Description: 
 * 
 * This is a camlp4 module, intented to be used as 
 * ``camlp4o generate_type_descr.cmo input.ml''
 * For each type definition found in the source it will generate a 
 * runtime type description suitable for Memcheck.check on stdout.
 *)



exception Unsupported of MLast.loc * string

let unvala = Pcaml.unvala

let output_header input_file = 
  Printf.sprintf "
(************************************************************************
 *
 * DO NOT EDIT!
 * Automatically generated by ocaml_ty_check
 * from %s
 *
 ***********************************************************************)

"
    input_file

let output_footer = "

(************************************************************************
 *
 * End of generated code.
 *
 ***********************************************************************)

"
    (* external name for the type constructor *)
let type_constr_descr_name name = name ^ "_type_constr_descr"

    (* external name for the type expression *)
let type_descr_name name = name ^ "_type_descr"

    (* generate a hopefully unique identifier for a type constructor *)
let type_constr_id (loc : Ploc.t) name =
  Printf.sprintf "%s file %s %d-%d line %d (char %d-%d)"
    name
    (Ploc.file_name loc)
    (Ploc.first_pos loc)
    (Ploc.last_pos loc)
    (Ploc.line_nb loc)
    (Ploc.first_pos loc - Ploc.bol_pos loc)
    (Ploc.last_pos loc - Ploc.bol_pos_last loc)


    (* output a reference to a type constructor applied to arguments *)
let rec do_type_constructor buf rec_names param_ids tconstr arguments =
  (* 
   * Printf.eprintf "{assoc %s in [%s]}\n"
   *   tconstr (String.concat "; " (List.map fst rec_names));
   *)
  if List.mem_assoc tconstr rec_names then 
    begin
      Printf.bprintf buf "Type_constructor_use(ref(Recursive_application(%d" 
	(List.assoc tconstr rec_names);
      Printf.bprintf buf ", [";
      List.iter
	(fun arg ->
	   do_type_expr buf rec_names param_ids arg;
	   Printf.bprintf buf "; ";
	)
	arguments;
      Printf.bprintf buf "])))"
    end
  else
    begin
      Printf.bprintf buf "(%s" (type_descr_name tconstr);
      List.iter
	(fun arg ->
	   Printf.bprintf buf " (";
	   do_type_expr buf rec_names param_ids arg;
	   Printf.bprintf buf ")";
	)
	arguments;
      Printf.bprintf buf ")";
    end      
    

    (* collect arguments of a type constructor application *)
and aggregate_type_application buf rec_names param_ids arguments typ =
  let loc = MLast.loc_of_ctyp typ
  in match typ with
    | <:ctyp< $uid:_$ . $t$ >> -> 
      aggregate_type_application buf rec_names param_ids arguments t
    | <:ctyp< $typ$ $arg$ >> ->
      aggregate_type_application buf rec_names param_ids 
	(arg :: arguments) typ
    | <:ctyp< $lid:tconstr$ >> ->
      do_type_constructor buf rec_names param_ids tconstr arguments;

    | _ -> raise (Unsupported(loc, "type constructor/application"))


    (* output a runtime type description for typ *)
and do_type_expr buf rec_names param_ids typ = 
  let loc = MLast.loc_of_ctyp typ
  in match typ with
    | <:ctyp< $uid:_$ . $_$ >> 
    | <:ctyp< $_$ $_$ >>
    | <:ctyp< $lid:_$ >> -> 
      aggregate_type_application buf rec_names param_ids [] typ

    | <:ctyp< ' $id$ >> ->
      (try
	 Printf.bprintf buf "Type_parameter(%d)" 
	   (List.assoc id param_ids)
       with
	 | Not_found -> assert false)

    | <:ctyp< ( $list:types$ ) >> ->
      Printf.bprintf buf "(tuple_type_descr [";
      List.iter
	(fun typ ->
	   do_type_expr buf rec_names param_ids typ;
	   Printf.bprintf buf "; ";
	)
	types;
      Printf.bprintf buf "])"

    | <:ctyp< [ $list:constr_list$ ] >> -> 
      do_variant_type buf rec_names param_ids constr_list

    | <:ctyp< { $list:ltl$ } >> ->
      Printf.bprintf buf "Record_type(invalid_type_tag, [";
      List.iter
	(fun (_loc, name, _mut, ctyp) ->
	   Printf.bprintf buf "(\"%s\", " name;
	   do_type_expr buf rec_names param_ids ctyp;
	   Printf.bprintf buf ");";
	)
	ltl;
      Printf.bprintf buf "])";
      
    | _ -> raise (Unsupported(loc, "type expression"))


    (* treat a variant type *)
and do_variant_type buf rec_names param_ids constr_list =
  let (const_constr, var_constr) = 
    List.partition
      (function (_loc, _name, arguments, _) -> unvala arguments = [])
      constr_list
  in
    Printf.bprintf buf "Static_variant(invalid_type_tag, [";
    List.iter
      (fun (_loc, name, _, _) ->
	 Printf.bprintf buf "\"%s\"; " (unvala name))
      const_constr;
    Printf.bprintf buf "], [";
    List.iter
      (fun (_loc, name, args, _) ->
	 Printf.bprintf buf "(\"%s\", [ " (unvala name);
	 List.iter
	   (fun arg ->
	      do_type_expr buf rec_names param_ids arg;
	      Printf.bprintf buf "; ";
	   )
	   (unvala args);
	 Printf.bprintf buf "]); "
      )
      var_constr;
    Printf.bprintf buf "])"

		  
    (* output a runtime type description for a type definition in the source *)
let do_type_def buf rec_names {MLast.tdNam = tdNam; tdPrm; tdDef; tdCon; _} =
  let tpl = unvala tdPrm
  in let loc = fst (unvala tdNam)
  in let name = unvala (snd (unvala tdNam))
  in let _ = 
    if unvala tdCon <> [] then 
      raise (Unsupported(loc, "type with constraints"))
  in let param_counter = ref (-1) in
  let param_ids = 
    List.map 
      (fun (name, _var) ->
       match unvala name with
	 | None -> raise (Unsupported(loc, "anonymous type parameter"))
	 | Some name -> (name, (incr param_counter; !param_counter)))
      tpl
  in 
  let param_args = ref []
  in
    (* first build the param_args list *)
    for n = List.length tpl downto 1 do
      param_args := (Printf.sprintf "p%d" n) :: !param_args
    done;

    (* start code generation now *)
    Printf.bprintf buf 
      "let %s = Type_constructor_def(\"%s\", %d, " 
      (type_constr_descr_name name)
      (type_constr_id loc name)
      (List.length tpl);
    do_type_expr buf rec_names param_ids tdDef;
    Printf.bprintf buf ")\n\n";
    Printf.bprintf buf "let %s %s =\n"
      (type_descr_name name)
      (String.concat " " !param_args);
    Printf.bprintf buf 
      "  Type_constructor_use(ref(Resolved_application(%s, [%s])))\n\n"
      (type_constr_descr_name name)
      (String.concat "; " !param_args)
    

    (* do all type definitions in a type ... and block *)
let do_type_def_list buf tdl = 
  let type_names = 
    List.map 
      (fun {MLast.tdNam = tdNam; _ } -> unvala (snd (unvala tdNam)))
      tdl 
  in
  let type_counter = ref (-1) in
  let type_names_id = 
    List.map
      (fun name -> (name, (incr type_counter; !type_counter)))
      type_names
  in 
  let type_descr_names = List.map type_constr_descr_name type_names 
  in
    List.iter (do_type_def buf type_names_id) tdl;

    Printf.bprintf buf "let _ = fix_rec_type_def\n    [%s]\n\n"
      (String.concat "; " type_descr_names)
    

    (* treat str_item *)
let scan_str_item buf ast = 
  match ast with
                                          (* intercept simple abbreviations *)
    | <:str_item< type $lid:name$ = $lid:oname$ >> -> 
      Printf.bprintf buf "let %s = %s\n\n" 
	(type_descr_name name) 
	(type_descr_name oname)

    | <:str_item< type $list:tdl$ >> -> 
      (try
	 do_type_def_list buf tdl
       with
	 | Unsupported(loc, msg) -> 
	     Printf.eprintf "File \"%s\", line %d, characters %d-%d:\n"
	       (Ploc.file_name loc)
	       (Ploc.line_nb loc)
	       (Ploc.first_pos loc - Ploc.bol_pos loc)
	       (Ploc.last_pos loc - Ploc.bol_pos_last loc);
	     Printf.eprintf "%s unsupported\n" msg
      )

    | _ -> ()


    (* do the file *)
let scan_implemantation (impl, _) = 
  let buf = Buffer.create 8192 in
  let _ = Printf.bprintf buf "%s" (output_header !Pcaml.input_file) in
  let rec doit = function
    | [] -> ()
    | (a, _) :: l -> scan_str_item buf a; doit l
  in
    doit impl;

    Printf.bprintf buf "%s" output_footer;
    print_endline (Buffer.contents buf)

    (* don't do mli's *)
let interface _ = 
  prerr_endline "interface files (mli) are not supported";
  exit 1

let _ = Pcaml.print_implem := scan_implemantation
let _ = Pcaml.print_interf := interface

(* switch off lexing of quotations *)
let _ = Plexer.no_quotations := true
