(* 
 * 
 *               Camlp4 quotations in original syntax
 * 
 *                 Copyright (C) 2005  Hendrik Tews
 * 
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file LICENCE in this or one
 *   of the parent directories for more details.
 * 
 *   Time-stamp: <Tuesday 14 June 05 0:08:56 tews@debian>
 * 
 *   $Id: pp_util.ml,v 1.16 2005/10/06 15:34:08 tews Exp $
 * 
 *)

module L = Lexing
module F = Format

let ocamlp4_version = "0.2"

let terminate_structures = ref false

let abort_when_incomplete = ref false

let test_more_quotations = ref false

(* Delete this when available from MLast. 
 * Delete also the -I +camlp4 in the Makefile
 *)
external loc_of_with_constr : MLast.with_constr -> MLast.loc = "%field0"


(****************************************************************
 *
 * Indent Buffers
 *
 *)

module type Indent_buffer_sig =
  sig
    type t
    val create : int -> t
    val reset : t -> unit
    val indent : t -> unit
    val unindent : t -> unit
    val check_indent : t -> int -> unit
    val add_string : t -> string -> unit
    val endline : t -> string -> unit
    val printf : t -> ('a, unit, string, unit) format4 -> 'a
    val flush : t -> out_channel -> unit
  end

module Indent_buffer : Indent_buffer_sig =
  struct
    type t = {
      mutable indent : int;
      mutable newline_pending : bool;
      buf : Buffer.t
    }

    let create n = {
      indent = 0;
      newline_pending = false;
      buf = Buffer.create n
    }

    let reset b = 
      begin 
	b.indent <- 0;
	b.newline_pending <- false;
	Buffer.clear b.buf
      end

    let check_indent b i =
      assert(b.indent = i)

    let newline_and_indent b =
      Buffer.add_char b.buf '\n';
      Buffer.add_string b.buf (String.make b.indent ' ')

    let add_string_intern b s slen =
      let i = ref 0
      in
	if b.newline_pending then begin
	  newline_and_indent b;
	  b.newline_pending <- false
	end;
	while !i < slen do
	  let j = 
	    try
	      String.index_from s !i '\n'
	    with
	      | Not_found -> slen
	  in
	    Buffer.add_substring b.buf s !i (j - !i);
	    if j < slen then begin
	      newline_and_indent b;
	      i := j + 1;
	    end else
	      i := j;
	done

    let add_string b s =
      let slen = String.length s 
      in
	if slen > 0 && s.[slen - 1] = '\n'
	then begin
	  add_string_intern b s (slen - 1);
	  b.newline_pending <- true;
	end else 
	  add_string_intern b s slen
	

    let indent b = b.indent <- b.indent + 2

    let unindent b = 
      b.indent <- b.indent - 2;
      assert(b.indent >= 0)

    let endline b s = add_string b s; add_string b "\n" 

    let printf b fmt = Printf.kprintf (add_string b) fmt
	
    let flush b oc = 
      if b.newline_pending then
	newline_and_indent b;
      Buffer.output_buffer oc b.buf
  end

module IB = Indent_buffer

(****************************************************************
 *
 * Install pretty printer in camlp4
 *
 *)


module type PP_TYPE =
sig
  val id : string

  val pp_implem : Indent_buffer.t -> (MLast.str_item * 'a) list -> unit
    
  val pp_interf : Indent_buffer.t -> (MLast.sig_item * 'a) list -> unit
end


module type INSTALL_TYPE = 
sig
  val main : unit -> unit
end

module Install_pp(PP : PP_TYPE) : INSTALL_TYPE =
struct
  let buf = IB.create 4096

  let version_string =
    Printf.sprintf "%s version %s" PP.id ocamlp4_version

  let output_version () =
    print_endline version_string;
    exit 0

  let main () =
    Pcaml.print_implem := 
      (fun x ->
	 IB.reset buf; 
	 PP.pp_implem buf x; 
	 IB.check_indent buf 0;
	 IB.endline buf "";
	 IB.flush buf stdout
      );
    Pcaml.print_interf := 
      (fun x ->
	 IB.reset buf; 
	 PP.pp_interf buf x;
	 IB.check_indent buf 0;
	 IB.endline buf "";
	 IB.flush buf stdout
      );
    Pcaml.add_option "-pp-version" (Arg.Unit output_version)
      ("output version (" ^ version_string ^ ") and exit");
    Pcaml.add_option "-fail" (Arg.Set abort_when_incomplete)
      ("abort on unknown ast nodes");
    Pcaml.add_option "-with-end-tags" (Arg.Set terminate_structures)
      ("add lines to terminate the description of nodes");
    Pcaml.add_option "-test" (Arg.Set test_more_quotations)
      ("enable additional testing code");
end



(****************************************************************
 *
 * Utility functions
 *
 *)

exception PP_Failure

let abort_maybe () =
  if !abort_when_incomplete then
    raise PP_Failure


let string_of_loc (start, ende) =
  (Printf.sprintf "file \"%s\", line %d (char %d"
     start.L.pos_fname start.L.pos_lnum
     (start.L.pos_cnum - start.L.pos_bol)) 
  ^
    if start.L.pos_lnum = ende.L.pos_lnum
    then
      if start.L.pos_cnum = ende.L.pos_cnum
      then
	")"
      else
	Printf.sprintf "-%d)"
	  (ende.L.pos_cnum - ende.L.pos_bol)
    else
      Printf.sprintf ") - line %d (char %d)"
	ende.L.pos_lnum
	(ende.L.pos_cnum - ende.L.pos_bol)


let print_location pp (mesg : string) loc =
  IB.printf pp "%s (%s)\n" mesg (string_of_loc loc)

  (* 
   * use as
   * 
   * iterate_items pp (Some astloc) name "%d items" "item %d:" printfun itemlist
   * 
   * iterate_items pp None "" "%d" "item %d:" printfun itemlist
   *)
let iterate_items pp locopt name sumformat itemformat printfun items =
  (match locopt with
     | None -> ()
     | Some loc ->
	 print_location pp 
	   (Printf.sprintf ("%s " ^^ sumformat) 
	      name (List.length items))
	   loc
  );
  ignore(
    List.fold_left
      (fun n item ->
	 IB.indent pp;
	 IB.printf pp (itemformat ^^ "\n") n;
	 printfun item;
	 IB.unindent pp;
	 n + 1;
      )
      1
      items
  );
  if !terminate_structures then
    match locopt with
      | None -> ()
      | Some _ -> IB.add_string pp ("end " ^ name)


(*****************************************************************
 *
 * conv_format as suggested by X Leroy on the caml mailing list
 * see http://caml.inria.fr/pub/ml-archives/caml-list/2003/07/bb08176385e0e3626d7033d67ea11fa8.en.html
 *
*****************************************************************)

(* 
 * let quote_format (s : string) =
 *   let b = Buffer.create (String.length s + 10) in
 *   let rec quote i =
 *     if i = String.length s 
 *     then Buffer.contents b
 *     else 
 *       begin
 * 	if s.[i] = '%' then Buffer.add_char b '%';
 * 	Buffer.add_char b s.[i];
 * 	quote (i+1)
 *       end
 *   in
 *     quote 0
 *     
 * let plain_format_of_string (s : string) =
 *   let rec check i =
 *     if i = String.length s 
 *     then true
 *     else 
 *       if s.[i] = '%' 
 *       then
 * 	if i +1 < String.length s 
 * 	then
 * 	  if s.[i+1] = '%'
 * 	  then check (i + 2)
 * 	  else false
 * 	else false
 *       else check (i+1)
 *   in
 *     if check 0 
 *     then (Obj.magic s : ('a, 'b, 'c, 'a) format4)
 *     else 
 *       begin
 * 	failwith "unit_format_of_string"
 *       end
 *)
    
